package oracle.ugbu.ss.portal.backing;


import com.oracle.ugbu.ss.api.model.account.OUCSSAccount;
import com.oracle.ugbu.ss.api.model.account.OUCSSAccountContextInfo;
import com.oracle.ugbu.ss.api.model.common.OUCSSConstants;
import com.oracle.ugbu.ss.api.model.common.OUCSSList;
import com.oracle.ugbu.ss.api.model.context.OUCSSContext;
import com.oracle.ugbu.ss.api.model.factory.OUCSSFactory;
import com.oracle.ugbu.ss.api.model.security.OUCSSUser;
import com.oracle.ugbu.ss.common.SSUtil;
import com.oracle.ugbu.ss.shared.model.cache.OUCSSCacheManager;
import com.oracle.ugbu.ss.shared.model.cache.OUCSSSystemProperties;
import com.oracle.ugbu.ss.shared.model.common.OUCSSModelUtil;
import com.oracle.ugbu.ss.shared.model.helper.OUCSSSessionHelperImpl;
import com.oracle.ugbu.ss.shared.view.type.OUCSSDisplayType;

import java.io.Serializable;

import java.util.Collections;
import java.util.Map;

import oracle.adf.share.ADFContext;
import oracle.adf.share.logging.ADFLogger;


/**
 * Class to manage the OUCSS related context in User Session.
 */
public class OUCSSSessionBean implements Serializable {
    @SuppressWarnings("compatibility")
    private static final long serialVersionUID = 1L;
    private static final ADFLogger _LOG =
        ADFLogger.createADFLogger(OUCSSSessionBean.class);
    private static final String DEFAULT_LANDING_PAGE_CD = "DEFAULT";
    private static Map<String, String> LOB_LANDING_PAGE_MAP =
        Collections.emptyMap();
    private static Map<String, String> MOB_LOB_LANDING_PAGE_MAP =
        Collections.emptyMap();
    private static final String MOB_PORTAL_LANDING_PAGE_CD =
        "MOB_B_PORTAL_LANDING_PAGE";
    private static final String PORTAL_LANDING_PAGE_CD = "PORTAL_LANDING_PAGE";

    private Boolean mValidUser = null;
    private Boolean mEnrolledUser = null;
    private Boolean mCSRUser = null;
    private OUCSSContext<OUCSSUser, OUCSSAccount> mDefaultContext;

    private transient OUCSSFactory<OUCSSUser, OUCSSAccount> mOUCSSFactory =
        null;
    private String mContextLOB;

    public OUCSSSessionBean() {
        //Setup the context lob.
        mContextLOB =
                (String)OUCSSSessionHelperImpl.getInstance().getSessionData(OUCSSConstants.OUCSS_USER_LOB_INFO_KEY);
    }

    /**
     * Method to check if user is valid OCUSS User. The user is declared valid if
     * member isCSRUser is true.
     * @return
     */
    public boolean isValidOUCSSUser() {
        if (mValidUser == null) {
            mValidUser = Boolean.FALSE;
            if (ADFContext.getCurrent().getSecurityContext().isAuthenticated()) {
                // Check if user is OUCSS valid user.
                mValidUser = getContext().getCurrentUser().isValidOUCSSUser();

                //Validate user if user is member of either WSSCSR or WSSAdmin Group.
                if (!mValidUser) {
                    if (isCsrUser()) {
                        mValidUser = Boolean.TRUE;
                    } else {
                        SSUtil.addMessage("USER_DETAILS_UPD_FAILED");
                    }
                }
            }
        }
        return mValidUser.booleanValue();
    }

    /**
     * Method to check if user has enrolled or invited accounts.
     * @return
     */
    public boolean isEnrolledUser() {
        return getContext().getCurrentUser().isEnrolledOUCSSUser() ||
            isCsrUser();
    }

    /**
     * Method to check if user has enough accounts enrolled to enable search.
     * @return
     */
    public boolean isAccountSearchEnabled() {
        boolean search = getContext().getCurrentUser().isEnrolledOUCSSUser();
        if (search) {
            search =
                    getContext().getCurrentUserAccountList().getSize() > OUCSSSystemProperties.getAccountListMaxRows();
        }
        return search;
    }

    /**
     * Method to check if the account is prepaid or regular account.
     * @return true if the account if of prepaid type, false otherwise.
     */
    public String getAccountType() {
        String accountType = "UNKNOWN";
        if (getContext().getCurrentUser().isEnrolledOUCSSUser()) {
            //Get the Account Context for current account.
            try {
                OUCSSAccountContextInfo accountContext =
                    getContext().getCurrentUserAccount().getAccountContextInfo();
                if (accountContext != null) {
                    //Determine if the account is prepaid or not.
                    boolean isPrepaid = accountContext.isPrePaid();
                    if (isPrepaid) {
                        accountType = "PREPAID";
                    } else {
                        accountType = "POSTPAID";
                    }
                }
            } catch (Exception e) {
                //Log exception if unable to determine the account type.
                _LOG.warning(e);
            }
        }
        return accountType;
    }

    /**
     * Method to check if logged in user is CSR or Admin
     * @return true if user is member of WSSCSRGroup or WSSAdminGroup, false otherwise.
     */
    public boolean isCsrUser() {
        if (mCSRUser == null) {
            mCSRUser = OUCSSModelUtil.isCSRUser();
        }
        return mCSRUser.booleanValue();
    }

    /**
     * Method to get the Context. Context sets up the user and user accounts on demand.
     * @return
     */
    public OUCSSContext<OUCSSUser, OUCSSAccount> getContext() {
        if (mDefaultContext == null) {
            //Create context with null user as Context will evaluate the context user id.
            mDefaultContext = getFactory().createContext();
        }
        return mDefaultContext;
    }

    /**
     * Method to get the Accounts Selected in Business Context.
     * @return
     */
    public OUCSSList<OUCSSAccount> getMultipleAccounts() {
        OUCSSList<OUCSSAccount> mSelectedAccounts =
            getContext().getBusinessContext().getSelectedAccounts();

        return mSelectedAccounts;
    }

    /**
     * Method to get the Selected Account as List.
     * @return
     */
    public OUCSSList<OUCSSAccount> getSelectedAccount() {
        OUCSSList<OUCSSAccount> mSelectedAccounts =
            getFactory().createAccountList();

        mSelectedAccounts.add(getContext().getCurrentUserAccount());

        return mSelectedAccounts;
    }

    /**
     * Method to get the Landing page.
     * @return
     */
    public String getLandingPage() {
        String landingPage = null;
        String lobEaId = getFactory().getLob();

        if (lobEaId == null &&
            getContext().getCurrentUser().isEnrolledOUCSSUser()) {
            //If lobEaId is null, then use the current account to decide landing page.
            lobEaId = getContext().getCurrentUserAccount().getLob();
        }

        if (lobEaId != null) {
            String lob =
                OUCSSCacheManager.getLOBCache().getItemAsString(lobEaId,
                                                                SSUtil.getUserLocaleLanguage());
            if (isSingleEnrolledAccount()) {
                lob = lob + "_SA";
            }

            landingPage = getLOBLandingPageMap().get(lob);
        }

        //Check if return page is null.
        if (landingPage == null) {
            //Find the default landing in case of no-context login or LOB landing page not found.
            landingPage = getLOBLandingPageMap().get(DEFAULT_LANDING_PAGE_CD);
        }
        return landingPage;
    }

    /**
     * Method to check if the user is enrolled to only one account (for logged in lob).
     * @return true if user has one account enrolled, false otherwise.
     */
    public boolean isSingleEnrolledAccount() {
        boolean singleAccount = false;
        if (getContext().getCurrentUser().isEnrolledOUCSSUser()) {
            singleAccount =
                    getContext().getCurrentUserAccountList().getSize() == 1;
        }
        return singleAccount;
    }

    /**
     * Method to get the landing page lookups.
     * @return
     */
    private static Map<String, String> getLOBLandingPageMap() {
        Map<String, String> landingPageMap = null;
        if (OUCSSDisplayType.isMobilePlatform()) {
            //Get the mobile landing page.
            if (MOB_LOB_LANDING_PAGE_MAP.isEmpty()) {
                MOB_LOB_LANDING_PAGE_MAP =
                        SSUtil.getLOBSiteResource(MOB_PORTAL_LANDING_PAGE_CD);
            }
            landingPageMap = MOB_LOB_LANDING_PAGE_MAP;
        } else {
            //Get the landing page.
            if (LOB_LANDING_PAGE_MAP.isEmpty()) {
                LOB_LANDING_PAGE_MAP =
                        SSUtil.getLOBSiteResource(PORTAL_LANDING_PAGE_CD);
            }
            landingPageMap = LOB_LANDING_PAGE_MAP;
        }
        return landingPageMap;
    }

    /**
     * Method to get the Context LOB.
     * @return
     */
    public String getContextLOB() {
        return mContextLOB;
    }

    /**
     * Method to get OUCSS Factory.
     * @return
     */
    protected OUCSSFactory<OUCSSUser, OUCSSAccount> getFactory() {
        if (mOUCSSFactory == null) {
            mOUCSSFactory = OUCSSModelUtil.getFactory(getContextLOB());
        }
        return mOUCSSFactory;
    }

  public String getRefreshToken()
  {
    //If token not set, then use session value.
    String token = (String) OUCSSSessionHelperImpl.getInstance().getSessionData(OUCSSConstants.OUCSS_REFRESH_TOKEN);
    return token;
  }

}
