/*
* Replicator.java
*
* Copyright 2001-2007 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its
* affiliates.
*
* This software is the confidential and proprietary information of Oracle
* Corporation. You shall not disclose such confidential and proprietary
* information and shall use it only in accordance with the terms of the
* license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
package com.tangosol.examples.extend;


import com.tangosol.net.BackingMapManager;
import com.tangosol.net.CacheFactory;
import com.tangosol.net.CacheService;
import com.tangosol.net.DefaultConfigurableCacheFactory;
import com.tangosol.net.Invocable;
import com.tangosol.net.InvocationService;
import com.tangosol.net.NamedCache;
import com.tangosol.net.Service;

import com.tangosol.util.Base;

import java.util.Map;


/**
* Command line utility for replicating the contents of one or more caches
* to another Coherence cluster via Coherence*Extend.
*
* @author jh  2006.08.03
*/
public class Replicator
        extends Base
        implements Invocable
    {
    // ----- constructors ---------------------------------------------------

    /**
    * Create a new Replicator that replicates the cache with the given name.
    *
    * @param sCacheName  the name of the cache to replicate
    */
    public Replicator(String sCacheName)
        {
        if (sCacheName == null || sCacheName.length() == 0)
            {
            throw new IllegalArgumentException("illegal cache name: " + sCacheName);
            }

        m_sCacheName = sCacheName;
        }


    // ----- application entry point ----------------------------------------

    /**
    * Start the Replicator with the given command line arguments.
    * <p/>
    * Usage:
    * <pre>
    * java Replicator <invocation service name> <cache name> ...
    * </pre>
    * Example:
    * <pre>
    * java Replicator Invocation boston-cache
    * </pre>
    *
    * @param asArgs  command line arguments
    */
    public static void main(String[] asArgs)
        {
        int cArgs = asArgs == null ? 0 : asArgs.length;
        if (cArgs < 2)
            {
            showInstructions();
            return;
            }

        // get the invocation service
        Service service = CacheFactory.getService(asArgs[0]);
        if (!(service instanceof InvocationService))
            {
            err("Invalid invocation service name: " + asArgs[0]);
            return;
            }

        InvocationService invocation = (InvocationService) service;

        log("Starting Replicator ...");
        for (int i = 1; i < cArgs; ++i)
            {
            String sName = asArgs[i];

            // find and validate the cache
            NamedCache cache;
            try
                {
                cache = CacheFactory.getCache(sName);
                }
            catch (RuntimeException e)
                {
                err("Invalid cache name: " + sName);
                continue;
                }

            log("Replicating cache " + sName + " ...");

            // invoke a Replicator on all members that run the target service
            invocation.query(new Replicator(sName),
                    cache.getCacheService().getInfo().getServiceMembers());
            }
        log("Done.");
        }

    /**
    * Display the instructions for the command-line utility.
    */
    public static void showInstructions()
        {
        out();
        out("java Replicator <invocation service name> <cache name> ...");
        out();
        }


    // ----- Invocable interface --------------------------------------------

    /**
    * {@inheritDoc}
    */
    public void init(InvocationService service)
        {
        }

    /**
    * {@inheritDoc}
    */
    public void run()
        {
        String            sName   = getCacheName();
        NamedCache        cache   = CacheFactory.getCache(sName);
        CacheService      service = cache.getCacheService();
        BackingMapManager manager = service.getBackingMapManager();

        if (manager instanceof DefaultConfigurableCacheFactory.Manager)
            {
            Map map = ((DefaultConfigurableCacheFactory.Manager) manager).getBackingMap(sName);
            if (map instanceof ExtendReadWriteBackingMap)
                {
                log("Replicating cache " + sName + " ...");
                ((ExtendReadWriteBackingMap) map).replicate();
                log("Done.");
                }
            }
        }

    /**
    * {@inheritDoc}
    */
    public Object getResult()
        {
        return null;
        }


    // ----- accessors ------------------------------------------------------

    /**
    * @return the name of the cache to replicate
    */
    public String getCacheName()
        {
        return m_sCacheName;
        }


    // ----- data members ---------------------------------------------------

    /**
    * The name of the cache to replicate.
    */
    protected final String m_sCacheName;
    }