/*
* utils.cpp
*
* Copyright 2001-2008 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its
* affiliates.
*
* This software is the confidential and proprietary information of Oracle
* Corporation. You shall not disclose such confidential and proprietary
* information and shall use it only in accordance with the terms of the
* license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
#include <iostream>

#include "utils.h"

using namespace std;

/**
 * Initialize JVM; 
 * All command line arguments are converted to JavaVMOption array
 */
void Utils::ensureJVM(int argc, char* const argv[]) {
    jint           res;
    JavaVMInitArgs vm_args;
    
    if (argc <= 1) {
        cerr << "one or more jvm arguments expected" << std::endl;
        exit(-1);
    }
    
    JavaVMOption   options[argc - 1];
    
    cout << "JVM options:" << std::endl;
    for (int i = 1; i < argc; i++) {
        options[i-1].optionString = argv[i];
        std::cout << "\t" << options[i-1].optionString << std::endl;
    }
    
    vm_args.version            = 0x00010002;
    vm_args.options            = options;
    vm_args.nOptions           = argc - 1;
    vm_args.ignoreUnrecognized = JNI_TRUE;
    
    //
    // Create the VM
    //
    res = JNI_CreateJavaVM(&s_jvm, (void**)&s_env, &vm_args);
    
    if (res < 0) {
        cerr << "Cannot create JVM, error=" << res;
        exit(res);
    }        
}

/**
 * Obtain jclass ref; Terminate if it cannot be found
 */
jclass Utils::ensureClass(const char* pszClassName) {
    jclass clazz = getEnv()->FindClass(pszClassName);
    if (clazz == NULL) {
        cerr << "Class " << pszClassName << " not found" << endl;
        exit(-1);
    }
    return clazz;
}

/**
 * Obtain jmethodID; Terminate if it cannot be found
 * (most likely because signature is wrong)
 */
jmethodID Utils::ensureMethod(const jclass clazz, const char* pszMethodName, const char* pszSignature) {
    jmethodID mid = getEnv()->GetMethodID(clazz, pszMethodName, pszSignature);
    if (mid == NULL) {
        cerr << "Unable to obtain methodID: " << pszMethodName << endl;
        exit(-1);
    }
    return mid;
}

/**
 * Obtain static jmethodID; Terminate if it cannot be found
 * (most likely because signature is wrong)
 */
jmethodID Utils::ensureStaticMethod(const jclass clazz, const char* pszMethodName, const char* pszSignature) {
    jmethodID mid = getEnv()->GetStaticMethodID(clazz, pszMethodName, pszSignature);
    if (mid == NULL) {
        cerr << "Unable to obtain static methodID: " << pszMethodName << endl;
        exit(-1);
    }
    return mid;
}

/**
 * Singleton JavaVM ref
 */
JavaVM* Utils::s_jvm = NULL;

/**
 * Singleton JNIEnv ref
 */
JNIEnv* Utils::s_env = NULL;
