/*
* cache.h
*
* Copyright 2001-2008 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its
* affiliates.
*
* This software is the confidential and proprietary information of Oracle
* Corporation. You shall not disclose such confidential and proprietary
* information and shall use it only in accordance with the terms of the
* license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
#ifndef _CACHE_H_
#define _CACHE_H_

#include <jni.h>

#include "utils.h"

/**
* An abstract Cache
* Concrete class is expected to implement these converters:
* 
* jobject convertValToJava(void *pObject)
* void*   convertValFromJava(jobject jObject)
* jobject convertKeyToJava(void *pKey)
* 
*/
class Cache {
public:
    
    Cache(const char* pszCacheName);

    /**
     * If the specified item is in the cache, return it.
     *
     * @param oKey  the key to the desired cached item
     *
     * @return the desired Object if it is in the cache, otherwise null
     */    
    void* get(void* pKey);
   
    /**
     * Store a value in the cache.
     *
     * @param oKey    the key with which to associate the cache value
     * @param oValue  the value to cache
     *
     * @return the value that was cached associated with that key, or null if
     *         no value was cached associated with that key
     */    
    void* put(void* pKey, void* pValue);
    
    /**
     * Store a value in the cache that will be automatically evicted according
     * to the specified expiration value.
     *
     * @param oKey     the key with which to associate the cache value
     * @param oValue   the value to cache
     * @param cMillis  the number of milliseconds until the entry will expire;
     *                 pass zero to use the cache's default ExpiryDelay settings;
     *                 pass -1 to indicate that the entry should never expire
     *
     * @return the value that was cached associated with that key, or null if
     *         no value was cached associated with that key
     *
     */    
    void* put(void* pKey, void *pValue, long cMillis);

    /**
     * Store a value in the cache, without returning previous value.
     * Similar to NamedCache.putAll(Collections.singletonMap(oKey, oValue)
     *
     * @param oKey    the key with which to associate the cache value
     * @param oValue  the value to cache
     *
     */    
    void putBlind(void* pKey, void* pValue);
    
    /**
     * Remove an entry from the cache.
     *
     * @param oKey  the key of a cached value
     *
     * @return the value that was cached associated with that key, or null if
     *         no value was cached associated with that key
     */    
    void* remove(void* pKey);

    /**
     * Attempt to lock the specified item and return immediately.
     *
     * Expensive: Locking always occurs on the back cache.
     *
     * @param oKey key being locked
     *
     * @return <tt>true</tt> if the item was successfully locked;
     *         <tt>false</tt> otherwise
     */
    bool lock(void* pKey);
    
    /**
     * Attempt to lock the specified item within the specified period of time.
     *
     * Expensive: Locking always occurs on the back cache.
     *
     * @param oKey     key being locked
     * @param lMillis  the number of milliseconds to continue trying to obtain
     *                 a lock; pass zero to return immediately; pass -1 to block
     *                 the calling thread until the lock could be obtained
     *
     * @return true if the item was successfully locked within the
     *              specified time; false otherwise
     */
    bool lock(void* pKey, long lMillis);
    

    /**
     * Unlock the specified item.
     *
     * @param oKey key being unlocked
     *
     * @return true if the item was successfully unlocked; false otherwise
     */
    bool unlock(void* pKey);
    
    
    /**
     * Returns the number of key-value mappings in this map.
     *
     * @return the number of key-value mappings in this map
     */                                              
    int size();

protected:
    
    /**
     * Converters to/from Java key and value counterpart classes
     */
    virtual jobject convertValToJava(void *pObject)     = 0;
    virtual void*   convertValFromJava(jobject jObject) = 0;
    virtual jobject convertKeyToJava(void *pKey)        = 0;
    
private:
    
    // NamedCache
    jobject m_cache;
                                  
    // Classes
    jclass m_classCacheFactory;             // com.tangosol.net.CacheFactory
    jclass m_classNamedCache;               // com.tangosol.net.NamedCache
    jclass m_classCollections;              // java.util.Collections
    
    // Methods
    jmethodID m_midCacheFactoryGetCache;    // NamedCache CacheFactory.getCache(String sName)
    jmethodID m_midNamedCacheGet;           // Object     NamedCache.get(Object oKey)
    jmethodID m_midNamedCachePut;           // Object     NamedCache.put(Object oKey, Object oValue)
    jmethodID m_midNamedCachePutAll;        // void       NamedCache.putAll(Map map)
    jmethodID m_midNamedCachePutExpiry;     // Object     NamedCache.put(Object oKey, Object oValue, long cMillis)
    jmethodID m_midNamedCacheRemove;        // Object     NamedCache.remove(Object oKey)
    jmethodID m_midNamedCacheLock;          // boolean    NamedCache.lock(Object oKey, long cMillis)
    jmethodID m_midNamedCacheUnlock;        // boolean    NamedCache.unlock(Object oKey)
    jmethodID m_midNamedCacheSize;          // int        NamedCache.size()
    jmethodID m_midCollectionsSingletonMap; // Map        Collections.singletonMap(Object oKey, Object oValue)
};

#endif

