/*
* cache.cpp
*
* Copyright 2001-2008 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its
* affiliates.
*
* This software is the confidential and proprietary information of Oracle
* Corporation. You shall not disclose such confidential and proprietary
* information and shall use it only in accordance with the terms of the
* license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
#include <iostream>

#include "cache.h"

using namespace std;

/**
* Cache constructor - initialize required method id's and NamedCache reference
*/ 
Cache::Cache(const char* pszCacheName) {

    m_classCacheFactory = Utils::ensureClass("com/tangosol/net/CacheFactory");
    m_classNamedCache   = Utils::ensureClass("com/tangosol/net/NamedCache");
    m_classCollections  = Utils::ensureClass("java/util/Collections");
    
    m_midCacheFactoryGetCache 
    = Utils::ensureStaticMethod(m_classCacheFactory, "getCache", "(Ljava/lang/String;)Lcom/tangosol/net/NamedCache;");

    m_midNamedCacheGet       = Utils::ensureMethod(m_classNamedCache, "get"   , "(Ljava/lang/Object;)Ljava/lang/Object;");
    m_midNamedCachePut       = Utils::ensureMethod(m_classNamedCache, "put"   , "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
    m_midNamedCachePutAll    = Utils::ensureMethod(m_classNamedCache, "putAll", "(Ljava/util/Map;)V");
    m_midNamedCachePutExpiry = Utils::ensureMethod(m_classNamedCache, "put"   , "(Ljava/lang/Object;Ljava/lang/Object;J)Ljava/lang/Object;");
    m_midNamedCacheRemove    = Utils::ensureMethod(m_classNamedCache, "remove", "(Ljava/lang/Object;)Ljava/lang/Object;");
    m_midNamedCacheLock      = Utils::ensureMethod(m_classNamedCache, "lock"  , "(Ljava/lang/Object;J)Z");
    m_midNamedCacheUnlock    = Utils::ensureMethod(m_classNamedCache, "unlock", "(Ljava/lang/Object;)Z");
    m_midNamedCacheSize      = Utils::ensureMethod(m_classNamedCache, "size"  , "()I");
    
    m_midCollectionsSingletonMap 
    = Utils::ensureStaticMethod(m_classCollections, "singletonMap", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/util/Map;");
    
    //
    // This will start cluster service and join the cluster
    //
    m_cache = Utils::getEnv()->CallStaticObjectMethod(m_classCacheFactory, m_midCacheFactoryGetCache, Utils::getEnv()->NewStringUTF(pszCacheName));
    if (m_cache == NULL) {
        cerr << "Unable to obtain NamedCache instance" << endl;
        exit(-1);
    }
}

/**
 * If the specified item is in the cache, return it.
 *
 * @param oKey  the key to the desired cached item
 *
 * @return the desired Object if it is in the cache, otherwise null
 */    
void* Cache::get(void* pKey) {
    void*   pValue = NULL;
    jobject jKey   = convertKeyToJava(pKey);
    jobject jValue = Utils::getEnv()->CallObjectMethod(m_cache, m_midNamedCacheGet, jKey);
    
    Utils::getEnv()->DeleteLocalRef(jKey);
    
    if (jValue != NULL) {
        pValue = convertValFromJava(jValue);
        Utils::getEnv()->DeleteLocalRef(jValue);
    }
    
    return pValue;
}

/**
 * Store a value in the cache.
 *
 * @param oKey    the key with which to associate the cache value
 * @param oValue  the value to cache
 *
 * @return the value that was cached associated with that key, or null if
 *         no value was cached associated with that key
 */    
void* Cache::put(void* pKey, void* pValue) {
    return put(pKey, pValue, 0L);
}

/**
 * Store a value in the cache that will be automatically evicted according
 * to the specified expiration value.
 *
 * @param oKey     the key with which to associate the cache value
 * @param oValue   the value to cache
 * @param cMillis  the number of milliseconds until the entry will expire;
 *                 pass zero to use the cache's default ExpiryDelay settings;
 *                 pass -1 to indicate that the entry should never expire
 *
 * @return the value that was cached associated with that key, or null if
 *         no value was cached associated with that key
 *
 */    
void* Cache::put(void* pKey, void *pValue, long cMillis) {
    void* pOldValue = NULL;
    
    jobject jKey      = convertKeyToJava(pKey);
    jobject jValue    = convertValToJava(pValue);
    // Need to use two separate methods in case backing map
    // doesn't implement CachingMap
    jobject jOldValue = cMillis <= 0 ?
    Utils::getEnv()->CallObjectMethod(m_cache, m_midNamedCachePut, jKey, jValue) :
    Utils::getEnv()->CallObjectMethod(m_cache, m_midNamedCachePutExpiry, jKey, jValue, cMillis);
    
    Utils::getEnv()->DeleteLocalRef(jKey);
    Utils::getEnv()->DeleteLocalRef(jValue);
    
    if (jOldValue != NULL) {
        pOldValue = convertValFromJava(jOldValue);
        Utils::getEnv()->DeleteLocalRef(jOldValue);
    }
    
    return pOldValue;
}

/**
 * Store a value in the cache, without returning previous value.
 * Similar to NamedCache.putAll(Collections.singletonMap(oKey, oValue)
 *
 * @param oKey    the key with which to associate the cache value
 * @param oValue  the value to cache
 *
 */    
void Cache::putBlind(void* pKey, void* pValue) {
    jobject jKey   = convertKeyToJava(pKey);
    jobject jValue = convertValToJava(pValue);
    jobject jMap   = Utils::getEnv()->CallStaticObjectMethod(m_classCollections, m_midCollectionsSingletonMap, jKey, jValue);
    
    Utils::getEnv()->CallVoidMethod(m_cache, m_midNamedCachePutAll, jMap);
    
    Utils::getEnv()->DeleteLocalRef(jKey);
    Utils::getEnv()->DeleteLocalRef(jValue);
    Utils::getEnv()->DeleteLocalRef(jMap);
}        

/**
 * Remove an entry from the cache.
 *
 * @param oKey  the key of a cached value
 *
 * @return the value that was cached associated with that key, or null if
 *         no value was cached associated with that key
 */    
void* Cache::remove(void* pKey) {
    void* pOldValue = NULL;
    
    jobject jKey      = convertKeyToJava(pKey);
    jobject jOldValue = Utils::getEnv()->CallObjectMethod(m_cache, m_midNamedCacheRemove, jKey);
    
    Utils::getEnv()->DeleteLocalRef(jKey);
    
    if (jOldValue != NULL) {
        pOldValue = convertValFromJava(jOldValue);
        Utils::getEnv()->DeleteLocalRef(jOldValue);
    }
    
    return pOldValue;
}

/**
 * Attempt to lock the specified item and return immediately.
 *
 * Expensive: Locking always occurs on the back cache.
 *
 * @param oKey key being locked
 *
 * @return <tt>true</tt> if the item was successfully locked;
 *         <tt>false</tt> otherwise
 */
bool Cache::lock(void* pKey)
{
    return lock(pKey, 0L);
}

/**
 * Attempt to lock the specified item within the specified period of time.
 *
 * Expensive: Locking always occurs on the back cache.
 *
 * @param oKey     key being locked
 * @param lMillis  the number of milliseconds to continue trying to obtain
 *                 a lock; pass zero to return immediately; pass -1 to block
 *                 the calling thread until the lock could be obtained
 *
 * @return true if the item was successfully locked within the
 *              specified time; false otherwise
 */
bool Cache::lock(void* pKey, long lMillis) {
    jobject  jKey     = convertKeyToJava(pKey);
    jboolean jbResult = Utils::getEnv()->CallBooleanMethod(m_cache, m_midNamedCacheLock, jKey, lMillis);
    
    Utils::getEnv()->DeleteLocalRef(jKey);
    
    return jbResult == JNI_TRUE;
}


/**
 * Unlock the specified item.
 *
 * @param oKey key being unlocked
 *
 * @return true if the item was successfully unlocked; false otherwise
 */
bool Cache::unlock(void* pKey) {
    jobject  jKey     = convertKeyToJava(pKey);
    jboolean jbResult = Utils::getEnv()->CallBooleanMethod(m_cache, m_midNamedCacheUnlock, jKey);
    
    Utils::getEnv()->DeleteLocalRef(jKey);
    
    return jbResult == JNI_TRUE;
    
}


/**
 * Returns the number of key-value mappings in this map.
 *
 * @return the number of key-value mappings in this map
 */                                              
int Cache::size() {
    return Utils::getEnv()->CallIntMethod(m_cache, m_midNamedCacheSize);
}


