/*
* WrapperMap.java
*
* Copyright 2001-2007 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its affiliates.
*
* This software is the confidential and proprietary information of
* Oracle Corporation. You shall not disclose such confidential and
* proprietary information and shall use it only in accordance with the
* terms of the license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
package com.tangosol.examples.extend;


import com.tangosol.util.Base;
import com.tangosol.util.ClassHelper;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;


/**
* Map implementation that delegates to another Map.
*
* @author jh  2007.mm.dd
*/
public class WrapperMap
        extends Base
        implements Map
    {
    // ----- constructors ---------------------------------------------------

    /**
    * Create a new WrapperMap that delegates to the given Map.
    *
    * @param map  the wrapped Map; must not be null
    */
    public WrapperMap(Map map)
        {
        if (map == null)
            {
            throw new IllegalArgumentException("null Map");
            }

        m_map = map;
        }


    // ----- Map interface --------------------------------------------------

    /**
    * {@inheritDoc}
    */
    public void clear()
        {
        getMap().clear();
        }

    /**
    * {@inheritDoc}
    */
    public boolean containsKey(Object oKey)
        {
        return getMap().containsKey(oKey);
        }

    /**
    * {@inheritDoc}
    */
    public boolean containsValue(Object oValue)
        {
        return getMap().containsValue(oValue);
        }

    /**
    * {@inheritDoc}
    */
    public Set entrySet()
        {
        Set set = m_setEntry;
        return set == null ? m_setEntry = instantiateEntrySet() : set;
        }

    /**
    * {@inheritDoc}
    */
    public Object get(Object oKey)
        {
        return getMap().get(oKey);
        }

    /**
    * {@inheritDoc}
    */
    public boolean isEmpty()
        {
        return getMap().isEmpty();
        }

    /**
    * {@inheritDoc}
    */
    public Set keySet()
        {
        Set set = m_setKey;
        return set == null ? m_setKey = instantiateKeySet() : set;
        }

    /**
    * {@inheritDoc}
    */
    public Object put(Object oKey, Object oValue)
        {
        return getMap().put(oKey, oValue);
        }

    /**
    * {@inheritDoc}
    */
    public void putAll(Map map)
        {
        getMap().putAll(map);
        }

    /**
    * {@inheritDoc}
    */
    public Object remove(Object oKey)
        {
        return getMap().remove(oKey);
        }

    /**
    * {@inheritDoc}
    */
    public int size()
        {
        return getMap().size();
        }

    /**
    * {@inheritDoc}
    */
    public Collection values()
        {
        Collection coll = m_collValue;
        return coll == null ? m_collValue = instantiateValueCollection() : coll;
        }


    // ----- Object methods -------------------------------------------------

    /**
    * {@inheritDoc}
    */
    public boolean equals(Object o)
        {
        if (this == o)
            {
            return true;
            }

        if (o instanceof WrapperMap)
            {
            return getMap().equals(((WrapperMap) o).getMap());
            }

        return false;
        }

    /**
    * {@inheritDoc}
    */
    public int hashCode()
        {
        return getMap().hashCode();
        }

    /**
    * {@inheritDoc}
    */
    public String toString()
        {
        return ClassHelper.getSimpleName(getClass()) + ":\n" + getMap();
        }


    // ----- Map.Entry implementation ---------------------------------------

    /**
    * Instantiate a Map.Entry for this WrapperMap.
    *
    * @param entry  the Map.Entry to wrap; must not be null
    *
    * @return a wrapped Map.Entry for the given Map.Entry
    */
    protected Map.Entry instantiateEntry(Map.Entry entry)
        {
        return new WrapperEntry(entry);
        }

    /**
    * Return the original Map.Entry that may have been wrapped by the given
    * Map.Entry.
    *
    * @param entry  the Map.Entry to unwrap
    *
    * @return the unwrapped Map.Entry if the given Map.Entry is a wrapper;
    *         otherwise, the passed Map.Entry
    */
    protected Map.Entry unwrapEntry(Map.Entry entry)
        {
        return entry instanceof WrapperEntry
                ? ((WrapperEntry) entry).getEntry()
                : entry;
        }

    /**
    * Map.Entry implementation that delegates to another Map.Entry.
    */
    protected class WrapperEntry
            extends Base
            implements Map.Entry
        {
        // ----- constructors ---------------------------------------------

        /**
        * Create a new WrapperEntry that delegates to the given Map.Entry.
        *
        * @param entry  the wrapped Map.Entry; must not be null
        */
        public WrapperEntry(Map.Entry entry)
            {
            if (entry == null)
                {
                throw new IllegalArgumentException("null Map.Entry");
                }

            m_entry = entry;
            }

        // ----- Map.Entry interface --------------------------------------

        /**
        * {@inheritDoc}
        */
        public Object getKey()
            {
            return getEntry().getKey();
            }

        /**
        * {@inheritDoc}
        */
        public Object getValue()
            {
            return getEntry().getValue();
            }

        /**
        * {@inheritDoc}
        */
        public Object setValue(Object o)
            {
            return put(getKey(), o);
            }

        // ----- Object methods -------------------------------------------

        /**
        * {@inheritDoc}
        */
        public boolean equals(Object o)
            {
            if (this == o)
                {
                return true;
                }

            if (o instanceof WrapperEntry)
                {
                return getEntry().equals(((WrapperEntry) o).getEntry());
                }

            return false;
            }

        /**
        * {@inheritDoc}
        */
        public int hashCode()
            {
            return getEntry().hashCode();
            }

        /**
        * {@inheritDoc}
        */
        public String toString()
            {
            return ClassHelper.getSimpleName(getClass()) + ": " + getEntry();
            }

        // ----- accessors ------------------------------------------------

        /**
        * @return the wrapped Map.Entry
        */
        protected Map.Entry getEntry()
            {
            return m_entry;
            }

        // ----- data members ---------------------------------------------

        /**
        * The wrapped Map.Entry.
        */
        protected final Map.Entry m_entry;
        }


    // ----- entry Set implementation ---------------------------------------

    /**
    * @return an entry Set for this WrapperMap.
    */
    protected Set instantiateEntrySet()
        {
        return new WrapperEntrySet();
        }

    /**
    * Entry Set implementation for this WrapperMap.
    */
    protected class WrapperEntrySet
            extends WrapperSet
        {
        // ----- constructors ---------------------------------------------

        /**
        * Create a new WrapperEntrySet.
        */
        public WrapperEntrySet()
            {
            super(getMap().entrySet());
            }

        // ----- Set interface --------------------------------------------

        /**
        * {@inheritDoc}
        */
        public boolean add(Object o)
            {
            throw new UnsupportedOperationException();
            }

        /**
        * {@inheritDoc}
        */
        public void clear()
            {
            WrapperMap.this.clear();
            }

        /**
        * {@inheritDoc}
        */
        public boolean contains(Object o)
            {
            return super.contains(unwrapEntry((Map.Entry) o));
            }

        /**
        * {@inheritDoc}
        */
        public boolean containsAll(Collection coll)
            {
            for (Iterator iter = coll.iterator(); iter.hasNext(); )
                {
                if (!contains(iter.next()))
                    {
                    return false;
                    }
                }

            return false;
            }

        /**
        * {@inheritDoc}
        */
        public boolean remove(Object o)
            {
            return super.remove(unwrapEntry((Map.Entry) o));
            }

        /**
        * {@inheritDoc}
        */
        public boolean removeAll(Collection coll)
            {
            boolean fModified = false;
            if (size() > coll.size())
                {
                for (Iterator iter = coll.iterator(); iter.hasNext();)
                    {
                    fModified |= remove(iter.next());
                    }
                }
            else
                {
                for (Iterator iter = iterator(); iter.hasNext(); )
                    {
                    if (coll.contains(unwrapEntry((Map.Entry) iter.next())))
                        {
                        iter.remove();
                        fModified = true;
                        }
                    }
                }

            return fModified;
            }

        /**
        * {@inheritDoc}
        */
        public boolean retainAll(Collection coll)
            {
            boolean fModified = false;
            for (Iterator iter = iterator(); iter.hasNext();)
                {
                if (!coll.contains(unwrapEntry((Map.Entry) iter.next())))
                    {
                    iter.remove();
                    fModified = true;
                    }
                }

            return fModified;
            }

        /**
        * {@inheritDoc}
        */
        public Object[] toArray()
            {
            Object[] ao = super.toArray();
            for (int i = 0, c = ao == null ? 0 : ao.length; i < c; ++i)
                {
                ao[i] = instantiateEntry((Entry) ao[i]);
                }

            return ao;
            }

        /**
        * {@inheritDoc}
        */
        public Object[] toArray(Object[] ao)
            {
            ao = super.toArray(ao);
            for (int i = 0, c = ao == null ? 0 : ao.length; i < c; ++i)
                {
                ao[i] = instantiateEntry((Entry) ao[i]);
                }

            return ao;
            }


        // ----- WrapperCollection methods --------------------------------

        /**
        * {@inheritDoc}
        */
        protected Iterator instantiateIterator()
            {
            return new WrapperIterator()
                {
                public Object next()
                    {
                    Map.Entry entry = (Map.Entry) getIterator().next();
                    entry = instantiateEntry(entry);
                    setLast(entry);

                    return entry;
                    }
                };
            }
        }


    // ----- key Set implementation -----------------------------------------

    /**
    * @return a key Set for this WrapperMap.
    */
    protected Set instantiateKeySet()
        {
        return new WrapperKeySet();
        }

    /**
    * Key Set implementation for this WrapperMap.
    */
    protected class WrapperKeySet
            extends WrapperSet
        {
        // ----- constructors ---------------------------------------------

        /**
        * Create a new WrapperKeySet.
        */
        public WrapperKeySet()
            {
            super(getMap().keySet());
            }

        // ----- Set interface --------------------------------------------

        /**
        * {@inheritDoc}
        */
        public boolean add(Object o)
            {
            throw new UnsupportedOperationException();
            }

        /**
        * {@inheritDoc}
        */
        public void clear()
            {
            WrapperMap.this.clear();
            }
        }


    // ----- value Collection implementation --------------------------------

    /**
    * @return a value Collection for this WrapperMap.
    */
    protected Collection instantiateValueCollection()
        {
        return new WrapperValueCollection();
        }

    /**
    * Value Collection implementation for this WrapperMap.
    */
    protected class WrapperValueCollection
            extends WrapperCollection
        {
        // ----- constructors ---------------------------------------------

        /**
        * Create a new WrapperValueCollection.
        */
        public WrapperValueCollection()
            {
            super(getMap().values());
            }

        // ----- Collection interface -------------------------------------

        /**
        * {@inheritDoc}
        */
        public boolean add(Object o)
            {
            throw new UnsupportedOperationException();
            }
        
        /**
        * {@inheritDoc}
        */
        public void clear()
            {
            WrapperMap.this.clear();
            }
        }


    // ----- accessors ------------------------------------------------------

    /**
    * @return the wrapped Map
    */
    protected Map getMap()
        {
        return m_map;
        }


    // ----- data members ---------------------------------------------------

    /**
    * The wrapped Map.
    */
    protected final Map m_map;

    /**
    * The wrapped entry Set.
    */
    protected Set m_setEntry;

    /**
    * The wrapped key Set.
    */
    protected Set m_setKey;

    /**
    * The wrapped value Collection.
    */
    protected Collection m_collValue;
    }