/*
* WrapperCollection.java
*
* Copyright 2001-2007 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its affiliates.
*
* This software is the confidential and proprietary information of
* Oracle Corporation. You shall not disclose such confidential and
* proprietary information and shall use it only in accordance with the
* terms of the license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
package com.tangosol.examples.extend;


import com.tangosol.util.Base;
import com.tangosol.util.ClassHelper;

import java.util.Collection;
import java.util.Iterator;


/**
* Collection implementation that delegates to another Collection.
*
* @author jh  2007.09.20
*/
public class WrapperCollection
        extends Base
        implements Collection
    {
    // ----- constructors ---------------------------------------------------

    /**
    * Create a new WrapperCollection that delegates to the given Collection.
    *
    * @param coll  the wrapped Collection; must not be null
    */
    public WrapperCollection(Collection coll)
        {
        if (coll == null)
            {
            throw new IllegalArgumentException("null Collection");
            }

        m_coll = coll;
        }


    // ----- Collection interface -------------------------------------------

    /**
    * {@inheritDoc}
    */
    public boolean add(Object o)
        {
        return getCollection().add(o);
        }

    /**
    * {@inheritDoc}
    */
    public boolean addAll(Collection coll)
        {
        boolean fModified = false;
        for (Iterator iter = coll.iterator(); iter.hasNext(); )
            {
            fModified |= add(iter.next());
            }

        return fModified;
        }

    /**
    * {@inheritDoc}
    */
    public void clear()
        {
        getCollection().clear();
        }

    /**
    * {@inheritDoc}
    */
    public boolean contains(Object o)
        {
        return getCollection().contains(o);
        }

    /**
    * {@inheritDoc}
    */
    public boolean containsAll(Collection coll)
        {
        return getCollection().containsAll(coll);
        }

    /**
    * {@inheritDoc}
    */
    public boolean isEmpty()
        {
        return getCollection().isEmpty();
        }

    /**
    * {@inheritDoc}
    */
    public Iterator iterator()
        {
        return instantiateIterator();
        }

    /**
    * {@inheritDoc}
    */
    public boolean remove(Object o)
        {
        return getCollection().remove(o);
        }

    /**
    * {@inheritDoc}
    */
    public boolean removeAll(Collection coll)
        {
        boolean fModified = false;
        if (size() > coll.size())
            {
            for (Iterator iter = coll.iterator(); iter.hasNext();)
                {
                fModified |= remove(iter.next());
                }
            }
        else
            {
            for (Iterator iter = iterator(); iter.hasNext(); )
                {
                if (coll.contains(iter.next()))
                    {
                    iter.remove();
                    fModified = true;
                    }
                }
            }

        return fModified;
        }

    /**
    * {@inheritDoc}
    */
    public boolean retainAll(Collection coll)
        {
        boolean fModified = false;
        for (Iterator iter = iterator(); iter.hasNext();)
            {
            if (!coll.contains(iter.next()))
                {
                iter.remove();
                fModified = true;
                }
            }

        return fModified;
        }

    /**
    * {@inheritDoc}
    */
    public int size()
        {
        return getCollection().size();
        }

    /**
    * {@inheritDoc}
    */
    public Object[] toArray()
        {
        return getCollection().toArray();
        }

    /**
    * {@inheritDoc}
    */
    public Object[] toArray(Object[] ao)
        {
        return getCollection().toArray(ao);
        }


    // ----- Object methods -------------------------------------------------

    /**
    * {@inheritDoc}
    */
    public boolean equals(Object o)
        {
        if (this == o)
            {
            return true;
            }

        if (o instanceof WrapperCollection)
            {
            return getCollection().equals(
                    ((WrapperCollection) o).getCollection());
            }

        return false;
        }

    /**
    * {@inheritDoc}
    */
    public int hashCode()
        {
        return getCollection().hashCode();
        }

    /**
    * {@inheritDoc}
    */
    public String toString()
        {
        return ClassHelper.getSimpleName(getClass()) + ":\n" + getCollection();
        }


    // ----- Iterator implementation ----------------------------------------

    /**
    * @return a new Iterator instance for this WrapperCollection
    */
    protected Iterator instantiateIterator()
        {
        return new WrapperIterator();
        }

    /**
    * Iterator implementation that delegates to another Iterator.
    */
    protected class WrapperIterator
            implements Iterator
        {
        // ----- constructor ----------------------------------------------

        /**
        * Create a new WrapperIterator.
        */
        public WrapperIterator()
            {
            m_iter = getCollection().iterator();
            }

        // ----- Iterator interface ---------------------------------------

        /**
        * {@inheritDoc}
        */
        public boolean hasNext()
            {
            return getIterator().hasNext();
            }

        /**
        * {@inheritDoc}
        */
        public Object next()
            {
            Object o = getIterator().next();
            setLast(o);

            return o;
            }

        /**
        * {@inheritDoc}
        */
        public void remove()
            {
            Object o = getLast();
            if (o == null)
                {
                throw new IllegalStateException("next() was not called");
                }

            WrapperCollection.this.remove(o);
            }

        // ----- Object methods -------------------------------------------

        /**
        * {@inheritDoc}
        */
        public String toString()
            {
            return ClassHelper.getSimpleName(getClass()) + '\n' + getIterator();
            }

        // ----- accessors ------------------------------------------------

        /**
        * @return the wrapped Iterator
        */
        protected Iterator getIterator()
            {
            return m_iter;
            }

        /**
        * @return the last object returned by this WrapperIterator.
        */
        protected Object getLast()
            {
            return m_oLast;
            }

        /**
        * Set the last object returned by this WrapperIterator.
        *
        * @param o  the last object returned by this WrapperIterator
        */
        protected void setLast(Object o)
            {
            m_oLast = o;
            }

        // ----- data members ---------------------------------------------

        /**
        * The wrapped Iterator.
        */
        protected final Iterator m_iter;

        /**
        * The last object that was returned by the WrapperIterator.
        */
        protected Object m_oLast;
        }


    // ----- accessors ------------------------------------------------------

    /**
    * @return the wrapped Collection
    */
    protected Collection getCollection()
        {
        return m_coll;
        }


    // ----- data members ---------------------------------------------------

    /**
    * The wrapped Collection.
    */
    protected final Collection m_coll;
    }