/*
* LiteEventCodec.java
*
* Copyright 2001-2007 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its affiliates.
*
* This software is the confidential and proprietary information of
* Oracle Corporation. You shall not disclose such confidential and
* proprietary information and shall use it only in accordance with the
* terms of the license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
package com.tangosol.examples.extend;


import com.tangosol.io.Evolvable;
import com.tangosol.io.ReadBuffer;
import com.tangosol.io.WriteBuffer;

import com.tangosol.io.pof.PofBufferReader;
import com.tangosol.io.pof.PofBufferWriter;
import com.tangosol.io.pof.PofContext;
import com.tangosol.io.pof.PofReader;
import com.tangosol.io.pof.PofWriter;
import com.tangosol.io.pof.PortableObject;

import com.tangosol.net.messaging.Channel;
import com.tangosol.net.messaging.Connection;
import com.tangosol.net.messaging.Message;

import com.tangosol.util.Base;
import com.tangosol.util.Binary;
import com.tangosol.util.ClassHelper;

import java.io.IOException;


/**
* Coherence*Extend Codec implementation that strips the old values from all
* heavy events sent to clients.
*
* @author jh  2007.09.20
*/
public class LiteEventCodec
        extends Base
        implements Connection.Codec
    {
    // ----- constructors ---------------------------------------------------

    /**
    * Default constructor.
    */
    public LiteEventCodec()
        {
        super();
        }

    // ----- Codec interface ------------------------------------------------

    /**
    * {@inheritDoc}
    */
    public Message decode(Channel channel, ReadBuffer.BufferInput in)
            throws IOException
        {
        azzert(channel instanceof PofContext);

        int        nTypeId    = in.readPackedInt();
        int        nVersionId = in.readPackedInt();
        Message    message    = channel.getMessageFactory().createMessage(nTypeId);
        PofContext ctx        = (PofContext) channel;
        PofReader  reader     = new PofBufferReader.UserTypeReader(in, ctx,
                nTypeId, nVersionId);

        azzert(message instanceof PortableObject);

        // set the version identifier
        boolean   fEvolvable = message instanceof Evolvable;
        Evolvable evolvable  = null;
        if (fEvolvable)
            {
            evolvable = (Evolvable) message;
            evolvable.setDataVersion(nVersionId);
            }

        // read the Message properties
        ((PortableObject) message).readExternal(reader);

        // read the future properties
        Binary binFuture = reader.readRemainder();
        if (fEvolvable)
            {
            evolvable.setFutureData(binFuture);
            }

        return message;
        }

    /**
    * {@inheritDoc}
    */
    public void encode(Channel channel, Message message,
            WriteBuffer.BufferOutput out)
            throws IOException
        {
        azzert(channel instanceof PofContext);
        azzert(message instanceof PortableObject);

        if (ClassHelper.getSimpleName(message.getClass())
                .equals(MAP_EVENT_MESSAGE_CLASS))
            {
            try
                {
                ClassHelper.invoke(message, OLD_VALUE_SETTER, NULL_ARG);
                }
            catch (Exception e)
                {
                log(e);
                }
            }

        PofContext ctx    = (PofContext) channel;
        PofWriter  writer = new PofBufferWriter.UserTypeWriter(out, ctx,
                message.getTypeId(), 0);

        // set the version identifier
        boolean   fEvolvable = message instanceof Evolvable;
        Evolvable evolvable  = null;
        if (fEvolvable)
            {
            evolvable = (Evolvable) message;
            writer.setVersionId(Math.max(evolvable.getDataVersion(),
                    evolvable.getImplVersion()));
            }

        // write the Message properties
        ((PortableObject) message).writeExternal(writer);

        // write the future properties
        writer.writeRemainder(fEvolvable ? evolvable.getFutureData() : null);
        }


    // ----- constants ------------------------------------------------------

    /**
    * The simple name of the Coherence*Extend message class that it used to
    * send events to clients.
    */
    public final static String MAP_EVENT_MESSAGE_CLASS = "MapEventMessage";

    /**
    * The name of the method used to set the old event value sent to clients.
    */
    public final static String OLD_VALUE_SETTER = "setValueOld";

    /**
    * A single null argument.
    */
    public final static Object[] NULL_ARG = new Object[] {null};
    }