/*
* ExtendBackingMap.java
*
* Copyright 2001-2007 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its affiliates.
*
* This software is the confidential and proprietary information of
* Oracle Corporation. You shall not disclose such confidential and
* proprietary information and shall use it only in accordance with the
* terms of the license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
package com.tangosol.examples.extend;


import com.tangosol.net.BackingMapManagerContext;
import com.tangosol.net.CacheFactory;
import com.tangosol.net.CacheService;
import com.tangosol.net.ClusterException;
import com.tangosol.net.MemberEvent;
import com.tangosol.net.MemberListener;
import com.tangosol.net.NamedCache;
import com.tangosol.net.Service;

import com.tangosol.net.cache.WrapperNamedCache;

import com.tangosol.util.Base;
import com.tangosol.util.ConcurrentMap;
import com.tangosol.util.Filter;
import com.tangosol.util.ImmutableArrayList;
import com.tangosol.util.MapEvent;
import com.tangosol.util.MapListener;
import com.tangosol.util.MapListenerSupport;
import com.tangosol.util.MultiplexingMapListener;
import com.tangosol.util.NullImplementation;
import com.tangosol.util.ObservableMap;
import com.tangosol.util.SafeHashMap;
import com.tangosol.util.TaskDaemon;
import com.tangosol.util.WrapperConcurrentMap;
import com.tangosol.util.WrapperObservableMap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
* Backing map implementation that allows one cluster to maintain a
* synchronized replica (or subset thereof) of a remote NamedCache.
* <p/>
* The ExtendBackingMap acts like a near cache of a remote cache; however,
* unlike the near cache, the ExtendBackingMap uses heavy listeners to
* synchronize rather than invalidate the local replica with the remote cache.
*
* @author jh  2007.12.07
*/
public class ExtendBackingMap
        extends WrapperMap
        implements ObservableMap
    {
    // ----- constructors ---------------------------------------------------

   /**
   * Create a new ExtendBackingMap that maintains a replica of the specified
   * remote NamedCache obtained via the specified remote CacheService.
   *
   * @param ctx           the context provided by the CacheService which is
   *                      using this backing map
   * @param sServiceName  the remote CacheService name
   * @param sCacheName    the remote NamedCache name
   */
    public ExtendBackingMap(BackingMapManagerContext ctx, String sServiceName,
            String sCacheName)
       {
       this(ctx, sServiceName, sCacheName, DEFAULT_RESTART);
       }

    /**
    * Create a new ExtendBackingMap that maintains a replica of the specified
    * remote NamedCache obtained via the specified remote CacheService.
    *
    * @param ctx           the BackingMapManagerContext of the NamedCache
    *                      that is backed by the returned ExtendBackingMap
    * @param sServiceName  the remote CacheService name
    * @param sCacheName    the remote NamedCache name
    * @param cRestartSec   the number of seconds between successive restart
    *                      attempts of the remote CacheService; if <= 0, the
    *                      restart task will not be started
    */
    public ExtendBackingMap(BackingMapManagerContext ctx, String sServiceName,
            String sCacheName, int cRestartSec)
        {
        super(new WrapperObservableMap(new SafeHashMap()));

        if (ctx == null)
            {
            throw new IllegalArgumentException("Missing BackingMapManagerContext");
            }
        m_ctx = ctx;

        if (sServiceName == null || sServiceName.length() == 0)
            {
            throw new IllegalArgumentException("Invalid CacheService name");
            }
        m_sServiceName = sServiceName;

        if (sCacheName == null || sCacheName.length() == 0)
            {
            throw new IllegalArgumentException("Invalid NamedCache name");
            }
        m_sCacheName = sCacheName;

        m_mapControl      = instantiateControlMap();
        m_mapDelta        = instantiateDeltaMap();
        m_listenerMap     = instantiateRemoteMapListener();
        m_listenerService = instantiateRemoteServiceListener();
        m_taskRestart     = instantiateRemoteCacheRestartTask();
        m_daemonReconcile = instantiateReconciliationDaemon();
        m_cacheRemote     = ensureRemoteCache();

        azzert(m_mapControl      != null);
        azzert(m_cacheRemote     != null);
        azzert(m_listenerMap     != null);
        azzert(m_listenerService != null);
        azzert(m_taskRestart     != null);
        azzert(m_daemonReconcile != null);
        azzert(m_cacheRemote     != null);

        if (cRestartSec > 0)
            {
            m_taskRestart.schedule(cRestartSec);
            }
        }


    // ----- ObservableMap interface ----------------------------------------

    /**
    * {@inheritDoc}
    */
    public void addMapListener(MapListener listener)
        {
        getInternalMap().addMapListener(listener);
        }

    /**
    * {@inheritDoc}
    */
    public void addMapListener(MapListener listener, Filter filter, boolean fLite)
        {
        getInternalMap().addMapListener(listener, filter, fLite);
        }

    /**
    * {@inheritDoc}
    */
    public void addMapListener(MapListener listener, Object oKey, boolean fLite)
        {
        getInternalMap().addMapListener(listener, oKey, fLite);
        }

    /**
    * {@inheritDoc}
    */
    public void removeMapListener(MapListener listener)
        {
        getInternalMap().removeMapListener(listener);
        }

    /**
    * {@inheritDoc}
    */
    public void removeMapListener(MapListener listener, Filter filter)
        {
        getInternalMap().removeMapListener(listener, filter);
        }

    /**
    * {@inheritDoc}
    */
    public void removeMapListener(MapListener listener, Object oKey)
        {
        getInternalMap().removeMapListener(listener, oKey);
        }


    // ----- lifecycle ------------------------------------------------------

    /**
    * Release all resources held by this ExtendBackingMap. After this method
    * is called, the ExtendBackingMap can no longer be used.
    */
    protected void release()
        {
        // stop the restart task
        getRemoteCacheRestartTask().cancel();

        // unregister the service listener
        getRemoteServiceListener().unregister();

        // stop the reconcile daemon
        getReconciliationDaemon().stop();

        // clear the cache and unregister all listeners
        clear();
        }


    // ----- Map interface --------------------------------------------------

    /**
    * {@inheritDoc}
    */
    public void clear()
        {
        ConcurrentMap mapControl = getControlMap();

        // Note: we cannot do a blocking LOCK_ALL as any event which came
        // in while the ThreadGate is in the closing state would cause the
        // service thread to spin, so try for up ~1s before giving up. We
        // don't even risk a timed LOCK_ALL as whatever time value we choose
        // would risk a useless spin for that duration
        for (int i = 0; !mapControl.lock(ConcurrentMap.LOCK_ALL, 0); ++i)
            {
            if (i == 100)
                {
                throw new IllegalStateException("Cache is in active use by other threads");
                }
            try
                {
                Thread.sleep(10);
                }
            catch (InterruptedException e)
                {
                Thread.currentThread().interrupt();
                throw Base.ensureRuntimeException(e);
                }
            }

        try
            {
            mapControl.put(GLOBAL_KEY, IGNORE_LIST);

            NamedCache cacheRemote = ensureRemoteCache();
            if (!cacheRemote.isActive())
                {
                // the remote cache is unavailable; clear the internal map
                getInternalMap().clear();
                return;
                }

            RemoteMapListener listener = getRemoteMapListener();
            for (Iterator iter = getInternalMap().keySet().iterator(); iter.hasNext(); )
                {
                try
                    {
                    listener.unregister(iter.next());
                    }
                catch (RuntimeException e)
                    {
                    validateRemoteException(e);
                    }
                iter.remove();
                }
            }
        finally
            {
            mapControl.remove(GLOBAL_KEY);
            mapControl.unlock(ConcurrentMap.LOCK_ALL);
            }
        }

    /**
    * {@inheritDoc}
    */
    public Object get(Object oKey)
        {
        ConcurrentMap mapControl = getControlMap();
        mapControl.lock(oKey, -1L);
        try
            {
            Map mapInternal = getInternalMap();
            if (mapInternal.containsKey(oKey))
                {
                return mapInternal.get(oKey);
                }

            NamedCache cacheRemote = ensureRemoteCache();
            if (!cacheRemote.isActive())
                {
                // the remote cache is unavailable; return null
                return null;
                }

            List listEvent = new ArrayList(0);
            mapControl.put(oKey, listEvent);
            try
                {
                getRemoteMapListener().register(oKey);
                }
            catch (RuntimeException e)
                {
                validateRemoteException(e);

                // the remote cache is no longer available; return null
                mapControl.remove(oKey);
                return null;
                }

            Object oValue = cacheRemote.get(oKey);
            return processDeferredEvents(oKey, oValue, listEvent, false);
            }
        finally
            {
            mapControl.unlock(oKey);
            }
        }

    /**
    * {@inheritDoc}
    */
    public Object put(Object oKey, Object oValue)
        {
        // on failback, ignore updates
        if (!getContext().isKeyOwned(oKey))
            {
            return null;
            }

        ConcurrentMap mapControl = getControlMap();
        mapControl.lock(oKey, -1L);
        try
            {
            NamedCache cacheRemote = ensureRemoteCache();
            if (!cacheRemote.isActive())
                {
                // the remote cache is unavailable; update both the delta
                // and internal map with the new mapping
                getDeltaMap().put(oKey, oValue);
                return getInternalMap().put(oKey, oValue);
                }

            List listEvent = new ArrayList(1);
            mapControl.put(oKey, listEvent);
            try
                {
                if (!getInternalMap().containsKey(oKey))
                    {
                    getRemoteMapListener().register(oKey);
                    }

                // perform a "blind" put() on the remote cache
                cacheRemote.putAll(Collections.singletonMap(oKey, oValue));
                }
            catch (RuntimeException e)
                {
                validateRemoteException(e);

                // the remote cache is no longer available; update both the
                // delta and internal map with the new mapping and remove
                // the event "net", as reconciliation is now due
                mapControl.remove(oKey);
                getDeltaMap().put(oKey, oValue);
                return getInternalMap().put(oKey, oValue);
                }

            return processDeferredEvents(oKey, oValue, listEvent, true);
            }
        finally
            {
            mapControl.unlock(oKey);
            }
        }

    /**
    * {@inheritDoc}
    */
    public void putAll(Map map)
        {
        for (Iterator iter = map.entrySet().iterator(); iter.hasNext(); )
            {
            Map.Entry entry = (Map.Entry) iter.next();
            put(entry.getKey(), entry.getValue());
            }
        }

    /**
    * {@inheritDoc}
    */
    public Object remove(Object oKey)
        {
        ConcurrentMap mapControl = getControlMap();
        mapControl.lock(oKey, -1L);
        try
            {
            Map mapInternal = getInternalMap();
            if (mapInternal.containsKey(oKey))
                {
                Object oValue;
                try
                    {
                    getRemoteMapListener().unregister(oKey);
                    }
                catch (RuntimeException e)
                    {
                    validateRemoteException(e);
                    }
                finally
                    {
                    oValue = mapInternal.remove(oKey);
                    }

                return oValue;
                }
            else
                {
                return null;
                }
            }
        finally
            {
            mapControl.unlock(oKey);
            }
        }


    // ----- entry Set implementation ---------------------------------------

    /**
    * {@inheritDoc}
    */
    protected Set instantiateEntrySet()
        {
        return new EntrySet();
        }

    /**
    * {@inheritDoc}
    */
    protected class EntrySet
            extends WrapperEntrySet
        {
        // ----- constructors ---------------------------------------------

        /**
        * {@inheritDoc}
        */
        public EntrySet()
            {
            super();
            }

        // ----- Set interface --------------------------------------------

        /**
        * {@inheritDoc}
        */
        public boolean remove(Object o)
            {
            Map.Entry entry  = (Map.Entry) o;
            Object    oKey   = entry.getKey();
            Object    oValue = entry.getValue();

            ConcurrentMap mapControl = getControlMap();
            mapControl.lock(oKey, -1L);
            try
                {
                Map mapInternal = getInternalMap();
                if (equals(oValue, mapInternal.get(oKey)))
                    {
                    try
                        {
                        getRemoteMapListener().unregister(oKey);
                        }
                    catch (RuntimeException e)
                        {
                        validateRemoteException(e);
                        }
                    finally
                        {
                        mapInternal.remove(oKey);
                        }

                    return true;
                    }
                else
                    {
                    return false;
                    }
                }
            finally
                {
                mapControl.unlock(oKey);
                }
            }
        }


    // ----- key Set implementation -----------------------------------------

    /**
    * {@inheritDoc}
    */
    protected Set instantiateKeySet()
        {
        return new KeySet();
        }

    /**
    * {@inheritDoc}
    */
    protected class KeySet
            extends WrapperKeySet
        {
        // ----- constructors ---------------------------------------------

        /**
        * {@inheritDoc}
        */
        public KeySet()
            {
            super();
            }

        // ----- Set interface --------------------------------------------

        /**
        * {@inheritDoc}
        */
        public boolean remove(Object oKey)
            {
            ConcurrentMap mapControl = getControlMap();
            mapControl.lock(oKey, -1L);
            try
                {
                Map mapInternal = getInternalMap();
                if (mapInternal.containsKey(oKey))
                    {
                    try
                        {
                        getRemoteMapListener().unregister(oKey);
                        }
                    catch (RuntimeException e)
                        {
                        validateRemoteException(e);
                        }
                    finally
                        {
                        mapInternal.remove(oKey);
                        }

                    return true;
                    }
                else
                    {
                    return false;
                    }
                }
            finally
                {
                mapControl.unlock(oKey);
                }
            }
        }


    // ----- value Collection implementation --------------------------------

    /**
    * {@inheritDoc}
    */
    protected Collection instantiateValueCollection()
        {
        return new ValueCollection();
        }

    /**
    * {@inheritDoc}
    */
    protected class ValueCollection
            extends WrapperValueCollection
        {
        // ----- constructors ---------------------------------------------

        /**
        * {@inheritDoc}
        */
        public ValueCollection()
            {
            super();
            }

        // ----- Collection interface -------------------------------------

        /**
        * {@inheritDoc}
        */
        public boolean remove(Object o)
            {
            for (Iterator iter = entrySet().iterator(); iter.hasNext(); )
                {
                Map.Entry entry = (Map.Entry) iter.next();
                if (equals(o, entry.getValue()))
                    {
                    iter.remove();
                    return true;
                    }
                }

            return false;
            }
        }


    // ----- remote MapListener support -------------------------------------

    /**
    * @return a MapListener responsible for keeping the internal map coherent
    *         with the remote cache
    */
    protected RemoteMapListener instantiateRemoteMapListener()
        {
        return new RemoteMapListener();
        }

    /**
    * MapListener for the remote cache responsible for keeping the internal
    * map coherent with the remote cache. This listener is registered as a
    * synchronous listener for heavy events.
    */
    protected class RemoteMapListener
            extends MultiplexingMapListener
            implements MapListenerSupport.SynchronousListener
        {
        // ----- MultiplexingMapListener methods --------------------------

        /**
        * {@inheritDoc}
        */
        public void onMapEvent(MapEvent evt)
            {
            ConcurrentMap mapControl = getControlMap();
            Object        oKey       = evt.getKey();
            long          ldtStart   = 0;

            for (int i = 0; true; ++i)
                {
                if (mapControl.lock(oKey, 0))
                    {
                    try
                        {
                        List listEvents = (List) mapControl.get(oKey);
                        if (listEvents == null)
                            {
                            // not in use; update the front entry
                            getInternalMap().put(oKey, evt.getNewValue());
                            }
                        else
                            {
                            listEvents.add(evt);
                            }
                        return;
                        }
                    finally
                        {
                        mapControl.unlock(oKey);
                        }
                    }
                else
                    {
                    // check for a key based action
                    List listEvent = (List) mapControl.get(oKey);
                    if (listEvent == null)
                        {
                        // check for a global action
                        listEvent = (List) mapControl.get(GLOBAL_KEY);
                        if (listEvent == null)
                            {
                            // has not been assigned yet, or has been just
                            // removed or switched; try again
                            Thread.yield();
                            long ldtNow = getSafeTimeMillis();
                            if (ldtStart == 0)
                                {
                                ldtStart = ldtNow;
                                }
                            else if (i > 5000 && ldtNow - ldtStart > 5000)
                                {
                                // we've been spinning and have given the other
                                // thread ample time to register the event list;
                                // the control map is corrupt
                                err("Detected a state corruption on the key \""
                                    + oKey + "\", of class "
                                    + oKey.getClass().getName()
                                    + " which is missing from the active key set "
                                    + mapControl.keySet()
                                    + ". This could be caused by a mutating or "
                                    + "inconsistent key implementation, or a "
                                    + "concurrent modification to the map passed to "
                                    + getClass().getName() + ".putAll()");

                                getInternalMap().put(oKey, evt.getNewValue());
                                return;
                                }
                            continue;
                            }
                        }

                    synchronized (listEvent)
                        {
                        List listKey = (List) mapControl.get(oKey);
                        if (listEvent == listKey || (listKey == null &&
                            listEvent == mapControl.get(GLOBAL_KEY)))
                            {
                            listEvent.add(evt);
                            return;
                            }
                        }
                    }
                }
            }

        // ----- helper registration methods ------------------------------

        /**
        * Register this listener with the remote cache for the specified key.
        */
        public void register(Object oKey)
            {
            NamedCache cacheRemote = getRemoteCache();
            if (cacheRemote != OFFLINE_CACHE)
                {
                cacheRemote.addMapListener(this, oKey, false);
                }
            }

        /**
        * Unregister this listener with the remote cache for the specified key.
        */
        public void unregister(Object oKey)
            {
            NamedCache cacheRemote = getRemoteCache();
            if (cacheRemote != OFFLINE_CACHE)
                {
                cacheRemote.removeMapListener(this, oKey);
                }
            }
        }


    // ----- remote MemberListener support ----------------------------------

    /**
    * @return a MemberListener responsible for invalidating the internal map
    *         in case of remote CacheService (re)start; must not be null
    */
    protected RemoteServiceListener instantiateRemoteServiceListener()
        {
        return new RemoteServiceListener();
        }

    /**
    * MemberListener for the remote cache's service.
    * <p>
    * The primary goal of this listener is reconciliation of the internal map
    * in case of remote CacheService (re)start.
    */
    protected class RemoteServiceListener
            implements MemberListener
        {
        // ----- MemberListener interface ---------------------------------

        /**
        * Invoked when a Member has joined the service.
        */
        public void memberJoined(MemberEvent evt)
            {
            if (evt.isLocal())
                {
                scheduleReconciliationTask();
                }
            }

        /**
        * Invoked when a Member is leaving the service.
        */
        public void memberLeaving(MemberEvent evt)
            {
            }

        /**
        * Invoked when a Member has left the service.
        */
        public void memberLeft(MemberEvent evt)
            {
            }

        // ----- helper registration methods ------------------------------

        /**
        * Register this listener with the remote CacheService.
        */
        public void register()
            {
            NamedCache cacheRemote = getRemoteCache();
            if (cacheRemote != OFFLINE_CACHE)
                {
                cacheRemote.getCacheService().addMemberListener(this);
                }
            }

        /**
        * Unregister this listener with the remote CacheService.
        */
        public void unregister()
            {
            NamedCache cacheRemote = getRemoteCache();
            if (cacheRemote != OFFLINE_CACHE)
                {
                cacheRemote.getCacheService().removeMemberListener(this);
                }
            }
        }


    // ----- remote CacheService restart support ----------------------------

    /**
    * @return a Runnable responsible for keeping the remote cache running;
    *         must not be null
    */
    protected RemoteCacheRestartTask instantiateRemoteCacheRestartTask()
        {
        return new RemoteCacheRestartTask();
        }

    /**
    * Runnable implementation responsible for keeping the remote cache
    * running.
    */
    protected class RemoteCacheRestartTask
            extends Base
            implements Runnable
        {
        // ----- Runnable interface ---------------------------------------

        /**
        * {@inheritDoc}
        */
        public synchronized void run()
            {
            if (m_fScheduled)
                {
                NamedCache cache = ensureRemoteCache();
                if (!cache.isActive())
                    {
                    try
                        {
                        cache.size();
                        }
                    catch (RuntimeException e) {}
                    }

                RESTART_DAEMON.scheduleTask(this, getSafeTimeMillis()
                        + m_cPeriodMillis);
                }
            }

        // ----- helper schedule methods ----------------------------------

        /**
        * Schedule this Runnable for execution.
        *
        * @param cPeriodSec the time in seconds between successive task
        *                   executions
        *
        * @throws IllegalStateException if the task has already been scheduled
        */
        public synchronized void schedule(int cPeriodSec)
            {
            if (m_cPeriodMillis > 0)
                {
                throw new IllegalStateException();
                }

            if (cPeriodSec <= 0)
                {
                throw new IllegalArgumentException("Illegal period: "
                        + cPeriodSec);
                }

            long cPeriodMillis = m_cPeriodMillis = cPeriodSec * 1000L;
            m_fScheduled = true;

            RESTART_DAEMON.scheduleTask(this, getSafeTimeMillis()
                    + cPeriodMillis);
            }

        /**
        * Stop execution of this task.
        */
        public synchronized void cancel()
            {
            m_fScheduled = false;
            }

        // ----- data members ---------------------------------------------

        /**
        * The execution period in milliseconds; only applicable if the task
        * has been scheduled.
        */
        protected long m_cPeriodMillis;

        /**
        * True iff this task has been scheduled for execution.
        */
        protected boolean m_fScheduled;
        }


    // ----- internal map synchronization support ---------------------------

    /**
    * @return a TaskDaemon responsible for reconciling the internal map with
    *         the remote cache after a period in which the remove cache was
    *         unavailable; must not be null
    */
    protected TaskDaemon instantiateReconciliationDaemon()
        {
        return new TaskDaemon("ExtendBackingMap:ReconcileDeamon("
                + getRemoteCacheName() + ')', Thread.NORM_PRIORITY, false);
        }

    /**
    * @return a Runnable responsible for reconciling the internal map with
    *         the remote cache after a period in which the remote cache was
    *         unavailable; must not be null
    */
    protected ReconciliationTask instantiateReconciliationTask()
        {
        return new ReconciliationTask();
        }

    /**
    * Schedule a new ReconciliationTask for execution.
    */
    protected void scheduleReconciliationTask()
        {
        instantiateReconciliationTask().schedule();
        }

    /**
    * Runnable implementation responsible for reconciling the internal map
    * with the remote cache after a period in which the remote cache was
    * unavailable.
    */
    protected class ReconciliationTask
            extends Base
            implements Runnable
        {
        // ----- Runnable interface ---------------------------------------

        /**
        * {@inheritDoc}
        */
        public void run()
            {
            ConcurrentMap mapControl = getControlMap();
            mapControl.lock(ConcurrentMap.LOCK_ALL, -1L);
            try
                {
                CacheFactory.log("Reconciling ExtendBackingMap.",
                        CacheFactory.LOG_INFO);

                preReconcile();
                try
                    {
                    reconcile();
                    }
                finally
                    {
                    postReconcile();
                    }

                CacheFactory.log("Finished reconciliation.",
                        CacheFactory.LOG_INFO);
                }
            catch (RuntimeException e)
                {
                CacheFactory.log("Error reconciling ExtendBackingMap: "
                        + printStackTrace(e), CacheFactory.LOG_ERR);
                }
            finally
                {
                mapControl.unlock(ConcurrentMap.LOCK_ALL);
                }
            }

        // ----- helper schedule methods ----------------------------------

        /**
        * Schedule this Runnable for execution.
        */
        public void schedule()
            {
            getReconciliationDaemon().executeTask(this);
            }

        // ----- reconciliation method ------------------------------------

        /**
        * Perform pre-reconciliation activities.
        * <p/>
        * This method adds a list to the control map to "catch" all events
        * that arrive while the reconciliation is in progress. This list
        * is then used during the post-reconciliation step to update the
        * internal map with the new values included in these deferred events.
        */
        protected void preReconcile()
            {
            // add a global list to catch all events while reconciling
            getControlMap().put(GLOBAL_KEY, new LinkedList());

            // make sure a listener has been registered for all keys
            RemoteMapListener listener = getRemoteMapListener();
            for (Iterator iter = getInternalMap().entrySet().iterator(); iter.hasNext(); )
                {
                listener.register(((Map.Entry) iter.next()).getKey());
                }
            }

        /**
        * Reconcile the internal map with the remote cache after a period in
        * which the remote cache was unavailable.
        * <p/>
        * This implementation refreshes the mappings in the internal map with
        * those in the remote cache; however, subclasses could make use of
        * the {@link ExtendBackingMap#getDeltaMap() delta map} to perform a
        * more sophisticated reconciliation.
        */
        protected void reconcile()
            {
            Map        mapInternal = getInternalMap();
            NamedCache cacheRemote = getRemoteCache();
            Object[]   aoKey       = mapInternal.keySet().toArray();
            int        cEntry      = aoKey.length;
            int        nBatch      = getBatchSize();
            List       listKey     = new ArrayList(nBatch);

            azzert(cacheRemote != OFFLINE_CACHE);

            for (int i = 0; cEntry > 0; )
                {
                listKey.clear();
                for (int j = 0, c = Math.min(nBatch, cEntry); j < c; ++i, ++j)
                    {
                    Object oKey = aoKey[i];
                    listKey.add(oKey);

                    // handle the case where the remote entry was removed;
                    // in this case, we still want to listen to future events
                    // for this key, so map this key to null in the internal
                    // map prior to performing a bulk load
                    mapInternal.put(oKey, null);
                    }

                mapInternal.putAll(cacheRemote.getAll(listKey));
                cEntry -= listKey.size();
                }
            }

        /**
        * Perform post-reconciliation activities.
        * <p/>
        * This method updates the internal map with an deferred events that
        * arrived while the reconciliation was in progress. It then removes
        * the global event "net" and clears the delta map.
        */
        protected void postReconcile()
            {
            Map mapControl  = getControlMap();
            Map mapInternal = getInternalMap();

            // update the internal map using the events that arrived during
            // reconciliation
            List list = (List) mapControl.get(GLOBAL_KEY);
            synchronized (list)
                {
                for (Iterator iter = list.iterator(); iter.hasNext(); )
                    {
                    MapEvent evt = (MapEvent) iter.next();
                    mapInternal.put(evt.getKey(), evt.getNewValue());
                    }
                mapControl.remove(GLOBAL_KEY);
                }

            getDeltaMap().clear();
            }

        // ----- accessors -----------------------------------------------

        /**
        * @return the batch size used during "bulk" operations, such as
        *         refreshing the internal map
        */
        public int getBatchSize()
            {
            return m_nBatch;
            }

        /**
        * Configure the batch sized used during "bulk" operations.
        *
        * @param cEntry  the new batch size
        */
        protected void setBatchSize(int cEntry)
            {
            if (cEntry < 1)
                {
                throw new IllegalArgumentException();
                }
            m_nBatch = cEntry;
            }


        // ----- data members ---------------------------------------------

        /**
        * The batch size used during "bulk" operations.
        */
        protected int m_nBatch = 100;
        }


    // ----- helper methods -------------------------------------------------

    /**
    * Return a reference to the remote NamedCache.
    *
    * @return the remote NamedCache; must not be null
    */
    protected synchronized NamedCache ensureRemoteCache()
        {
        NamedCache cache = m_cacheRemote;
        if (cache == OFFLINE_CACHE)
            {
            try
                {
                // retrieve the CacheService by configured name
                Service service = CacheFactory.getConfigurableCacheFactory()
                        .ensureService(m_sServiceName);

                // make sure the CacheService is in fact a remote CacheService
                if (CacheService.TYPE_REMOTE.equals(service.getInfo().getServiceType()))
                    {
                    // retrieve the remote cache by configured name
                    cache = ((CacheService) service).ensureCache(m_sCacheName,
                            getContext().getClassLoader());
                    cache = m_cacheRemote = instantiateConverterNamedCache(cache);

                    // register the service listener
                    getRemoteServiceListener().register();

                    // reconcile any updates that were made while the
                    // ExtendBackingMap was offline
                    scheduleReconciliationTask();

                    log("ExtendBackingMap is now online.");
                    }
                else
                    {
                    throw new IllegalArgumentException("The Service with name\""
                            + m_sServiceName
                            + "\" is not a remote CacheService");
                    }
                }
            catch (RuntimeException e)
                {
                log("Error retrieving remote NamedCache \"" + m_sCacheName
                        + "\" from CacheService \"" + m_sServiceName
                        + "\" (" + e + ");"
                        + " the ExtendBackingMap will remain offline.");
                }
            }

        return cache;
        }

    /**
    * @return a map that is used for synchronizing the addition and removal
    *         of remote cache listeners and internal map access; must not be
    *         null
    */
    protected ConcurrentMap instantiateControlMap()
        {
        return new WrapperConcurrentMap(new SafeHashMap(), false, 0L);
        }

    /**
    * @return A map used to record changes to the internal map that were made
    *         while the remote cache was unavailable.
    */
    protected Map instantiateDeltaMap()
        {
        return new SafeHashMap();
        }

    /**
    * Create a NamedCache that views the specified NamedCache through the
    * the CacheService's BackingMapManagerContext object that it provided to
    * the BackingMapManager that created this backing map.
    *
    * @param cache  the NamedCache to wrap
    *
    * @return a converter wrapper around the specified cache
    */
    protected NamedCache instantiateConverterNamedCache(NamedCache cache)
        {
        BackingMapManagerContext ctx = getContext();
        return new ConverterNamedCache(cache,
                ctx.getKeyToInternalConverter(),
                ctx.getKeyFromInternalConverter(),
                ctx.getValueToInternalConverter(),
                ctx.getValueFromInternalConverter());
        }

    /**
    * Convert the given key into its internal form. If an exception occurs
    * during the conversion, return the "external" form.
    *
    * @param oKey  the key
    *
    * @return the internal form of the specified key
    */
    protected Object convertKeyToInternal(Object oKey)
        {
        try
            {
            return getContext().getKeyToInternalConverter().convert(oKey);
            }
        catch (RuntimeException e)
            {
            return oKey;
            }
        }

    /**
    * Convert the given key from its internal form. If an exception occurs
    * during the conversion, return the internal form.
    *
    * @param oKey  the key
    *
    * @return the "external" form of the specified key
    */
    protected Object convertKeyFromInternal(Object oKey)
        {
        try
            {
            return getContext().getKeyFromInternalConverter().convert(oKey);
            }
        catch (RuntimeException e)
            {
            return oKey;
            }
        }

    /**
    * Convert the given value into its internal form. If an exception occurs
    * during the conversion, return the "external" form.
    *
    * @param oValue  the value
    *
    * @return the internal form of the specified value
    */
    protected Object convertValueToInternal(Object oValue)
        {
        try
            {
            return getContext().getValueToInternalConverter().convert(oValue);
            }
        catch (RuntimeException e)
            {
            return oValue;
            }
        }

    /**
    * Convert the given value from its internal form. If an exception occurs
    * during the conversion, return the internal form.
    *
    * @param oValue  the value
    *
    * @return the "external" form of the specified value
    */
    protected Object convertValueFromInternal(Object oValue)
        {
        try
            {
            return getContext().getValueFromInternalConverter().convert(oValue);
            }
        catch (RuntimeException e)
            {
            return oValue;
            }
        }

    /**
    * Update the specified key in the internal map with the given value,
    * taking into account any deferred events that occured during a remote
    * operation.
    *
    * @param oKey       the key of the entry to update
    * @param oValue     the new value
    * @param listEvent  the event "net"
    * @param fPut       if true, the internal map is being updated due to a
    *                   remote put()
    *
    * @return the result of the remote operation
    */
    protected Object processDeferredEvents(Object oKey, Object oValue,
            List listEvent, boolean fPut)
        {
        // update the internal map while synchronizing on the event list.
        // If the event list is empty, the value added to the map is the
        // value passed to this method; otherwise the value is the new value
        // associated with the last event
        synchronized (listEvent)
            {
            // remove the event "net" while holding synchronization
            getControlMap().remove(oKey);

            int cEvent = listEvent.size();
            if (cEvent > 0)
                {
                oValue = ((MapEvent) listEvent.get(cEvent - 1))
                        .getNewValue();
                }
            else if (fPut)
                {
                CacheFactory.log("Expected an insert/update event for key\""
                        + convertKeyFromInternal(oKey)
                        + "\", but none have been received.",
                        CacheFactory.LOG_WARN);
                }

            Object oValueOld = getInternalMap().put(oKey, oValue);
            return fPut ? oValueOld : oValue;
            }
        }

    /**
    * Validate that the given exception was potentially thrown due to a
    * communication error with a remote cluster.
    *
    * @param e  the exception to validate
    */
    protected void validateRemoteException(RuntimeException e)
        {
        if (getRemoteCache().isActive() && !(e instanceof ClusterException))
            {
            CacheFactory.log("Logging for debugging purposes only; "
                    + "this exception can be safely disregarded: "
                    + printStackTrace(e), 5);
            }
        }


    // ----- accessors ------------------------------------------------------

    /**
    * @return the CacheService's BackingMapManagerContext object that it
    *         provided to the BackingMapManager that created this backing
    *         map
    */
    public BackingMapManagerContext getContext()
        {
        return m_ctx;
        }

    /**
    * @return the ConcurrentMap for synchronizing the addition and removal of
    *         remote cache listeners and access to the internal map
    */
    protected ConcurrentMap getControlMap()
        {
        return m_mapControl;
        }

    /**
    * @return the local replica of the remote cache
    */
    protected ObservableMap getInternalMap()
        {
        return (ObservableMap) getMap();
        }

    /**
    * @return a map used to record the changes to the internal map that were
    *         made while the remote cache was unavailable.
    */
    protected Map getDeltaMap()
        {
        return m_mapDelta;
        }

    /**
    * @return the remote cache
    */
    protected NamedCache getRemoteCache()
        {
        return m_cacheRemote;
        }

    /**
    * @return the name of the remote cache
    */
    protected String getRemoteCacheName()
        {
        return m_sCacheName;
        }

    /**
    * @return the Runnable responsible for keeping the remote cache running
    */
    protected RemoteCacheRestartTask getRemoteCacheRestartTask()
        {
        return m_taskRestart;
        }

    /**
    * @return the MapListener responsible for keeping the internal map
    *         coherent with the remote cache
    */
    protected RemoteMapListener getRemoteMapListener()
        {
        return m_listenerMap;
        }

    /**
    * @return the MemberListener responsible for internal map invalidation in
    *         case of remote CacheService (re)start
    */
    protected RemoteServiceListener getRemoteServiceListener()
        {
        return m_listenerService;
        }

    /**
    * @return the name of the remote CacheService
    */
    protected String getRemoteServiceName()
        {
        return m_sServiceName;
        }

    /**
    * @return the TaskDaemon responsible for reconciling the internal map
    *         with the remote cache after a period in which the remove cache
    *         was unavailable
    */
    protected TaskDaemon getReconciliationDaemon()
        {
        return m_daemonReconcile;
        }


    // ----- constants ------------------------------------------------------

    /**
    * The default number of seconds between successive restart attempts of
    * the remote NamedCache.
    */
    public final static int DEFAULT_RESTART = 5;

    /**
    * A unique Object that serves as a control key for global operations
    * such as clear.
    */
    protected final Object GLOBAL_KEY = new Object();

    /**
    * Empty list that ignores any add operations.
    */
    protected final static List IGNORE_LIST = new ImmutableArrayList(new Object[0])
        {
        public boolean add(Object o)
            {
            return true;
            }
        };

    /**
    * A singleton NamedCache representing an "offline" remote cache.
    */
    protected final static NamedCache OFFLINE_CACHE = new WrapperNamedCache(
            NullImplementation.getMap(), null)
        {
        public boolean isActive()
            {
            return false;
            }
        };

    /**
    * The restart TaskDaemon.
    */
    protected final static TaskDaemon RESTART_DAEMON = new TaskDaemon(
            "ExtendBackingMap:RestartDeamon", Thread.NORM_PRIORITY, false);


    // ----- data members ---------------------------------------------------

    /**
    * The CacheService's BackingMapManagerContext object that it provided to
    * the BackingMapManager that created this backing map.
    */
    protected final BackingMapManagerContext m_ctx;

    /**
    * The remote CacheService name.
    */
    protected final String m_sServiceName;

    /**
    * The remote cache name.
    */
    protected final String m_sCacheName;

    /**
    * The ConcurrentMap for synchronizing the addition and removal of remote
    * cache listeners and internal map access.
    */
    protected final ConcurrentMap m_mapControl;

    /**
    * A map used to record the changes to the internal map that were made
    * while the remote cache was unavailable.
    */
    protected final Map m_mapDelta;

    /**
    * The MapListener responsible for keeping the internal map coherent with
    * the remote cache.
    */
    protected final RemoteMapListener m_listenerMap;

    /**
    * The MemberListener responsible for internal map invalidation in case of
    * remote CacheService (re)start.
    */
    protected final RemoteServiceListener m_listenerService;

    /**
    * The Runnable responsible for keeping the remote cache running.
    */
    protected final RemoteCacheRestartTask m_taskRestart;

    /**
    * The TaskDaemon responsible for reconciling the ExtendBackingMap after
    * a disconnect.
    */
    protected final TaskDaemon m_daemonReconcile;

    /**
    * The remote cache.
    */
    protected volatile NamedCache m_cacheRemote = OFFLINE_CACHE;
    }