/*
* ConverterNamedCache.java
*
* Copyright 2001-2007 by Oracle. All rights reserved.
*
* Oracle is a registered trademarks of Oracle Corporation and/or its affiliates.
*
* This software is the confidential and proprietary information of
* Oracle Corporation. You shall not disclose such confidential and
* proprietary information and shall use it only in accordance with the
* terms of the license agreement you entered into with Oracle.
*
* This notice may not be removed or altered.
*/
package com.tangosol.examples.extend;


import com.tangosol.net.CacheService;
import com.tangosol.net.NamedCache;

import com.tangosol.util.Converter;
import com.tangosol.util.ConverterCollections;
import com.tangosol.util.Filter;
import com.tangosol.util.MapListener;

import java.util.Collection;
import java.util.Map;
import java.util.Set;


/**
* A NamedCache that views an underlying ObservableMap through a set of key
* and value Converters.
*
* @author jh  2007.09.19
*/
public class ConverterNamedCache
        extends ConverterCollections.ConverterObservableMap
        implements NamedCache
    {
    // ----- constructors ---------------------------------------------------

    /**
    * Constructor.
    *
    * @param cache        the underlying NamedCache
    * @param convKeyUp    the Converter to view the underlying
    *                     SortedMap's keys through
    * @param convKeyDown  the Converter to use to pass keys down to the
    *                     underlying SortedMap
    * @param convValUp    the Converter to view the underlying
    *                     SortedMap's values through
    * @param convValDown  the Converter to use to pass values down to the
    *                     underlying SortedMap
    */
    public ConverterNamedCache(NamedCache cache, Converter convKeyUp, Converter convKeyDown, Converter convValUp, Converter convValDown)
        {
        super(cache, convKeyUp, convKeyDown, convValUp, convValDown);

        m_cache = cache;
        }


    // ----- NamedCache interface -------------------------------------------

    /**
    * {@inheritDoc}
    */
    public void addMapListener(MapListener listener, Object oKey, boolean fLite)
        {
        super.addMapListener(listener, m_convKeyDown.convert(oKey), fLite);
        }

    /**
    * {@inheritDoc}
    */
    public void removeMapListener(MapListener listener, Object oKey)
        {
        super.removeMapListener(listener, m_convKeyDown.convert(oKey));
        }

    /**
    * {@inheritDoc}
    */
    public boolean lock(Object oKey)
        {
        return m_cache.lock(m_convKeyDown.convert(oKey));
        }

    /**
    * {@inheritDoc}
    */
    public boolean lock(Object oKey, long cWait)
        {
        return m_cache.lock(m_convKeyDown.convert(oKey), cWait);
        }

    /**
    * {@inheritDoc}
    */
    public boolean unlock(Object oKey)
        {
        return m_cache.unlock(m_convKeyDown.convert(oKey));
        }

    /**
    * {@inheritDoc}
    */
    public Object aggregate(Collection collKeys, EntryAggregator agent)
        {
        Collection colKeysConv = collKeys instanceof Set ?
                instantiateSet((Set) collKeys, m_convKeyDown, m_convKeyUp) :
                instantiateCollection(collKeys, m_convKeyDown, m_convKeyUp);

        return m_cache.aggregate(colKeysConv, agent);
        }

    /**
    * {@inheritDoc}
    */
    public Object aggregate(Filter filter, EntryAggregator agent)
        {
        return m_cache.aggregate(filter, agent);
        }

    /**
    * {@inheritDoc}
    */
    public Object invoke(Object oKey, EntryProcessor agent)
        {
        return m_cache.invoke(m_convKeyDown.convert(oKey), agent);
        }

    /**
    * {@inheritDoc}
    */
    public Map invokeAll(Collection collKeys, EntryProcessor agent)
        {
        Collection colKeysConv = collKeys instanceof Set ?
                instantiateSet((Set) collKeys, m_convKeyDown, m_convKeyUp) :
                instantiateCollection(collKeys, m_convKeyDown, m_convKeyUp);

        return m_cache.invokeAll(colKeysConv, agent);
        }

    /**
    * {@inheritDoc}
    */
    public Map invokeAll(Filter filter, EntryProcessor agent)
        {
        return m_cache.invokeAll(filter, agent);
        }

    /**
    * {@inheritDoc}
    */
    public String getCacheName()
        {
        return m_cache.getCacheName();
        }

    /**
    * {@inheritDoc}
    */
    public CacheService getCacheService()
        {
        return m_cache.getCacheService();
        }

    /**
    * {@inheritDoc}
    */
    public boolean isActive()
        {
        return m_cache.isActive();
        }

    /**
    * {@inheritDoc}
    */
    public void release()
        {
        m_cache.release();
        }

    /**
    * {@inheritDoc}
    */
    public void destroy()
        {
        m_cache.destroy();
        }


    // ----- data members ---------------------------------------------------

    /**
    * The wrapped NamedCache.
    */
    protected final NamedCache m_cache;
    }