/*
* SessionServlet.java
*
* Copyright 2000-2005 by Tangosol, Inc.  All rights reserved.
*
* This software is the confidential and proprietary information of
* Tangosol, Inc.  You shall not disclose such confidential and pro-
* prietary information and shall use it only in accordance with the
* terms of the license agreement you entered into with Tangosol, Inc.
*
* Tangosol, Inc. makes no representations or warranties about the suit-
* ability of the software, either express or implied, including but not
* limited to the implied warranties of merchantability, fitness for a
* particular purpose, or non-infringement.  Tangosol, Inc. shall not be
* liable for any damages suffered by licensee as a result of using,
* modifying or distributing this software or its derivatives.
*
* Tangosol, Inc. is located at http://www.tangosol.com and can be
* contacted by e-mail at info@tangosol.com.
*
* This notice may not be removed or altered.
*/
package com.tangosol.examples.coherence.session;


import com.tangosol.net.CacheFactory;

import com.tangosol.util.Base;

import java.io.IOException;
import java.io.Serializable;

import javax.servlet.ServletException;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionActivationListener;
import javax.servlet.http.HttpSessionBindingEvent;
import javax.servlet.http.HttpSessionBindingListener;
import javax.servlet.http.HttpSessionEvent;


/**
* Controller Servlet for testing the Coherence Session Management Module.
* This Servlet intercepts HTTP requests and manipulates the ServletContext,
* Session, and HttpServletRequest based upon request parameters.
*
* @author erm 2003.09.12
* @author jh 2004.10.06
*
* @see HttpSessionActivationListener
* @see HttpSessionBindingListener
*
* @version 2.5
*/
public class SessionServlet
        extends HttpServlet
    {
    // ----- SessionServlet methods -------------------------------------------

    /**
    * Parses the given request and manipulates the ServletContext, Session,
    * and HttpServletRequest based upon request parameters.
    *
    * @param req the client HTTP request
    * @param res the client HTTP response
    *
    * @throws IOException      on I/O error
    * @throws ServletException on application error
    */
    protected void execute(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException
        {
        HttpSession session = req.getSession();

        // parse request parameters
        String sAction = req.getParameter("action");
        String sKey = req.getParameter("key");
        String sValue = req.getParameter("value");
        boolean fApp = "application".equals(req.getParameter("scope"));
        boolean fBind = req.getParameter("bind") != null;
        boolean fActivate = req.getParameter("activate") != null;

        // manipulate the application based upon the request parameters
        if (sAction != null && sAction.length() > 0)
            {
            switch (sAction.charAt(0))
                {
                // set an attribute. the attribute can either be a session or
                // servlet context attribute, and may implement
                // HttpSessionActivationListener, HttpSessionBindingListener,
                // or both.
                case 'a':
                    if (sKey != null && sKey.length() > 0)
                        {
                        Object oValue;

                        if (fBind && fActivate)
                            {
                            oValue = new BindActivateValue(sValue);
                            }
                        else if (fBind)
                            {
                            oValue = new BindValue(sValue);
                            }
                        else if (fActivate)
                            {
                            oValue = new ActivateValue(sValue);
                            }
                        else
                            {
                            oValue = sValue;
                            }

                        if (fApp)
                            {
                            getServletConfig().getServletContext()
                                    .setAttribute(sKey, oValue);
                            }
                        else
                            {
                            session.setAttribute(sKey, oValue);
                            }
                        }
                    break;

                case 'r':
                    // refresh the page
                    if (sAction.equals("remove") && sKey != null &&
                            sKey.length() > 0)
                        {
                        if (fApp)
                            {
                            getServletConfig().getServletContext()
                                    .removeAttribute(sKey);
                            }
                        else
                            {
                            session.removeAttribute(sKey);
                            }
                        }
                    break;

                // invalidate the session
                case 'i':
                    session.invalidate();
                    break;

                default:
                    CacheFactory.log("Unknown action: " + sAction, 1);
                }
            }

        // forward to the appropriate JSP
        String sJsp = fApp ? "/application.jsp" : "/session.jsp";
        req.getRequestDispatcher(sJsp).forward(req, res);
        }


    // ----- HttpServlet methods ----------------------------------------------

    protected void doGet(HttpServletRequest req, HttpServletResponse res)
            throws ServletException, IOException
        {
        execute(req, res);
        }

    protected void doPost(HttpServletRequest req, HttpServletResponse res)
            throws ServletException, IOException
        {
        execute(req, res);
        }


    // ----- inner classes ----------------------------------------------------

    /**
    * Wrapper class that implements HttpSessionBindingListener. When added to
    * an HttpSession, an instance of this class will log information when it is
    * bound or unbound from the session.
    *
    * @see HttpSessionBindingListener
    * @see HttpSessionBindingEvent
    */
    public static class BindValue
            extends Base
            implements HttpSessionBindingListener, Serializable
        {
        // ----- constructors -------------------------------------------------

        /**
        * Creates a new BindValue that wraps the given string.
        *
        * @param sValue the string to be wrapped
        */
        public BindValue(String sValue)
            {
            m_sValue = sValue;
            }


        // ----- BindValue methods --------------------------------------------

        /**
        * Logs information about an HttpSessionBindingEvent.
        *
        * @param sEvent contextual information
        * @param event  the event
        */
        protected void show(String sEvent, HttpSessionBindingEvent event)
            {
            CacheFactory.log(sEvent + "[" + m_sValue + "]"
                    + ": session=" + event.getSession().getId()
                    + ", name=" + event.getName()
                    + ", value=" + (event.getValue() == null ? "null"
                    : event.getValue() == this ? "this"
                    : String.valueOf(event.getValue())), 3);
            }


        // ----- HttpSessionBindingListener interface -------------------------

        public void valueBound(HttpSessionBindingEvent event)
            {
            show("valueBound", event);
            }

        public void valueUnbound(HttpSessionBindingEvent event)
            {
            show("valueUnbound", event);
            }


        // ----- Object methods -----------------------------------------------

        public String toString()
            {
            return m_sValue + " (bind)";
            }


        // ----- data members -------------------------------------------------

        /**
        * The wrapped string value.
        */
        protected String m_sValue;
        }

    /**
    * Wrapper class that implements HttpSessionActivationListener. When added
    * to an HttpSession, an instance of this class will log information when
    * the session is activated or passivated.
    *
    * @see HttpSessionActivationListener
    * @see HttpSessionEvent
    */
    public static class ActivateValue
            extends Base
            implements HttpSessionActivationListener, Serializable
        {
        // ----- constructors -------------------------------------------------

        /**
        * Creates a new ActivateValue that wraps the given string.
        *
        * @param sValue the string to be wrapped
        */
        public ActivateValue(String sValue)
            {
            m_sValue = sValue;
            }


        // ----- ActivateValue methods ----------------------------------------

        /**
        * Logs information about an HttpSessionEvent.
        *
        * @param sEvent contextual information
        * @param event  the event
        */
        protected void show(String sEvent, HttpSessionEvent event)
            {
            CacheFactory.log(sEvent + "[" + m_sValue + "]"
                    + ": session=" + event.getSession().getId(), 3);
            }


        // ----- HttpSessionActivationListener interface ----------------------

        public void sessionDidActivate(HttpSessionEvent se)
            {
            show("sessionDidActivate", se);
            }

        public void sessionWillPassivate(HttpSessionEvent se)
            {
            show("sessionWillPassivate", se);
            }


        // ----- Object methods -----------------------------------------------

        public String toString()
            {
            return m_sValue + " (activate)";
            }


        // ----- data members -------------------------------------------------

        /**
        * The wrapped string value.
        */
        protected String m_sValue;
        }

    /**
    * Wrapper class that implements HttpSessionBindingListener and
    * HttpSessionActivationListener. When added to an HttpSession, an instance
    * of this class will log information when it is bound or unbound from the
    * session or the session is activated or passivated.
    *
    * @see HttpSessionBindingListener
    * @see HttpSessionBindingEvent
    * @see HttpSessionActivationListener
    * @see HttpSessionEvent
    */
    public static class BindActivateValue
            extends ActivateValue
            implements HttpSessionBindingListener
        {
        // ----- constructors -------------------------------------------------

        /**
        * Creates a new BindActivateValue that wraps the given string.
        *
        * @param sValue the string to be wrapped
        */
        public BindActivateValue(String sValue)
            {
            super(sValue);
            }


        // ----- BindActivateValue methods ------------------------------------

        /**
        * Logs information about an HttpSessionBindingEvent.
        *
        * @param sEvent contextual information
        * @param event  the event
        */
        protected void show(String sEvent, HttpSessionBindingEvent event)
            {
            CacheFactory.log(sEvent + "[" + m_sValue + "]"
                    + ": session=" + event.getSession().getId()
                    + ", name=" + event.getName()
                    + ", value=" + (event.getValue() == null ? "null"
                    : event.getValue() == this ? "this"
                    : String.valueOf(event.getValue())), 3);
            }


        // ----- HttpSessionBindingListener interface -------------------------

        public void valueBound(HttpSessionBindingEvent event)
            {
            show("valueBound", event);
            }

        public void valueUnbound(HttpSessionBindingEvent event)
            {
            show("valueUnbound", event);
            }


        // ----- Object methods -----------------------------------------------

        public String toString()
            {
            return m_sValue + " (bind+activate)";
            }
        }
    }