package com.tangosol.examples.l2;


import org.apache.log4j.Logger;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;

import java.sql.Timestamp;
import java.util.Random;
import java.util.UUID;
import java.util.List;
import java.util.Collection;

import com.tangosol.net.NamedCache;
import com.tangosol.net.CacheFactory;


/**
 * This sample program demonstrates the use of a the Coherence Hibernate
 * L2 cache plugin. It populates a database with sample {@link SampleEntity}
 * objects via Hibernate and performs some queries.  Afterwards, the data that
 * Hibernate caches via Coherence is logged as a demonstration.
 *
 * See http://wiki.tangosol.com/display/COH33UG/Using+Coherence+and+Hibernate
 * for more details and further documentation.
 *
 * @author pperalta Jun 7, 2007
 */
public class L2Main
    {
    public static void main(String[] args)
        {
        // Set up Coherence properties.  Normally this is set
        // up on the java command line using the -D flag, but it
        // is done here for illustrative purposes.

        // Hibernate L2 caching uses the config file specified by
        // tangosol.coherence.hibernate.cacheconfig.  By setting
        // these system properties to point to the same file, we'll
        // be able to query the Hibernate cache using
        // CacheFactory.getCache(...)

        System.setProperty("tangosol.coherence.cacheconfig",
                "sample-cache-config.xml");
        System.setProperty("tangosol.coherence.hibernate.cacheconfig",
                "sample-cache-config.xml");

        SessionFactory sf = new Configuration().configure().buildSessionFactory();

        SampleEntityDao dao = new SampleEntityDao(sf);

        LOG.info("Populating values....");

        // Use this entity ID later to test the L2 cache
        String idToQuery = null;

        // Seed the cache/database with sample values
        for (int i = 0; i < 1000; i++)
            {
            SampleEntity se = new SampleEntity();
            se.setId(UUID.randomUUID().toString());
            se.setName(getRandomString());
            se.setDate(new Timestamp(System.currentTimeMillis()));
            se.setAge(i);

            dao.saveSimpleEntity(se);
            if (i == 1) idToQuery = se.getId();
            }

        LOG.info("Populated values...performing query to get all objects");

        // Perform a query using the DAO.  This DAO uses Hibernate to
        // perform the query.

        Collection<SampleEntity> allEntities = dao.getSimpleEntities();

        LOG.info("Total number of entities: " + allEntities.size());

        // Load a specific entity.  Since this item is already in L2,
        // this should not result in a SQL query being issued by Hibernate.
        // Log file looks like this with Hibernate cache and SQL logging
        // set to debug:
        // ...
        // L2Main [main] INFO - Loading ID 70f5968a-e528-4fd6-8f64-9dd3fdf2c117
        // ReadOnlyCache [main] DEBUG- Cache hit: com.tangosol.examples.l2.SampleEntity#70f...
        // L2Main [main] INFO - Loaded SampleEntity{id='70f5968a-e528-4fd6-8f64-9dd3fdf2c11...

        LOG.info("Loading ID " + idToQuery);

        SampleEntity se = dao.getSimpleEntity(idToQuery);

        LOG.info("Loaded " + se);

        // This query will NOT hit the cache, as it is a query with a parameter
        // However, once this query is cached, the next time it is executed
        // with the given parameter there will be a cache hit

        LOG.info("Performing query by name using wildcard");

        List<SampleEntity> results = dao.getSimpleEntityByName("i%");

        LOG.info("----- Query results -----");
        for (SampleEntity e : results)
            {
            LOG.info(e);
            }

        LOG.info("Query results total: " + results.size());

        // Now inspect the Coherence cache to see what Hibernate put
        // into it

        LOG.info("----- Cache inspection -----");

        // This cache contains the entities
        NamedCache entityCache = CacheFactory.getCache(SampleEntity.class.getName());
        LOG.info("entity cache items: " + entityCache);
        LOG.info("entity cache size: " + entityCache.size());

        // These caches are used for caching queries
        NamedCache standardQueryCache = CacheFactory.getCache(
                "org.hibernate.cache.StandardQueryCache");
        NamedCache timestampCache = CacheFactory.getCache(
                "org.hibernate.cache.UpdateTimestampsCache");

        LOG.info("query cache: " + standardQueryCache);
        LOG.info("query cache size: " + standardQueryCache.size());
        LOG.info("query cache contents: " + standardQueryCache.entrySet());

        LOG.info("timestamp cache: " + timestampCache);
        LOG.info("timestamp cache size: " + timestampCache.size());
        LOG.info("timestamp cache contents: " + timestampCache.entrySet());
        }

    /**
     * Generate a random string
     * @return a random string
     */
    public static String getRandomString()
        {
        Random rnd = RANDOM;
        int cch = 10;
        char[] ach = new char[cch];
        for (int of = 0; of < cch; ++of)
            {
            ach[of] = (char) ('a' + rnd.nextInt(26));
            }
        return new String(ach);
        }

    private static Random RANDOM = new Random(System.currentTimeMillis());

    private static final Logger LOG = Logger.getLogger(L2Main.class);

    }
