
package com.tangosol.examples.versionedbeans;


import com.tangosol.net.CacheFactory;
import com.tangosol.net.NamedCache;

import com.tangosol.util.ClassHelper;
import com.tangosol.util.CommandLine;

import java.util.Iterator;
import java.util.Map;


/**
* Test of clustered cache of Address objects.
*/
public class Test
        extends CommandLine
    {
    // ----- command line ---------------------------------------------------

    /**
    * Command line entry point.
    *
    * @param asArg  command line arguments
    */
    public static void main(String[] asArg)
        {
        new Test(asArg).run();
        }


    // ----- constructors ---------------------------------------------------

    /**
    * Command line test constructor.
    *
    * @param asArg  command line arguments
    */
    protected Test(String[] asArg)
        {
        super(asArg);
        }


    // ----- initialization -------------------------------------------------

    /**
    * Called when starting. This method should register all possible
    * commands.
    */
    public void init()
        {
        register("new"  , "doNew"  , "Creates a new current address");
        register("get"  , "doGet"  , "get <property> : Gets a property from the currrent address");
        register("set"  , "doSet"  , "set <property> <value> : Sets a property on the current address");
        register("load" , "doLoad" , "load <key> : Loads the specified address from the cache into the current address");
        register("store", "doStore", "store <key> : Stores the current address into the cache using the specified key");
        register("show" , "doShow" , "Shows the current address");
        register("list" , "doList" , "Lists the addresses in the cache");
        register("ver"  , "doVer"  , "Show the version of the Address implementation for this JVM");

        super.init();
        }

    // ----- commands -------------------------------------------------------

    public void doNew()
        {
        setAddress(new Address());
        doShow();
        }

    public void doGet(String[] asArgs)
        {
        Address addr = getAddress();
        if (addr == null)
            {
            out("*** no current address (use \"new\" or \"load\") ***");
            return;
            }

        String sProp = asArgs.length >= 2 ? asArgs[1] : null;
        if (sProp == null || sProp.length() == 0)
            {
            out("*** property name required (e.g. \"Street\", \"City\", \"State\") ***");
            return;
            }

        Object oResult;
        try
            {
            oResult = ClassHelper.invoke(addr, "get" + sProp, ClassHelper.VOID);
            }
        catch (Throwable e)
            {
            out("*** exception getting property ***");
            out(e);
            return;
            }
        
        out(sProp + "=" + oResult);
        }

    public void doSet(String[] asArgs)
        {
        Address addr = getAddress();
        if (addr == null)
            {
            out("*** no current address (use \"new\" or \"load\") ***");
            return;
            }

        String sProp = asArgs.length >= 2 ? asArgs[1] : null;
        if (sProp == null || sProp.length() == 0)
            {
            out("*** property name required (e.g. \"Street\", \"City\", \"State\") ***");
            return;
            }

        // this works only because all props happen to be Strings
        String sValue = null;
        for (int i = 2, c = asArgs.length; i < c; ++i)
            {
            String sPart = asArgs[i];
            if (sPart != null)
                {
                sValue = sValue == null ? sPart : sValue + ' ' + sPart;
                }
            }

        try
            {
            ClassHelper.invoke(addr, "set" + sProp, new Object[] {sValue});
            }
        catch (Throwable e)
            {
            out("*** exception setting property ***");
            out(e);
            return;
            }
        
        doShow();
        }

    public void doLoad(String[] asArgs)
        {
        String sKey = asArgs.length >= 2 ? asArgs[1] : null;
        if (sKey == null || sKey.length() == 0)
            {
            out("*** key required (e.g. \"1\", \"2\", \"3\", ...) ***");
            return;
            }

        Address addr = (Address) getCache().get(sKey);
        if (addr == null)
            {
            out("*** no such address (try \"list\") ***");
            return;
            }

        setAddress(addr);

        doShow();
        }

    public void doStore(String[] asArgs)
        {
        Address addr = getAddress();
        if (addr == null)
            {
            out("*** no current address (use \"new\" or \"load\") ***");
            return;
            }

        String sKey = asArgs.length >= 2 ? asArgs[1] : null;
        if (sKey == null || sKey.length() == 0)
            {
            out("*** key required (e.g. \"1\", \"2\", \"3\", ...) ***");
            return;
            }

        getCache().put(sKey, addr);
        }

    public void doShow()
        {
        out("Current=" + getAddress());
        }

    public void doList()
        {
        out("Cache.size()=" + getCache().size());
        int c = 0;
        for (Iterator iter = getCache().entrySet().iterator(); iter.hasNext(); )
            {
            Map.Entry entry = (Map.Entry) iter.next();
            Object    oKey  = entry.getKey();
            Address   addr  = (Address) entry.getValue();
            out(oKey + "=" + addr);
            ++c;
            }
        out("(iterated " + c + " Address objects)");
        }

    public void doVer()
        {
        try
            {
            out("Address.CURRENT_VERSION=" + Address.class.getField("CURRENT_VERSION").get(null));
            }
        catch (Throwable e)
            {
            out("*** could not get Address.CURRENT_VERSION ***");
            out(e);
            }
        }

    // ----- properties -----------------------------------------------------

    /**
    * Obtain the address cache.
    *
    * @return the cache containing the clustered address objects
    */
    public NamedCache getCache()
        {
        return m_cache;
        }

    /**
    * Obtain the current address.
    *
    * @return the current address
    */
    public Address getAddress()
        {
        return m_addr;
        }

    /**
    * Specify the current address.
    *
    * @param addr  the current address
    */
    public void setAddress(Address addr)
        {
        m_addr = addr;
        }


    // ----- data members ---------------------------------------------------

    /**
    * The Address cache.
    */
    private NamedCache m_cache = CacheFactory.getCache("Addresses");

    /**
    * The current Address.
    */
    private Address m_addr;
    }