
package com.tangosol.examples.coherence;


import com.tangosol.util.Binary;
import com.tangosol.util.SafeHashMap;

import com.tangosol.net.cache.CacheLoader;
import com.tangosol.net.cache.LocalCache;
import com.tangosol.net.cache.OldCache;


/**
 * This class is an extension to LocalCache that measures (and thus limits)
 * units in terms of bytes instead of the number of objects. In other words, the
 * LocalCache implementation assumes that each object is one unit, so if the
 * cache is configured to be limited to 1000 units, that means it is limited to
 * 1000 objects. With this implementation, the cache would be limited to 1000
 * bytes!
 *
 * @author Cameron 2004.11.19
 */
public class MemoryLimitedCache
    extends LocalCache
    {
    // ----- constructors ---------------------------------------------------

    /**
     * Construct the cache manager.
     */
    public MemoryLimitedCache()
        {
        super();
        }

    /**
     * Construct the cache manager.
     *
     * @param cUnits the number of units that the cache manager will cache
     *               before pruning the cache
     */
    public MemoryLimitedCache(int cUnits)
        {
        super(cUnits);
        }

    /**
     * Construct the cache manager.
     *
     * @param cUnits        the number of units that the cache manager will
     *                      cache before pruning the cache
     * @param cExpiryMillis the number of milliseconds that each cache entry
     *                      lives before being automatically expired
     */
    public MemoryLimitedCache(int cUnits, int cExpiryMillis)
        {
        super(cUnits, cExpiryMillis);
        }

    /**
     * Construct the cache manager.
     *
     * @param cUnits        the number of units that the cache manager will
     *                      cache before pruning the cache
     * @param cExpiryMillis the number of milliseconds that each cache entry
     *                      lives before being automatically expired
     * @param loader        the CacheLoader or CacheStore to use
     */
    public MemoryLimitedCache(int cUnits, int cExpiryMillis, CacheLoader loader)
        {
        super(cUnits, cExpiryMillis, loader);
        }

    // ----- inner class: Entry ---------------------------------------------

    /**
     * Factory method.
     *
     * @return an instance of Entry that holds the passed cache value
     */
    protected SafeHashMap.Entry instantiateEntry()
        {
        return new Entry();
        }

    /**
     * A holder for a cached Binary key-value pair.
     * <p/>
     * For technical reasons, this class extends directly from OldCache.Entry
     * rather than from LocalCache.Entry. When OldCache is merged directly
     * into LocalCache, this will need to be updated.
     */
    public class Entry
        extends OldCache.Entry
        {
        /**
         * Calculate a cache cost for the specified object.
         * <p/>
         * The default implementation weighs all entries equally as 1, and thus
         * weighs the cache only by the number of entries in the cache.
         * <p/>
         * This implementation returns the length of the cached value. The only
         * type of values supported is Binary.
         *
         * @param oValue the cache value to evaluate for unit cost
         *
         * @return an integer value 0 or greater, with a larger value signifying
         *         a higher cost
         *
         * @throws ClassCastException if the value is not a Binary
         */
        protected int calculateUnits(Object oValue)
            {
            return ((Binary) oValue).length();
            }
        }
    }
