
package com.tangosol.examples.coherence;


import com.tangosol.net.CacheFactory;
import com.tangosol.net.Invocable;
import com.tangosol.net.InvocationObserver;
import com.tangosol.net.InvocationService;
import com.tangosol.net.Member;

import com.tangosol.util.Base;

import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
* An Invocation service example (the client part).
*
* Command:
*   java -cp %classpath%;coherence.jar com.tangosol.examples.coherence.InvocationClient
*/
public class InvocationClient
        extends    Base
        implements InvocationObserver
    {
    /**
    * Entry point.
    */
    public static void main(String[] asArgs)
            throws InterruptedException
        {
        InvocationService service = CacheFactory.getInvocationService("Example");
        
        Member memberThis = service.getCluster().getLocalMember();
        Set    setMembers = service.getInfo().getServiceMembers();

        out("*** Started invocation service on " + memberThis);

        Task task = new Task("Request at: " + new Date());

        // example of synchronous execution
        out("\n*** Synchronous Query to everyone but itself:");
        
        setMembers.remove(memberThis);

        Map mapResult = service.query(task, setMembers);

        for (Iterator iter = mapResult.keySet().iterator(); iter.hasNext();)
            {
            Member member = (Member) iter.next();
            out("Query response from member " + member.getId() + ": " + mapResult.get(member));
            }

        // example of asynchronous execution
        out("\n*** Asynchronous execution by everyone:");
        
        InvocationObserver observer = new InvocationClient();
        synchronized (observer)
            {
            service.execute(task, null, observer);
            observer.wait(0);
            }
        }

    /**
    * This method is called by the {@link InvocationService} to inform the
    * InvocationObserver that a member has finished running the
    * {@link Invocable} object; the result of the invocation, if any, is
    * provided to the InvocationObserver. The result will be null
    * if no result is registered by the Invocable object, or if it
    * explicitly registers a result whose value is null.
    * 
    * @param member   cluster Member that has finished the execution
    *                 of the Invocable object
    * @param oResult  the result, if any, of the invocation
    */
    public void memberCompleted(Member member, Object oResult)
        {
        CacheFactory.log("Received \"memberCompleted\" notification from Member " + 
            member.getId() + "; result=" + oResult, 0);
        }

    /**
    * This method is called by the {@link InvocationService} to inform the
    * InvocationObserver that a member has thrown an exception while
    * running the {@link Invocable} object.
    *
    * @param member    cluster Member that encountered an exception
    *                  while executing the Invocable object
    * @param eFailure  the Throwable object that was encountered
    */
    public void memberFailed(Member member, Throwable eFailure)
        {
        CacheFactory.log("Received \"memberFailed\" notification from Member " +
            member.getId() + "; exception=" + eFailure, 0);

        }

    /**
    * This method is called by the {@link InvocationService} to inform the
    * InvocationObserver that a member that the {@link Invocable} object was
    * intended for execution upon has left the service (or the cluster).
    * <p>
    * It cannot be determined whether the member never received the
    * Invocable object, received and began execution of it and left
    * before finishing, or even completed execution of it without
    * managing to report a result.
    *
    * @param member    cluster Member that left the service before
    *                  reporting the completion of the execution of
    *                  the Invocable object
    */
    public void memberLeft(Member member)
        {
        CacheFactory.log("Received \"memberLeft\" notification from Member " + 
            member.getId(), 0);
        }

    /**
    * This method is called by the {@link InvocationService} to inform the 
    * InvocationObserver that all service members have either finished 
    * the {@link Invocable} object execution or are not (or no longer) running
    * the InvocationService.
    */
    public synchronized void invocationCompleted()
        {
        CacheFactory.log("Received \"invocationCompleted\" notification", 0);
        notifyAll();
        }

    // ----- inner classes ------------------------------------------------

    public static class Task
            implements Invocable
        {
        /**
        * Construct a Task object.
        * 
        * @param sMsg  the request message
        */
        public Task(String sMsg)
            {
            m_sMsg = sMsg;
            }

        /**
        * Called by the InvocationService exactly once on this Invocable object 
        * as part of its initialization.
        *
        * @param service the containing InvocationService
        */
        public void init(InvocationService service)
            {
            m_service = service;
            }

        /**
        * Called exactly once by the InvocationService to invoke this Invocable 
        * object.
        */
        public void run()
            {
            CacheFactory.log(m_sMsg);
            try
                {
                int iDelay = (int) (Math.random() * 5); // 0 to 5 sec

                Thread.sleep(iDelay*1000);
                }
            catch (InterruptedException e) {}

            // return the local time
            m_oResult = new Date();
            }

        /**
        * Determine the result from the invocation of this object.
        *
        * @return the invocation result, if any
        */
        public Object getResult()
            {
            return m_oResult;
            }

        // ----- fileds -----------------------------------------------------

        /**
        * The request message.
        */
        private String m_sMsg;

        /**
        * The enclosing InvocationService
        */
        private transient InvocationService m_service;

        /**
        * The result of invocation.
        */
        private transient Object m_oResult;
        }
    }
