/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002-2010 Oracle.  All rights reserved.
 *
 */

package com.sleepycat.je;

import java.io.File;

import junit.framework.TestCase;

import com.sleepycat.bind.tuple.IntegerBinding;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.MemoryBudget;
import com.sleepycat.je.util.TestUtils;

public class EnvironmentStatTest extends TestCase {

    private final File envHome;
    private static final String DB_NAME = "foo";

    public EnvironmentStatTest() {
        envHome = new File(System.getProperty(TestUtils.DEST_DIR));
    }

    @Override
    public void setUp()
        throws Exception {

        TestUtils.removeLogFiles("Setup", envHome, false);
    }

    @Override
    public void tearDown()
        throws Exception {

        TestUtils.removeLogFiles("TearDown", envHome, false);
    }

    /**
     * Basic cache management stats.
     */
    public void testCacheStats()
        throws Exception {

        /* Init the Environment. */
        EnvironmentConfig envConfig = TestUtils.initEnvConfig();
        envConfig.setTransactional(true);
        envConfig.setConfigParam(EnvironmentParams.NODE_MAX.getName(), "6");
        envConfig.setAllowCreate(true);
        Environment env = new Environment(envHome, envConfig);

        EnvironmentStats stat = env.getStats(TestUtils.FAST_STATS);
        env.close();
        assertEquals(0, stat.getNCacheMiss());
        assertEquals(0, stat.getNNotResident());

        // Try to open and close again, now that the environment exists
        envConfig.setAllowCreate(false);
        env = new Environment(envHome, envConfig);

        /* Open a database and insert some data. */
        DatabaseConfig dbConfig = new DatabaseConfig();
        dbConfig.setTransactional(true);
        dbConfig.setAllowCreate(true);
        Database db = env.openDatabase(null, DB_NAME, dbConfig);
        db.put(null, new DatabaseEntry(new byte[0]),
                     new DatabaseEntry(new byte[0]));
        Transaction txn = env.beginTransaction(null, null);
        db.put(txn, new DatabaseEntry(new byte[0]),
                    new DatabaseEntry(new byte[0]));

        /* Do the check. */
        stat = env.getStats(TestUtils.FAST_STATS);
        MemoryBudget mb =
            DbInternal.getEnvironmentImpl(env).getMemoryBudget();

        assertEquals(mb.getCacheMemoryUsage(), stat.getCacheTotalBytes());

        /* 
         * The size of each log buffer is calculated by:
         * mb.logBufferBudget/numBuffers, which is rounded down to the nearest
         * integer. The stats count the precise capacity of the log
         * buffers. Because of rounding down, the memory budget may be slightly
         * > than the stats buffer bytes, but the difference shouldn't be 
         * greater than the numBuffers.
         */
        assertTrue((mb.getLogBufferBudget() - stat.getBufferBytes() <=
                    stat.getNLogBuffers()));
        assertEquals(mb.getTreeMemoryUsage() + mb.getTreeAdminMemoryUsage(),
                     stat.getDataBytes());
        assertEquals(mb.getLockMemoryUsage(), stat.getLockBytes());
        assertEquals(mb.getAdminMemoryUsage(), stat.getAdminBytes());

        assertTrue(stat.getBufferBytes() > 0);
        assertTrue(stat.getDataBytes() > 0);
        assertTrue(stat.getLockBytes() > 0);
        assertTrue(stat.getAdminBytes() > 0);

        /* Account for rounding down when calculating log buffer size. */
        assertTrue(stat.getCacheTotalBytes() -
                   (stat.getBufferBytes() +
                     stat.getDataBytes() +
                     stat.getLockBytes() +
                    stat.getAdminBytes()) <= stat.getNLogBuffers());

        assertEquals(11, stat.getNCacheMiss());
        assertEquals(11, stat.getNNotResident());

        /* Test deprecated getCacheDataBytes method. */
        final EnvironmentStats finalStat = stat;
        final long expectCacheDataBytes = mb.getCacheMemoryUsage() -
                                          mb.getLogBufferBudget();
        (new Runnable() {
            @Deprecated
            public void run() {
                assertTrue((expectCacheDataBytes -
                           finalStat.getCacheDataBytes()) <= 
                               finalStat.getNLogBuffers());
            }
        }).run();

        txn.abort();
        db.close();
        env.close();
    }

    /**
     * Check stats to see if we correctly record nLogFsyncs (any fsync of the
     * log) and nFSyncs(commit fsyncs)
     */
    public void testFSyncStats()
        throws Exception {

        /* The usual env and db setup */
        EnvironmentConfig envConfig = TestUtils.initEnvConfig();
        envConfig.setTransactional(true);
        envConfig.setAllowCreate(true);
        Environment env = new Environment(envHome, envConfig);

        DatabaseConfig dbConfig = new DatabaseConfig();
        dbConfig.setTransactional(true);
        dbConfig.setAllowCreate(true);

        Database db = env.openDatabase(null, "foo", dbConfig);
        DatabaseEntry value = new DatabaseEntry();
        Transaction txn = env.beginTransaction(null, null);
        IntegerBinding.intToEntry(10, value);
        db.put(txn, value, value);

        StatsConfig statConfig = new StatsConfig();
        statConfig.setClear(true);
        /* Get a snapshot of the stats, for use as the starting point. */
        EnvironmentStats start = env.getStats(statConfig);

        /* 
         * The call to env.sync() provokes a ckpt, which does a group mgr type
         * commit, so both getNFsyncs and nLogFSyncs are incremented. 
         */
        env.sync();
        EnvironmentStats postSync = env.getStats(statConfig);
        assertEquals(1, postSync.getNFSyncs());
        assertEquals(1, postSync.getNLogFSyncs());

        /* Should be a transaction related fsync */
        txn.commitSync();
        EnvironmentStats postCommit = env.getStats(statConfig);
        assertEquals(1, postCommit.getNFSyncs());
        assertEquals(1, postCommit.getNLogFSyncs());

        /* Should be a transaction related fsync */
        DbInternal.getEnvironmentImpl(env).forceLogFileFlip();
        EnvironmentStats postFlip = env.getStats(statConfig);
        assertEquals(0, postFlip.getNFSyncs());
        assertEquals(1, postCommit.getNLogFSyncs());

        db.close();
        env.close();
    }
}
