/*
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2005
*      Sleepycat Software.  All rights reserved.
*
* $Id: MemoryBudgetTest.java,v 1.9 2004/12/22 14:11:43 linda Exp $
*/

package com.sleepycat.je.dbi;

import java.io.File;
import java.io.IOException;

import junit.framework.TestCase;

import com.sleepycat.je.DbInternal;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.util.TestUtils;

/**
 */
public class MemoryBudgetTest extends TestCase {
    private File envHome;
    
    public MemoryBudgetTest() {
        envHome = new File(System.getProperty(TestUtils.DEST_DIR));
    }

    public void setUp()
        throws IOException {

        TestUtils.removeLogFiles("Setup", envHome, false);
    }
    
    public void tearDown()
        throws Exception {

        TestUtils.removeLogFiles("TearDown", envHome, false);
    }
    
    public void testDefaults() 
        throws Exception {

        EnvironmentConfig envConfig = new EnvironmentConfig();
        envConfig.setAllowCreate(true);
        Environment env = new Environment(envHome, envConfig);
        EnvironmentImpl envImpl = DbInternal.envGetEnvironmentImpl(env);
        MemoryBudget testBudget = envImpl.getMemoryBudget();

	/*
        System.out.println("max=    " + testBudget.getMaxMemory());
        System.out.println("tree=   " + testBudget.getTreeBudget());
        System.out.println("log=    " + testBudget.getLogBufferBudget());
        System.out.println("thresh= " + testBudget.getEvictorCheckThreshold());
	*/

        assertTrue(testBudget.getMaxMemory() > 0);
        assertTrue(testBudget.getTreeBudget() > 0);
        assertTrue(testBudget.getLogBufferBudget() > 0);

        assertTrue(testBudget.getMaxMemory() <=
                   MemoryBudget.getRuntimeMaxMemory());
        assertTrue((testBudget.getLogBufferBudget() + 
                    testBudget.getTreeBudget()) <=
                    testBudget.getMaxMemory());
                   
        /* 
         * The tree and log buffer budget together is equal to
         * the max memory budget.
         */
        assertEquals((testBudget.getTreeBudget() +
                      testBudget.getLogBufferBudget()),
                      testBudget.getMaxMemory());

        env.close();
    }

    /* Verify that the proportionally based setting works. */
    public void testCacheSizing()
        throws Exception {

        long jvmMemory = MemoryBudget.getRuntimeMaxMemory();

	/*
	 * Runtime.maxMemory() may return Long.MAX_VALUE if there is no
	 * inherent limit.
	 */
	if (jvmMemory == Long.MAX_VALUE) {
	    jvmMemory = 1 << 26;
	}

        /* The default cache size ought to be percentage based. */
        EnvironmentConfig envConfig = new EnvironmentConfig();
        envConfig.setAllowCreate(true);
        Environment env = new Environment(envHome, envConfig);
        EnvironmentImpl envImpl = DbInternal.envGetEnvironmentImpl(env);
        long percentConfig = envImpl.getConfigManager().
            getInt(EnvironmentParams.MAX_MEMORY_PERCENT);

        EnvironmentConfig c = env.getConfig();
        long expectedMem = (jvmMemory * percentConfig)/100;
        assertEquals(expectedMem, c.getCacheSize());
        assertEquals(expectedMem, envImpl.getMemoryBudget().getMaxMemory());
        env.close();

        /* Try setting the percentage.*/
        expectedMem = (jvmMemory * 30)/100;
        envConfig = new EnvironmentConfig();
        envConfig.setCachePercent(30);
        env = new Environment(envHome, envConfig);
        envImpl = DbInternal.envGetEnvironmentImpl(env);
        c = env.getConfig();
        assertEquals(expectedMem, c.getCacheSize());
        assertEquals(expectedMem, envImpl.getMemoryBudget().getMaxMemory());
        env.close();

        /* Try overriding */
        envConfig = new EnvironmentConfig();
        envConfig.setCacheSize(1234);
        env = new Environment(envHome, envConfig);
        envImpl = DbInternal.envGetEnvironmentImpl(env);
        c = env.getConfig();
        assertEquals(1234, c.getCacheSize());
        assertEquals(1234, envImpl.getMemoryBudget().getMaxMemory());
        env.close();
    }
}

