/*
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2002-2005
 *      Sleepycat Software.  All rights reserved.
 *
 * $Id: DeltaInfo.java,v 1.14 2004/12/22 14:11:35 linda Exp $
 */
package com.sleepycat.je.tree;

import java.nio.ByteBuffer;

import com.sleepycat.je.log.LogException;
import com.sleepycat.je.log.LogReadable;
import com.sleepycat.je.log.LogUtils;
import com.sleepycat.je.log.LogWritable;
import com.sleepycat.je.utilint.DbLsn;

/**
 * DeltaInfo holds the delta for one BIN entry in a partial BIN log entry.
 * The data here is all that we need to update a BIN to its proper state.
 */

public class DeltaInfo implements LogWritable, LogReadable {
    private Key key;
    private long lsn;
    private boolean knownDeleted;
		  
    DeltaInfo(Key key, long lsn, boolean knownDeleted) {
        this.key = key;
        this.lsn = lsn;
        this.knownDeleted = knownDeleted;
    }

    /**
     * For reading from the log only.
     */
    DeltaInfo() {
        key = new Key();
        lsn = DbLsn.NULL_LSN;
    }

    /* 
     * @see com.sleepycat.je.log.LogWritable#getLogSize()
     */
    public int getLogSize() {
        return key.getLogSize() +
	    LogUtils.getLongLogSize() + // LSN
            LogUtils.getBooleanLogSize();
    }

    /* 
     * @see LogWritable#writeToLog(java.nio.ByteBuffer)
     */
    public void writeToLog(ByteBuffer logBuffer) {
        key.writeToLog(logBuffer);
	LogUtils.writeLong(logBuffer, lsn);
        LogUtils.writeBoolean(logBuffer, knownDeleted);
    }

    /* 
     * @see com.sleepycat.je.log.LogReadable#readFromLog
     *          (java.nio.ByteBuffer)
     */
    public void readFromLog(ByteBuffer itemBuffer) throws LogException {
        key.readFromLog(itemBuffer);
	lsn = LogUtils.readLong(itemBuffer);
        knownDeleted = LogUtils.readBoolean(itemBuffer);
    }

    /* 
     * @see LogReadable#dumpLog(java.lang.StringBuffer)
     */
    public void dumpLog(StringBuffer sb, boolean verbose) {
        key.dumpLog(sb, verbose);
	sb.append(DbLsn.toString(lsn));
        sb.append("<knownDeleted val=\"").append(knownDeleted).append("\"/>");
    }

    /**
     * @see LogReadable#logEntryIsTransactional
     */
    public boolean logEntryIsTransactional() {
	return false;
    }

    /**
     * @see LogReadable#getTransactionId
     */
    public long getTransactionId() {
	return 0;
    }

    /**
     * @return the Key.
     */
    Key getKey() {
        return key;
    }

    /**
     * @return true if this is known to be deleted.
     */
    boolean isKnownDeleted() {
        return knownDeleted;
    }

    /**
     * @return the LSN.
     */
    long getLsn() {
        return lsn;
    }

}
