/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: RunAction.java,v 1.12 2004/08/13 17:14:48 linda Exp $
*/

package com.sleepycat.je.util;

import java.io.File;
import java.text.DecimalFormat;

import com.sleepycat.je.CheckpointConfig;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.utilint.CmdUtil;

/**
 * RunAction is a debugging aid that runs one of the background activities
 * (cleaning, compressing, evicting, checkpointing). 
 */
public class RunAction {

    private static final int CLEAN = 1;
    private static final int COMPRESS = 2;
    private static final int EVICT = 3;
    private static final int CHECKPOINT = 4;

    public static void main(String [] argv) {

        long recoveryStart = 0;
        long actionStart = 0;
        long actionEnd = 0;

        try {
            int whichArg = 0;
            String dbName = null;
            int doAction = 0;

            /*
             * Usage: -h  <envHomeDir> (optional)
             *        -a <clean|compress|evict|checkpoint>
             *        -ro (read only)
             */
            String envHome = "."; // default to current directory
            boolean readOnly = false;
            while (whichArg < argv.length) {
                String nextArg = argv[whichArg];

                if (nextArg.equals("-h")) {
                    whichArg++;
                    envHome = CmdUtil.getArg(argv, whichArg);
                } else if (nextArg.equals("-a")) {
                    whichArg++;
                    String action = CmdUtil.getArg(argv, whichArg);
                    if (action.equalsIgnoreCase("clean")) {
                        doAction = CLEAN;
                    } else if (action.equalsIgnoreCase("compress")) {
                        doAction = COMPRESS;
                    } else if (action.equalsIgnoreCase("checkpoint")) {
                        doAction = CHECKPOINT;
                    } else if (action.equalsIgnoreCase("evict")) {
                        doAction = EVICT;
                    } else {
                        usage();
                        System.exit(1);
                    }
                } else if (nextArg.equals("-ro")) {
                    readOnly = true;
                } else {
                    throw new IllegalArgumentException
                        (nextArg + " is not a supported option.");
                }
                whichArg++;
            }

            /* Make an environment */
            EnvironmentConfig envConfig = new EnvironmentConfig();
            // Do debug log to the console
            envConfig.setConfigParam
                (EnvironmentParams.JE_LOGGING_CONSOLE.getName(), "true");

            if (readOnly) {
                // Don't debug log to the database log.
                envConfig.setConfigParam
                    (EnvironmentParams.JE_LOGGING_DBLOG.getName(), "false");

                envConfig.setReadOnly(true);
            }
            recoveryStart = System.currentTimeMillis();

            Environment envHandle =
		new Environment(new File(envHome), envConfig);
            
            actionStart = System.currentTimeMillis();
            switch(doAction) {
            case CLEAN:
                envHandle.cleanLog();
                break;
            case COMPRESS:
                envHandle.compress();
                break;
            case EVICT:
                envHandle.evictMemory();
                break;
            case CHECKPOINT:
                CheckpointConfig forceConfig = new CheckpointConfig();
                forceConfig.setForce(true);
                envHandle.checkpoint(forceConfig);
                break;
            }
            actionEnd = System.currentTimeMillis();

            envHandle.close();


        } catch (Exception e) {
            e.printStackTrace();
            System.out.println(e.getMessage());
            usage();
            System.exit(1);
        } finally {
            DecimalFormat f = new DecimalFormat();
            f.setMaximumFractionDigits(2);

            long recoveryDuration = actionStart - recoveryStart;
            System.out.println("\nrecovery time = " +
                               f.format(recoveryDuration) +
                               " millis " +
                               f.format((double)recoveryDuration/60000) +
                               " minutes");

            long actionDuration = actionEnd - actionStart;
            System.out.println("action time = " +
                               f.format(actionDuration) +
                               " millis " +
                               f.format(actionDuration/60000) +
                               " minutes");

                               }
        }

    private static void usage() {
        System.out.println("Usage: RunAction");
	System.out.println("  -h <environment home> ");
        System.out.println("  -a <clean|compress|evict|checkpoint>");
        System.out.println("  -ro (read-only - defaults to read-write)");
    }
}
