/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: DbLsnTest.java,v 1.10 2004/08/17 16:48:40 linda Exp $
*/

package com.sleepycat.je.util;

import java.io.File;

import junit.framework.TestCase;

import com.sleepycat.je.DbInternal;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.utilint.DbLsn;

public class DbLsnTest extends TestCase {
    long[] values = { 0xFF, 0xFFFF, 0xFFFFFF, 0x7FFFFFFF, 0xFFFFFFFFL };

    public void testDbLsn() {
	for (int i = 0; i < values.length; i++) {
	    long value = values[i];
	    DbLsn lsn = new DbLsn(value, value);
	    assertTrue((lsn.getFileNumber() == value) &&
		       (lsn.getFileOffset() == value));
	}
    }

    public void testComparableEquality() {
	/* Test equality */

	for (int i = 0; i < values.length; i++) {
	    long value = values[i];
	    DbLsn lsn1 = new DbLsn(value, value);
	    DbLsn lsn2 = new DbLsn(value, value);
	    assertTrue(lsn1.compareTo(lsn2) == 0);
	}
    }

    public void testComparableNullPointerException() {
	/* Check that compareTo throws NullPointerException */

	try {
	    DbLsn lsn1 = new DbLsn(0, 0);
	    lsn1.compareTo(null);
	    fail("compareTo(null) didn't throw NullPointerException");
	} catch (NullPointerException NPE) {
	}
    }

    public void testComparableClassCastException() {
	/* Check that compareTo throws ClassCastException */

	try {
	    DbLsn lsn1 = new DbLsn(0, 0);
	    lsn1.compareTo(new Object());
	    fail("compareTo(non-DbLsn) didn't throw ClassCastException");
	} catch (ClassCastException CCE) {
	}
    }

    public void testComparableInequalityFileNumber() {
	/* Check for inequality in the file number */

	for (int i = 0; i < values.length; i++) {
	    long value = values[i];
	    DbLsn lsn1 = new DbLsn(value, value);
	    DbLsn lsn2 = new DbLsn(0, value);
	    assertTrue(lsn1.compareTo(lsn2) == 1);
	    assertTrue(lsn2.compareTo(lsn1) == -1);
	}
    }

    public void testComparableInequalityFileOffset() {
	/* Check for inequality in the file offset */

	for (int i = 0; i < values.length - 1; i++) {
	    long value = values[i];
	    DbLsn lsn1 = new DbLsn(value, value);
	    DbLsn lsn2 = new DbLsn(value, 0);
	    /* Can't compareTo(NULL_LSN). */
	    if (!lsn1.equals(DbLsn.NULL_LSN) &&
		!lsn2.equals(DbLsn.NULL_LSN)) {
		assertTrue(lsn1.compareTo(lsn2) == 1);
		assertTrue(lsn2.compareTo(lsn1) == -1);
	    }
	}
    }

    public void testSubtractNoCleaning() {
        DbLsn a = new DbLsn(1, 10);
        DbLsn b = new DbLsn(3, 40);
        assertEquals(230, b.getNoCleaningDistance(a, 100));
        assertEquals(230, a.getNoCleaningDistance(b, 100));
        
        DbLsn c = new DbLsn(1, 50);
        assertEquals(40, a.getNoCleaningDistance(c, 100));
        assertEquals(40, c.getNoCleaningDistance(a, 100));
    }

    public void testSubtractWithCleaning()
        throws Exception {
        /* Try with non-consecutive files (due to cleaning). */

        File envHome = new File(System.getProperty("testdestdir"));
        TestUtils.removeLogFiles("TestSubtract", envHome, false);
        EnvironmentConfig envConfig = new EnvironmentConfig();
        envConfig.setAllowCreate(true);
        Environment env = new Environment(envHome, envConfig);

        try {
            File file1 = new File (envHome, "00000001.jdb");
            File file2 = new File (envHome, "00000003.jdb");
            file1.createNewFile();
            file2.createNewFile();
            DbLsn a = new DbLsn(1, 10);
            DbLsn b = new DbLsn(3, 40);
            FileManager fileManager =
                DbInternal.envGetEnvironmentImpl(env).getFileManager();
            assertEquals(130, b.getWithCleaningDistance(fileManager, a, 100));
            assertEquals(130, a.getWithCleaningDistance(fileManager, b, 100));

            DbLsn c = new DbLsn(1, 50);
            assertEquals(40, a.getWithCleaningDistance(fileManager, c, 100));
            assertEquals(40, c.getWithCleaningDistance(fileManager, a, 100));
        } finally {
            env.close();
        }
    }
}
