/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: TxnTest.java,v 1.42 2004/06/01 20:43:10 cwl Exp $
*/

package com.sleepycat.je.txn;

import java.io.File;
import java.io.IOException;

import junit.framework.TestCase;

import com.sleepycat.je.Cursor;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.je.DatabaseEntry;
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.DeadlockException;
import com.sleepycat.je.DbInternal;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.EnvironmentMutableConfig;
import com.sleepycat.je.LockMode;
import com.sleepycat.je.LockNotGrantedException;
import com.sleepycat.je.LockStats;
import com.sleepycat.je.OperationStatus;
import com.sleepycat.je.Transaction;
import com.sleepycat.je.TransactionConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.DatabaseImpl;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.tree.ChildReference;
import com.sleepycat.je.tree.IN;
import com.sleepycat.je.tree.LN;
import com.sleepycat.je.tree.WithRootLatched;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.util.TestUtils;

/*
 * Simple transaction testing
 */
public class TxnTest extends TestCase {
    private File envHome;
    private Environment env;
    private Database db;

    public TxnTest()
        throws DatabaseException {

        envHome = new File(System.getProperty("testdestdir"));
    }

    public void setUp() 
        throws IOException, DatabaseException {

        TestUtils.removeFiles("Setup", envHome, FileManager.JE_SUFFIX);

        EnvironmentConfig envConfig = new EnvironmentConfig();
        envConfig.setConfigParam(EnvironmentParams.NODE_MAX.getName(), "6");
        envConfig.setTransactional(true);
        envConfig.setAllowCreate(true);
        env = new Environment(envHome, envConfig);

        DatabaseConfig dbConfig = new DatabaseConfig();
        dbConfig.setTransactional(true);
        dbConfig.setAllowCreate(true);
        db = env.openDatabase(null, "foo", dbConfig);
    }

    public void tearDown()
        throws IOException, DatabaseException {

        db.close();
        env.close();
        TestUtils.removeFiles("TearDown", envHome, FileManager.JE_SUFFIX);
    }

    /**
     * Test transaction locking and releasing
     */
    public void testBasicLocking()
        throws Throwable {

        try {

            LN ln = new LN(new byte[0]);

            /*
             * Make a null txn that will lock. Take a lock and then end the 
             * operation.
             */
            EnvironmentImpl environment =
		DbInternal.envGetEnvironmentImpl(env);
            Locker nullTxn = new BasicLocker(environment);
            LockGrantType lockGrant = nullTxn.readLock(ln);
            assertEquals(LockGrantType.NEW, lockGrant);
            checkHeldLocks(nullTxn, 1, 0);

            nullTxn.operationEnd();
            checkHeldLocks(nullTxn, 0, 0);

            // Take a lock, release it.
            lockGrant = nullTxn.readLock(ln);
            assertEquals(LockGrantType.NEW, lockGrant);
            checkHeldLocks(nullTxn, 1, 0);

            nullTxn.releaseLock(ln);
            checkHeldLocks(nullTxn, 0, 0);

            /*
             * Make a user transaction, check lock and release.
             */
            Txn userTxn = new Txn(environment, new TransactionConfig());
            lockGrant = userTxn.readLock(ln);
            assertEquals(LockGrantType.NEW, lockGrant);
            checkHeldLocks(userTxn, 1, 0);

            // Try demoting, nothing should happen.
            try {
                userTxn.demoteLock(ln.getNodeId());
                fail("exception not thrown on phoney demoteLock");
            } catch (DatabaseException DBE) {
            }
            checkHeldLocks(userTxn, 1, 0);

            // Make it a write lock, then demote.
            lockGrant =
		userTxn.writeLock(ln, DbInternal.dbGetDatabaseImpl(db)).
		getLockGrant();
            assertEquals(LockGrantType.PROMOTION, lockGrant);
            checkHeldLocks(userTxn, 0, 1);
            userTxn.demoteLock(ln.getNodeId());
            checkHeldLocks(userTxn, 1, 0);

            // Shouldn't release at operation end
            userTxn.operationEnd();
            checkHeldLocks(userTxn, 1, 0);

            userTxn.releaseLock(ln);
            checkHeldLocks(userTxn, 0, 0);
            userTxn.commit(true);
        } catch (Throwable t) {
            /* print stack trace before going to teardown. */
            t.printStackTrace();
            throw t;
        }
    }

    private void checkHeldLocks(Locker txn, int numReadLocks, int numWriteLocks)
        throws DatabaseException {

        LockStats stat = txn.collectStats(new LockStats());
        assertEquals(numReadLocks, stat.getNReadLocks());
        assertEquals(numWriteLocks, stat.getNWriteLocks());
    }

    /**
     * Test transaction commit, from the locking point of view.
     */
    public void testCommit()
        throws Throwable {

        try {
            LN ln1 = new LN(new byte[0]);
            LN ln2 = new LN(new byte[0]);
                                          
            EnvironmentImpl environment =
		DbInternal.envGetEnvironmentImpl(env);
            Txn userTxn = new Txn(environment, new TransactionConfig());

            // get read lock 1
            LockGrantType lockGrant = userTxn.readLock(ln1);
            assertEquals(LockGrantType.NEW, lockGrant);
            checkHeldLocks(userTxn, 1, 0);

            // get read lock 2
            lockGrant = userTxn.readLock(ln2);
            assertEquals(LockGrantType.NEW, lockGrant);
            checkHeldLocks(userTxn, 2, 0);

            // upgrade read lock 2 to a write
            lockGrant =
		userTxn.writeLock(ln2, DbInternal.dbGetDatabaseImpl(db)).
		getLockGrant();
            assertEquals(LockGrantType.PROMOTION, lockGrant);
            checkHeldLocks(userTxn, 1, 1);

            // read lock 1 again, shouldn't increase count
            lockGrant = userTxn.readLock(ln1);
            assertEquals(LockGrantType.EXISTING, lockGrant);
            checkHeldLocks(userTxn, 1, 1);

            // Shouldn't release at operation end
            DbLsn commitLsn = userTxn.commit(true);
            checkHeldLocks(userTxn, 0, 0);

            TxnCommit commitRecord =
                (TxnCommit) environment.getLogManager().get(commitLsn);

            assertEquals(userTxn.getId(), commitRecord.getId());
            assertEquals(userTxn.getLastLsn(), commitRecord.getLastLsn());
        } catch (Throwable t) {
            /* print stack trace before going to teardown. */
            t.printStackTrace();
            throw t;
        }
    }

    /**
     * Make sure an abort never tries to split the tree.
     */
    public void testAbortNoSplit() 
        throws Throwable {

        try {
            Transaction txn = env.beginTransaction(null, null);

            DatabaseEntry keyDbt = new DatabaseEntry();
            DatabaseEntry dataDbt = new DatabaseEntry();
            dataDbt.setData(new byte[1]);
        
            /* Insert enough data so that the tree is ripe for a split. */
            int numForSplit = 25;
            for (int i = 0; i < numForSplit; i++) {
                keyDbt.setData(TestUtils.getTestArray(i));
                db.put(txn, keyDbt, dataDbt);
            }

            /* Check that we're ready for a split. */
            DatabaseImpl database = DbInternal.dbGetDatabaseImpl(db);
            CheckReadyToSplit splitChecker = new CheckReadyToSplit(database);
            database.getTree().withRootLatched(splitChecker);
            assertTrue(splitChecker.getReadyToSplit());

            /* 
             * Make another txn that will get a read lock on the map
             * lsn. Then abort the first txn. It shouldn't try to do a
             * split, if it does, we'll run into the
             * no-latches-while-locking check.
             */
            Transaction txnSpoiler = env.beginTransaction(null, null);
            DatabaseConfig dbConfig = new DatabaseConfig();
            dbConfig.setTransactional(true);
            Database dbSpoiler = env.openDatabase(txnSpoiler, "foo", dbConfig);

            txn.abort();

            /*
             * The database should be empty
             */
            Cursor cursor = dbSpoiler.openCursor(txnSpoiler, null);
            
            assertTrue(cursor.getFirst(keyDbt, dataDbt, LockMode.DEFAULT) != 
                       OperationStatus.SUCCESS);
            cursor.close();
            txnSpoiler.abort();
        } catch (Throwable t) {
            /* print stack trace before going to teardown. */
            t.printStackTrace();
            throw t;
        } 
    }

    public void testTransactionName() 
        throws Throwable {

        try {
            Transaction txn = env.beginTransaction(null, null);
	    txn.setName("blort");
	    assertEquals("blort", txn.getName());
            txn.abort();
        } catch (Throwable t) {
            /* print stack trace before going to teardown. */
            t.printStackTrace();
            throw t;
        } 
    }

    public void testSyncConfig() 
        throws Throwable {

        /*
         * Legend:
         *   envNoSync=1/0  : EnvironmentMutableConfig.NoSync is true/false
         *   txnNoSync=1/0  : TransactionConfig.NoSync is true/false
         *   txnSync=1/0    : TransactionConfig.Sync is true/false
         */
        try {
            TransactionConfig defaultConfig = new TransactionConfig();
            TransactionConfig syncConfig = new TransactionConfig();
            syncConfig.setSync(true);
            TransactionConfig noSyncConfig = new TransactionConfig();
            noSyncConfig.setNoSync(true);
            TransactionConfig syncNoSyncConfig = new TransactionConfig();
            syncNoSyncConfig.setSync(true);
            syncNoSyncConfig.setNoSync(true);
            Transaction txn;

            /* envNoSync=0 with default env config */
            assertTrue(!env.getMutableConfig().getTxnNoSync());

            /* envNoSync=0 auto-commit */
            assertTrue(syncAutoCommit());

            /* envNoSync=0 txnNoSync=0 txnSync=0 */
            assertTrue(syncExplicit(null));
            assertTrue(syncExplicit(defaultConfig));

            /* envNoSync=0 txnNoSync=0 txnSync=1 */
            assertTrue(syncExplicit(syncConfig));

            /* envNoSync=0 txnNoSync=1 txnSync=0 */
            assertTrue(!syncExplicit(noSyncConfig));

            /* envNoSync=0 txnNoSync=1 txnSync=1 */
            assertTrue(syncExplicit(syncNoSyncConfig));

            /* envNoSync=1 */
            EnvironmentMutableConfig config = env.getMutableConfig();
            config.setTxnNoSync(true);
            env.setMutableConfig(config);
            assertTrue(env.getMutableConfig().getTxnNoSync());

            /* envNoSync=1 auto-commit */
            assertTrue(!syncAutoCommit());

            /* envNoSync=1 txnNoSync=0 txnSync=0 */
            assertTrue(!syncExplicit(null));
            assertTrue(!syncExplicit(defaultConfig));

            /* envNoSync=1 txnNoSync=0 txnSync=1 */
            assertTrue(syncExplicit(syncConfig));

            /* envNoSync=1 txnNoSync=1 txnSync=0 */
            assertTrue(!syncExplicit(noSyncConfig));

            /* envNoSync=1 txnNoSync=1 txnSync=1 */
            assertTrue(syncExplicit(syncNoSyncConfig));

        } catch (Throwable t) {
            /* print stack trace before going to teardown. */
            t.printStackTrace();
            throw t;
        } 
    }

    /**
     * Does an explicit commit and returns whether an fsync occured.
     */
    private boolean syncExplicit(TransactionConfig config)
        throws DatabaseException {

        DatabaseEntry key = new DatabaseEntry(new byte[1]);
        DatabaseEntry data = new DatabaseEntry(new byte[1]);

        long before = getNSyncs();
        Transaction txn = env.beginTransaction(null, config);
        db.put(txn, key, data);
        txn.commit();
        long after = getNSyncs();
        boolean syncOccured = after > before;

        /* Make sure explicit sync/noSync always works. */

        before = getNSyncs();
        txn = env.beginTransaction(null, config);
        db.put(txn, key, data);
        txn.commitSync();
        after = getNSyncs();
        assert(after > before);

        before = getNSyncs();
        txn = env.beginTransaction(null, config);
        db.put(txn, key, data);
        txn.commitNoSync();
        after = getNSyncs();
        assert(after == before);

        return syncOccured;
    }

    /**
     * Does an auto-commit and returns whether an fsync occured.
     */
    private boolean syncAutoCommit()
        throws DatabaseException {

        DatabaseEntry key = new DatabaseEntry(new byte[1]);
        DatabaseEntry data = new DatabaseEntry(new byte[1]);
        long before = getNSyncs();
        db.put(null, key, data);
        long after = getNSyncs();
        return after > before;
    }

    /**
     * Returns number of fsyncs statistic.
     */
    private long getNSyncs() {
        return DbInternal.envGetEnvironmentImpl(env)
                         .getFileManager()
                         .getNFSyncs();
    }

    public void testNoWaitConfig() 
        throws Throwable {

        try {
            TransactionConfig defaultConfig = new TransactionConfig();
            TransactionConfig noWaitConfig = new TransactionConfig();
            noWaitConfig.setNoWait(true);
            Transaction txn;

            /* noWait=false */

            assertTrue(!isNoWaitTxn(null));

            txn = env.beginTransaction(null, null);
            assertTrue(!isNoWaitTxn(txn));
            txn.abort();

            txn = env.beginTransaction(null, defaultConfig);
            assertTrue(!isNoWaitTxn(txn));
            txn.abort();

            /* noWait=true */

            txn = env.beginTransaction(null, noWaitConfig);
            assertTrue(isNoWaitTxn(txn));
            txn.abort();

        } catch (Throwable t) {
            /* print stack trace before going to teardown. */
            t.printStackTrace();
            throw t;
        } 
    }

    /**
     * Returns whether the given txn is a no-wait txn, or if the txn parameter
     * is null returns whether an auto-commit txn is a no-wait txn.
     */
    private boolean isNoWaitTxn(Transaction txn) 
        throws DatabaseException {

        DatabaseEntry key = new DatabaseEntry(new byte[1]);
        DatabaseEntry data = new DatabaseEntry(new byte[1]);

        /* Use a wait txn to get a write lock. */
        Transaction txn2 = env.beginTransaction(null, null);
        db.put(txn2, key, data);

        try {
            db.put(txn, key, data);
            throw new IllegalStateException
                ("Lock should not have been granted");
        } catch (LockNotGrantedException e) {
            return true;
        } catch (DeadlockException e) {
            return false;
        } finally {
            txn2.abort();
        }
    }

    class CheckReadyToSplit implements WithRootLatched {
        private boolean readyToSplit;
        private DatabaseImpl database;

        CheckReadyToSplit(DatabaseImpl database) {
            readyToSplit = false;
            this.database = database;
        }
        
        public boolean getReadyToSplit() {
            return readyToSplit;
        }

        public IN doWork(ChildReference root) 
            throws DatabaseException {

            IN rootIN = (IN) root.fetchTarget(database, null);
            readyToSplit = rootIN.needsSplitting();
            return null;
        }
    }
}
