/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: LockTest.java,v 1.30 2004/06/04 02:26:33 linda Exp $
*/

package com.sleepycat.je.txn;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import junit.framework.TestCase;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.TransactionConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.EnvironmentImpl;

public class LockTest extends TestCase {
    private EnvironmentImpl env;
    private File envHome;

    public LockTest() {
        envHome =  new File(System.getProperty("testdestdir"));
    }

    public void setUp()
	throws DatabaseException {

        EnvironmentConfig envConfig = new EnvironmentConfig();
        envConfig.setConfigParam(EnvironmentParams.NODE_MAX.getName(), "6");
        envConfig.setAllowCreate(true);
        env = new EnvironmentImpl(envHome, envConfig);

    }

    public void tearDown()
	throws DatabaseException {

        env.close();
    }

    public void testLockConflicts() 
        throws Exception {

	Locker txn1 = new BasicLocker(env);
	Locker txn2 = new BasicLocker(env);
	Locker txn3 = new BasicLocker(env);
	try {
            /* 
             * Start fresh. Ask for a read lock from txn1 twice,
             * should only be one owner. Then add multiple
             * would-be-writers as waiters.
             */
	    Lock lock = new Lock(1);
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
	    assertEquals(LockGrantType.EXISTING,
                         lock.lock(LockType.READ, txn1, false));
	    assertEquals(1, lock.nOwners());
	    assertEquals(0, lock.nWaiters());

            /* ask for a read lock from txn 2, get it. */
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn2, false));
            /* txn1's write request must wait */
	    assertEquals(LockGrantType.WAIT_PROMOTION,
                         lock.lock(LockType.WRITE, txn1, false));
            /* txn2's write request must wait */
	    assertEquals(LockGrantType.WAIT_PROMOTION,
                         lock.lock(LockType.WRITE, txn2, false));
	    assertEquals(2, lock.nOwners());
	    assertEquals(2, lock.nWaiters());


            /* Start fresh. Get a write lock, then get a read lock. */
	    lock = new Lock(1);

	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.WRITE, txn1, false));
	    assertEquals(LockGrantType.EXISTING,
                         lock.lock(LockType.READ, txn1, false));
	    assertEquals(1, lock.nOwners());
	    assertEquals(0, lock.nWaiters());

            /* Start fresh. Get a read lock, upgrade to a write lock. */
	    lock = new Lock(1);
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
	    assertEquals(LockGrantType.PROMOTION,
                         lock.lock(LockType.WRITE, txn1, false));
	    assertEquals(1, lock.nOwners());
	    assertEquals(0, lock.nWaiters());

            /* 
             * Start fresh. Get a read lock, then ask for a non-blocking
             * write lock. The latter should be denied.
             */
	    lock = new Lock(1);

	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
	    assertEquals(LockGrantType.DENIED,
                         lock.lock(LockType.WRITE, txn2, true));
	    assertEquals(1, lock.nOwners());
	    assertEquals(0, lock.nWaiters());

            /* Two write requsts, should be one owner. */
	    lock = new Lock(1);
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.WRITE, txn1, false));
	    assertEquals(LockGrantType.EXISTING,
                         lock.lock(LockType.WRITE, txn1, false));
	    assertEquals(1, lock.nOwners());
	    assertEquals(0, lock.nWaiters());

	    /* 
             * Ensure that a read request behind a write request that waits
	     * also waits.  blocking requests.
             */
	    lock = new Lock(1);

	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
		       
	    assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.WRITE, txn2, false));
			
	    assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn3, false));
			
            assertEquals(1, lock.nOwners());
	    assertEquals(2, lock.nWaiters());

	    /* Check non blocking requests */
	    lock = new Lock(1);

	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
		       
	    /* Since non-blocking request, this fails and doesn't go
	       on the wait queue. */
	    assertEquals(LockGrantType.DENIED,
                         lock.lock(LockType.WRITE, txn2, true));
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn3, true));
	    assertEquals(2, lock.nOwners());
	    assertEquals(0, lock.nWaiters());

	    lock = new Lock(1);

	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn2, false));
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn3, false));
	    assertEquals(3, lock.nOwners());
	    assertEquals(0, lock.nWaiters());
	} catch (DatabaseException DBE) {
	    fail("caught DatabaseException");
	} finally {
            txn1.operationEnd();
            txn2.operationEnd();
            txn3.operationEnd();
        }
    }


    public void testOwners() 
        throws Exception {

	Locker txn1 = new BasicLocker(env);
	Locker txn2 = new BasicLocker(env);
	Locker txn3 = new BasicLocker(env);
	Locker txn4 = new BasicLocker(env);

        try {
            /*
             * Build up 3 owners and waiters for a lock, to test the
             * lazy initialization and optimization for single owner/waiter.
             */
            Lock lock = new Lock(1);
            /* should be no writer. */
            assertTrue(lock.getWriteOwnerLocker() == null);

            assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn2, false));
	    assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn3, false));
            
            /* should be no writer. */
            assertTrue(lock.getWriteOwnerLocker() == null);

            /* should be node 1 */
            assertEquals(1, lock.getNodeId());

            /* expect 3 owners, 0 waiters. */
            Set expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn1, LockType.READ));
            expectedOwners.add(new LockInfo(txn2, LockType.READ));
            expectedOwners.add(new LockInfo(txn3, LockType.READ));
            checkOwners(expectedOwners, lock, 0);

            /* release the first locker. */
            lock.release(txn1);
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn2, LockType.READ));
            expectedOwners.add(new LockInfo(txn3, LockType.READ));
            checkOwners(expectedOwners, lock, 0);

            /* Add more. */
            assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn4, false));
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn2, LockType.READ));
            expectedOwners.add(new LockInfo(txn3, LockType.READ));
            expectedOwners.add(new LockInfo(txn4, LockType.READ));
            checkOwners(expectedOwners, lock, 0);

            /* release */
            lock.release(txn2);
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn3, LockType.READ));
            expectedOwners.add(new LockInfo(txn4, LockType.READ));
            checkOwners(expectedOwners, lock, 0);

            /* release */
            lock.release(txn3);
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn4, LockType.READ));
            /* only 1 lock, in the owner set, but not a write owner. */
            assertTrue(lock.getWriteOwnerLocker() == null);

            /* release */
            lock.release(txn4);
            expectedOwners = new HashSet();
            checkOwners(expectedOwners, lock, 0);

            /* Add owners again. */
            assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
            assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn2, false));
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn1, LockType.READ));
            expectedOwners.add(new LockInfo(txn2, LockType.READ));
            checkOwners(expectedOwners, lock, 0);

        } catch (Exception e) {
            e.printStackTrace();
             throw e;
        } finally {
            txn1.operationEnd();
            txn2.operationEnd();
            txn3.operationEnd();
            txn4.operationEnd();
        }
    }

    public void testWaiters() 
        throws Exception {

	Locker txn1 = new AutoTxn(env, new TransactionConfig());
	Locker txn2 = new AutoTxn(env, new TransactionConfig());
	Locker txn3 = new AutoTxn(env, new TransactionConfig());
	Locker txn4 = new AutoTxn(env, new TransactionConfig());
	Locker txn5 = new AutoTxn(env, new TransactionConfig());

        try {
            /*
             * Build up 1 owners and 3waiters for a lock, to test the
             * lazy initialization and optimization for single owner/waiter.
             */
            Lock lock = new Lock(1);
            assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn1, false));
            assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.READ, txn2, false));
	    assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.WRITE, txn3, false));
	    assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.WRITE, txn4, false));
	    assertEquals(LockGrantType.WAIT_PROMOTION,
                         lock.lock(LockType.WRITE, txn1, false));
            
            /* should be no writer. */
            assertTrue(lock.getWriteOwnerLocker() == null);

            /* should be node 1 */
            assertEquals(1, lock.getNodeId());

            /* expect 2 owners, 3 waiters. */
            Set expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn1, LockType.READ));
            expectedOwners.add(new LockInfo(txn2, LockType.READ));
            checkOwners(expectedOwners, lock, 3);

            List waiters = new ArrayList();
            waiters.add(new LockInfo(txn1, LockType.WRITE));
            waiters.add(new LockInfo(txn3, LockType.WRITE));
            waiters.add(new LockInfo(txn4, LockType.WRITE));
            checkWaiters(waiters, lock);

            /* release a waiter, shouldn't change anything. */
            lock.release(txn4);
            checkWaiters(waiters, lock);

            /* 
             * Release the other read lock, expect txn1 to be promoted to a
             * write lock.
             */
            lock.release(txn2);
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn1, LockType.WRITE));
            checkOwners(expectedOwners, lock, 2);

            waiters.remove(0);
            checkWaiters(waiters, lock);

            /* release */
            lock.release(txn1);
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn3, LockType.WRITE));
            checkOwners(expectedOwners, lock, 1);

            waiters.remove(0);
            checkWaiters(waiters, lock);

            /* 
             * Add multiple read lock waiters so that we can promoting multiple
             * waiters.
             */
            assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn1, false));
            assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn2, false));
            assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn5, false));

            checkOwners(expectedOwners, lock, 4);
            waiters.add(new LockInfo(txn1, LockType.READ));
            waiters.add(new LockInfo(txn2, LockType.READ));
            waiters.add(new LockInfo(txn5, LockType.READ));
            checkWaiters(waiters, lock);

            /* flush one of the waiters. */
            lock.flushWaiter(txn5);
            waiters.remove(3);
            checkWaiters(waiters, lock);

            /* re-add. */
            assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn5, false));
            waiters.add(new LockInfo(txn5, LockType.READ));

            /* release txn3 */
            lock.release(txn3);
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn4, LockType.WRITE));
            checkOwners(expectedOwners, lock, 3);
            waiters.remove(0);
            checkWaiters(waiters, lock);

            /* release txn4, expect all read locks to promote. */
            lock.release(txn4);
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn1, LockType.READ));
            expectedOwners.add(new LockInfo(txn2, LockType.READ));
            expectedOwners.add(new LockInfo(txn5, LockType.READ));
            checkOwners(expectedOwners, lock, 0);
            waiters.clear();
            checkWaiters(waiters, lock);

        } catch (Exception e) {
            e.printStackTrace();
             throw e;
        } finally {
            txn1.operationEnd();
            txn2.operationEnd();
            txn3.operationEnd();
            txn4.operationEnd();
            txn5.operationEnd();
        }
    }

    public void testPromotion() 
        throws Exception {

	Locker txn1 = new AutoTxn(env, new TransactionConfig());
	Locker txn2 = new AutoTxn(env, new TransactionConfig());
	Locker txn3 = new AutoTxn(env, new TransactionConfig());
	Locker txn4 = new AutoTxn(env, new TransactionConfig());
	Locker txn5 = new AutoTxn(env, new TransactionConfig());

        try {
            /*
             * Build up 1 owners and 3 read waiters for a lock. Then
             * check that all the waiters promote properly.
             */
            Lock lock = new Lock(1);
            assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.WRITE, txn1, false));
	    assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn2, false));
	    assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn3, false));
	    assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn4, false));

            /* Check that 1 owner, 3 waiters exist. */
            Set expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn1, LockType.WRITE));
            checkOwners(expectedOwners, lock, 3);

            List waiters = new ArrayList();
            waiters.add(new LockInfo(txn2, LockType.READ));
            waiters.add(new LockInfo(txn3, LockType.READ));
            waiters.add(new LockInfo(txn4, LockType.READ));
            checkWaiters(waiters, lock);

            /* Release the writer, expect all 3 waiters to promote. */
            lock.release(txn1);
            expectedOwners = new HashSet();
            expectedOwners.add(new LockInfo(txn2, LockType.READ));
            expectedOwners.add(new LockInfo(txn3, LockType.READ));
            expectedOwners.add(new LockInfo(txn4, LockType.READ));
            checkOwners(expectedOwners, lock, 0);
            waiters.clear();
            checkWaiters(waiters, lock);
        } catch (Exception e) {
            e.printStackTrace();
             throw e;
        } finally {
            txn1.operationEnd();
            txn2.operationEnd();
            txn3.operationEnd();
            txn4.operationEnd();
            txn5.operationEnd();
        }
    }

    private void checkOwners(Set expectedOwners,
                             Lock lock,
                             int numExpectedWaiters) {

        /* check number of owners. */
        Set owners = lock.getOwnersClone();
        assertEquals(expectedOwners.size(), owners.size());

        /* check number of waiters. */
        assertEquals(numExpectedWaiters, lock.nWaiters());

        /* Make sure that isOwner returns the right thing. */
        Iterator iter = expectedOwners.iterator();
        while (iter.hasNext()) {
            LockInfo info = (LockInfo) iter.next();

            /* Make sure it's an owner, of the right type of lock. */
            assertEquals(info.isWriteLock(),
                         lock.isOwnedWriteLock(info.getLocker()));
            assertTrue(lock.isOwner(info.getLocker(), info.getLockType()));
        }
    }
      
    private void checkWaiters(List expectedWaiters,
                              Lock lock) {
        List waiters = lock.getWaitersListClone();
        assertEquals(expectedWaiters.size(), waiters.size());

        /* check order of the list. */
        for (int i = 0; i < expectedWaiters.size(); i++) {
            LockInfo info = (LockInfo) expectedWaiters.get(i);
            LockInfo waiterInfo = (LockInfo) waiters.get(i);
            assertEquals("i=" + i, info.getLocker(), waiterInfo.getLocker());
            assertEquals("i=" + i,
                         info.getLockType(), waiterInfo.getLockType());
            assertFalse(lock.isOwner(info.getLocker(), info.getLockType()));
            assertTrue(lock.isWaiter(info.getLocker()));
        }

                     
    }

    public void testTransfer() 
        throws Exception {

	Locker txn1 = new AutoTxn(env, new TransactionConfig());
	Locker txn2 = new AutoTxn(env, new TransactionConfig());
	Locker txn3 = new AutoTxn(env, new TransactionConfig());
	Locker txn4 = new AutoTxn(env, new TransactionConfig());
	Locker txn5 = new AutoTxn(env, new TransactionConfig());


        try {
            /* Transfer from one locker to another locker. */
            Lock lock = new Lock(1);
            assertEquals(LockGrantType.NEW,
                         lock.lock(LockType.WRITE, txn1, false));
            assertEquals(LockGrantType.WAIT_NEW,
                         lock.lock(LockType.READ, txn2, false));
            assertTrue(lock.isOwner(txn1, LockType.WRITE));
            assertFalse(lock.isOwner(txn2, LockType.READ));

            lock.transfer(txn1, txn2);
            assertFalse(lock.isOwner(txn1, LockType.WRITE));
            assertFalse(lock.isOwner(txn1, LockType.READ));
            assertTrue(lock.isOwnedWriteLock(txn2));

            /* Transfer to multiple lockers. */
            Locker [] destLockers = new Locker[3];
            destLockers[0] = txn3;
            destLockers[1] = txn4;
            destLockers[2] = txn5;
            lock.demote(txn2);
            lock.transferMultiple(txn2, destLockers);
            assertFalse(lock.isOwner(txn2, LockType.WRITE));
            assertFalse(lock.isOwner(txn2, LockType.READ));

            for (int i = 0; i < destLockers.length; i++) {
                assertTrue(lock.isOwner(destLockers[i], LockType.READ));
                assertFalse(lock.isOwner(destLockers[i], LockType.WRITE));
            }
            

        } finally {
            txn1.operationEnd();
            txn2.operationEnd();
            txn3.operationEnd();
            txn4.operationEnd();
            txn5.operationEnd();
        }
    }
}
