/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: TreeTestBase.java,v 1.40 2004/08/27 21:49:28 linda Exp $
*/

package com.sleepycat.je.tree;

import java.io.File;
import java.io.IOException;

import junit.framework.TestCase;

import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.DbInternal;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.NullCursor;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.txn.LockResult;
import com.sleepycat.je.util.TestUtils;

public class TreeTestBase extends TestCase {
    static protected final boolean DEBUG = true;

    static protected int N_KEY_BYTES = 10;
    static protected int N_ITERS = 1;
    static protected int N_KEYS = 10000;
    static protected int MAX_ENTRIES_PER_NODE = 6;

    protected Tree tree = null;
    protected Key minKey = null;
    protected Key maxKey = null;
    protected Database db = null;
    protected Environment env = null;
    protected File envHome = null;

    public TreeTestBase() {
        envHome = new File(System.getProperty("testdestdir"));
    }

    public void setUp()
        throws IOException  {

        TestUtils.removeFiles("Setup", envHome, FileManager.JE_SUFFIX);
    }

    void initEnv(boolean duplicatesAllowed) 
        throws DatabaseException {

        EnvironmentConfig envConfig = new EnvironmentConfig();
        envConfig.setConfigParam(EnvironmentParams.ENV_RUN_EVICTOR.getName(),
                                 "false");
        envConfig.setConfigParam(EnvironmentParams.NODE_MAX.getName(),
                                 Integer.toString(MAX_ENTRIES_PER_NODE));
        envConfig.setAllowCreate(true);
        envConfig.setTxnNoSync(Boolean.getBoolean(TestUtils.NO_SYNC));
        env = new Environment(envHome, envConfig);

        DatabaseConfig dbConfig = new DatabaseConfig();
        dbConfig.setAllowCreate(true);
        dbConfig.setSortedDuplicates(duplicatesAllowed);
        db = env.openDatabase(null, "foo", dbConfig);

        tree = DbInternal.dbGetDatabaseImpl(db).getTree();
        minKey = null;
        maxKey = null;
    }

    public void tearDown()
	throws DatabaseException, IOException {

        db.close();
        if (env != null) {
            env.close();
        }
        env = null;
        db = null;
        tree = null;
        minKey = null;
        maxKey = null;
        TestUtils.removeFiles("TearDown", envHome, FileManager.JE_SUFFIX);
    }

    protected IN makeDupIN(IN old) {
        IN ret = new IN(DbInternal.dbGetDatabaseImpl(db),
			old.getIdentifierKey(),
                        MAX_ENTRIES_PER_NODE, 2);
        ret.setNodeId(old.getNodeId());
        ret.setIsRoot(old.isRoot());
        for (int i = 0; i < old.getNEntries(); i++) {
            ret.setEntry(i, old.getEntry(i));
        }

        return ret;
    }

    /**
     * Helper routine to insert a key and immediately read it back.
     */
    protected void insertAndRetrieve(NullCursor cursor, Key key, LN ln)
        throws DatabaseException {

        if (minKey == null) {
            minKey = key;
        } else if (key.compareTo(minKey) < 0) {
            minKey = key;
        }

        if (maxKey == null) {
            maxKey = key;
        } else if (maxKey.compareTo(key) < 0) {
            maxKey = key;
        }

        TestUtils.checkLatchCount();
        assertTrue(tree.insert(ln, key, false, cursor,
			       new LockResult(null, null)));
        TestUtils.checkLatchCount();
        assertTrue(retrieveLN(key) == ln);
    }

    /**
     * Helper routine to read the LN referred to by key.
     */
    protected LN retrieveLN(Key key)
        throws DatabaseException {

        TestUtils.checkLatchCount();
        IN n = tree.search(key, Tree.SearchType.NORMAL, -1);
        if (!(n instanceof BIN)) {
            fail("search didn't return a BIN for key: " + key);
        }
        BIN bin = (BIN) n;
        try {
            int index = bin.findEntry(key, false, true);
            if (index == -1) {
                fail("Didn't read back key: " + key);
            } else {
                Node node = bin.getEntry(index).getTarget();
                if (node instanceof LN) {
                    return (LN) node;
                } else {
                    fail("Didn't read back LN for: " + key);
                }
            }
            /* We never get here, but the compiler doesn't know that. */
            return null;
        } finally {
            bin.releaseLatch();
            TestUtils.checkLatchCount();
        }
    }

    /**
     * Using getNextBin, count all the keys in the database.  Ensure that
     * they're returned in ascending order.
     */
    protected int countAndValidateKeys(Tree tree)
        throws DatabaseException {

        TestUtils.checkLatchCount();
        BIN nextBin = (BIN) tree.getFirstNode();
        byte[] zero = { 0x00 };
        Key prevKey = new Key(zero);

        int cnt = 0;

        while (nextBin != null) {
            for (int i = 0; i < nextBin.getNEntries(); i++) {
                ChildReference entry = nextBin.getEntry(i);
                Key curKey = entry.getKey();
                if (curKey.compareTo(prevKey) <= 0) {
                    throw new InconsistentNodeException
                        ("keys are out of order");
                }
                cnt++;
                prevKey = curKey;
            }
            nextBin.releaseLatch();
            TestUtils.checkLatchCount();
            nextBin = tree.getNextBin(nextBin, null);
        }
        TestUtils.checkLatchCount();
        return cnt;
    }

    /**
     * Using getPrevBin, count all the keys in the database.  Ensure that
     * they're returned in descending order.
     */
    protected int countAndValidateKeysBackwards(Tree tree)
        throws DatabaseException {

        TestUtils.checkLatchCount();
        BIN nextBin = (BIN) tree.getLastNode();
        Key prevKey = null;

        int cnt = 0;

        while (nextBin != null) {
            for (int i = nextBin.getNEntries() - 1; i >= 0; i--) {
                ChildReference entry = nextBin.getEntry(i);
                Key curKey = entry.getKey();
                if (prevKey != null &&
                    prevKey.compareTo(curKey) <= 0) {
                    throw new InconsistentNodeException
                        ("keys are out of order");
                }
                cnt++;
                prevKey = curKey;
            }
            nextBin.releaseLatch();
            TestUtils.checkLatchCount();
            nextBin = tree.getPrevBin(nextBin, null);
        }
        return cnt;
    }
}
