/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: RecoveryCheckpointTest.java,v 1.25 2004/05/07 03:56:28 linda Exp $
*/

package com.sleepycat.je.recovery;

import java.util.Hashtable;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.EnvironmentStats;
import com.sleepycat.je.Transaction;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.util.TestUtils;

public class RecoveryCheckpointTest extends RecoveryTestBase {

    public void setExtraProperties()
	throws DatabaseException {

        /*
         * Make sure that the environments in this unit test always
         * run with checkpointing off, so we can call it explcitly.
         */
        envConfig.setConfigParam
            (EnvironmentParams.ENV_RUN_CHECKPOINTER.getName(), "false");
    }

    /**
     * Run checkpoints on empty dbs.
     */
    public void testEmptyCheckpoint()
        throws Throwable {

        createEnvAndDbs(1 << 20, true, NUM_DBS);

        try {
            /*
             * Run checkpoint on empty environment. Should be the second one
             * run, the first was run by recovery when the environment was
             * opened.
             */
            env.checkpoint(forceConfig);
            EnvironmentStats stats = env.getStats(TestUtils.FAST_STATS);
            assertEquals(2, stats.getNCheckpoints());
            assertEquals(2, stats.getLastCheckpointId());

            /* Shutdown, recover */
            Hashtable expectedData = new Hashtable(); // expected values
            closeEnv();  
            recoverAndVerify(expectedData, NUM_DBS); // 3 checkpoints

            /* Another checkpoint */
            EnvironmentConfig envConfig = new EnvironmentConfig();
            envConfig.setTransactional(true);
            env = new Environment(envHome, envConfig);
            env.checkpoint(forceConfig);
            stats = env.getStats(TestUtils.FAST_STATS);
            assertEquals(2, stats.getNCheckpoints());
            assertEquals(7, stats.getLastCheckpointId());

            /* Shutdown, recover */
            env.close();
            recoverAndVerify(expectedData, NUM_DBS);
        } catch (Throwable t) {
            // print stacktrace before trying to clean up files
            t.printStackTrace();
            throw t;
        }
    }

    /**
     * Test checkpoints that end up using BINDeltas -- the recovery must
     * work.
     */
    
    public void testBinDelta()
	throws Throwable {

	createEnvAndDbs(1 << 20, true, NUM_DBS);

        try {
            int numRecs = 28;
            Hashtable expectedData = new Hashtable();
            Transaction txn = env.beginTransaction(null, null);
            insertData(txn, 0, numRecs, expectedData, 1, true, NUM_DBS);
            env.checkpoint(forceConfig);
            insertData(txn, numRecs+1, numRecs+2, expectedData, 1, true, NUM_DBS);
            txn.commit();
            
            /* This next checkpoint will end up using a BINDelta to log the 
             * last inserted record. It will have practically nothing but 
             * the root in the checkpoint.
             */
            env.checkpoint(forceConfig);

            /* Shutdown, recover from a checkpoint that uses BINDeltas. */
            closeEnv();
            recoverAndVerify(expectedData, NUM_DBS);
        } catch (Throwable t) {
            // print stacktrace before trying to clean up files
            t.printStackTrace();
            throw t;
        }
    }

    /** 
     * Test the rollback of transactions that are active during a checkpoint.
     */
    public void testActiveWhileCheckpointing() 
        throws Throwable {

	createEnvAndDbs(1 << 20, true, NUM_DBS);

        try {
            int numRecs = 1;
            Hashtable expectedData = new Hashtable();
            Transaction txn = env.beginTransaction(null, null);
            insertData(txn, 0, numRecs, expectedData, 1, false, NUM_DBS);

            /* Now run a checkpoint while this operation hasn't finished */
            env.checkpoint(forceConfig);
            txn.abort();
            
            /* Shutdown, recover */
            closeEnv();
            recoverAndVerify(expectedData, NUM_DBS);
        } catch (Throwable t) {
            // print stacktrace before trying to clean up files
            t.printStackTrace();
            throw t;
        }
    }
}
