/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: CheckpointActivationTest.java,v 1.7 2004/08/27 13:59:49 mark Exp $
*/

package com.sleepycat.je.recovery;

import java.io.File;
import java.io.IOException;
import java.util.logging.Level;

import junit.framework.TestCase;

import com.sleepycat.je.CheckpointConfig;
import com.sleepycat.je.DbInternal;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.EnvironmentStats;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.util.TestUtils;
import com.sleepycat.je.utilint.Tracer;

public class CheckpointActivationTest extends TestCase {

    private File envHome;

    public CheckpointActivationTest() {
        envHome = new File(System.getProperty("testdestdir"));
    }

    public void setUp()
        throws IOException {

        TestUtils.removeLogFiles("Setup", envHome, false);
    }
    
    public void tearDown()
        throws Exception {

        TestUtils.removeLogFiles("TearDown", envHome, false);
    }

    /**
     * Write elements to the log, check that the right number of
     * checkpoints ran.
     */
    public void testLogSizeBasedCheckpoints() 
        throws Exception {

        final int BYTE_INTERVAL = 5000;
        final int TRACER_OVERHEAD = 26;
        final int N_TRACES = 100;
        final int N_EXTRA_TRACES = 12;
        
        /* Init trace message with zeros. */
        assert BYTE_INTERVAL % N_TRACES == 0;
        int msgBytesPerTrace = (BYTE_INTERVAL / N_TRACES) - TRACER_OVERHEAD;
        StringBuffer traceBuf = new StringBuffer();
        traceBuf.setLength(msgBytesPerTrace);
        String traceMsg = traceBuf.toString();

        Environment env = null;
        try {
            EnvironmentConfig envConfig = new EnvironmentConfig();
            envConfig.setAllowCreate(true);
            envConfig.setConfigParam(EnvironmentParams.
                                     CHECKPOINTER_BYTES_INTERVAL.getName(),
                                     String.valueOf(BYTE_INTERVAL));

            /*
             * This test needs to control exactly how much goes into the log,
             * so disable daemons. 
             */
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_EVICTOR.getName(), "false");
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_INCOMPRESSOR.getName(), "false");
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_CLEANER.getName(), "false");
            env = new Environment(envHome, envConfig);

            /* 
             * Get a first reading on number of checkpoints run. Read once
             * to clear, then read again.
             */
            StatsConfig statsConfig = new StatsConfig();
            statsConfig.setFast(true);
            statsConfig.setClear(true);
            EnvironmentStats stats = env.getStats(statsConfig); // clear stats

            stats = env.getStats(statsConfig);  // read again
            assertEquals(0, stats.getNCheckpoints());

            /* Run several checkpoints to ensure they occur as expected.  */
            for (int i = 0; i < 10; i += 1) {

                /* Write enough to prompt a checkpoint. */
                EnvironmentImpl envImpl =
                    DbInternal.envGetEnvironmentImpl(env);
                /*
                 * Extra traces are used to account for the fact that the
                 * CheckpointMonitor only activates the Checkpointer every
                 * 1/10 of the checkpoint byte interval.
                 */
                for (int j = 0; j < N_TRACES + N_EXTRA_TRACES; j += 1) {
                    Tracer.trace(Level.SEVERE, envImpl, traceMsg);
                    Thread.yield();
                }

                /* Wait for the daemon to do its work. */
                boolean done = false;
                for (int j = 0; j < 10 && !done; j += 1) {
                    Thread.sleep(100);
                    Thread.yield();
                    stats = env.getStats(statsConfig);
                    if (stats.getNCheckpoints() > 0) {
                        done = true;
                    }
                }
                if (!done) {
                    fail("i=" + i);
                }
            }
        } catch (Exception e) {
            /* 
             * print stack trace now, else it gets subsumed in exceptions
             * caused by difficulty in removing log files.
             */
            e.printStackTrace();
            throw e;
        } finally {
            if (env != null) {
                env.close();
            }
        }
    }

    /* Test programmatic call to checkpoint. */
    public void testApiCalls() 
        throws Exception {

        Environment env = null;
        try {
            EnvironmentConfig envConfig = new EnvironmentConfig();
            envConfig.setAllowCreate(true);
            envConfig.setConfigParam(EnvironmentParams.
                                     CHECKPOINTER_BYTES_INTERVAL.getName(),
                                     "1000");

            /* Disable all daemons */
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_EVICTOR.getName(), "false");
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_INCOMPRESSOR.getName(), "false");
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_CLEANER.getName(), "false");
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_CHECKPOINTER.getName(), "false");
            env = new Environment(envHome, envConfig);

            /* 
             * Get a first reading on number of checkpoints run. Read once
             * to clear, then read again.
             */
            StatsConfig statsConfig = new StatsConfig();
            statsConfig.setFast(true);
            statsConfig.setClear(true);
            EnvironmentStats stats = env.getStats(statsConfig); // clear stats

            stats = env.getStats(statsConfig);  // read again
            assertEquals(0, stats.getNCheckpoints());

            /* From the last checkpoint start lsn, there should be the
             * checkpoint end log entry and a trace message. These take 196
             * bytes. 
             */
            CheckpointConfig checkpointConfig = new CheckpointConfig();

            /* Should not cause a checkpoint, too little growth. */
            checkpointConfig.setKBytes(1);
            env.checkpoint(checkpointConfig);
            stats = env.getStats(statsConfig);  // read again
            assertEquals(0, stats.getNCheckpoints());

            /* Fill up the log, there should be a checkpoint. */
            String filler = "123456789012345678901245678901234567890123456789";
            EnvironmentImpl envImpl = DbInternal.envGetEnvironmentImpl(env);
            for (int i = 0; i < 10; i++) {
                Tracer.trace(Level.SEVERE,  envImpl, filler);
            }
            env.checkpoint(checkpointConfig);
            stats = env.getStats(statsConfig);  // read again
            assertEquals(1, stats.getNCheckpoints());

            /* Try time based, should not checkpoint. */
            checkpointConfig.setKBytes(0);
            checkpointConfig.setMinutes(1);
            env.checkpoint(checkpointConfig);
            stats = env.getStats(statsConfig);  // read again
            assertEquals(0, stats.getNCheckpoints());

            /* Sleep, enough time has passed for a checkpoint, but nothing
             * was written to the log.
             */
            Thread.sleep(1000);
            env.checkpoint(checkpointConfig);
            stats = env.getStats(statsConfig);  // read again
            assertEquals(0, stats.getNCheckpoints());

            /* Log something, now try a checkpoint. */
            Tracer.trace(Level.SEVERE,  envImpl, filler);
            env.checkpoint(checkpointConfig);
            stats = env.getStats(statsConfig);  // read again
            // TODO: make this test more timing independent. Sometimes 
            // the assertion will fail.
            // assertEquals(1, stats.getNCheckpoints());
                        
        } catch (Exception e) {
            /* 
             * print stack trace now, else it gets subsumed in exceptions
             * caused by difficulty in removing log files.
             */
            e.printStackTrace();
            throw e;
        } finally {
            if (env != null) {
                env.close();
            }
        }
    }
}
