/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: EvictActionTest.java,v 1.6 2004/08/05 15:33:31 mark Exp $
*/

package com.sleepycat.je.evictor;

import java.io.File;
import java.io.IOException;

import junit.framework.TestCase;

import com.sleepycat.bind.EntryBinding;
import com.sleepycat.bind.tuple.TupleBinding;
import com.sleepycat.je.Cursor;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.je.DatabaseEntry;
import com.sleepycat.je.DbInternal;
import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.LockMode;
import com.sleepycat.je.OperationStatus;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.dbi.MemoryBudget;
import com.sleepycat.je.util.TestUtils;


/**
 * This tests exercises the act of eviction and determines whether the 
 * expected nodes have been evicted properly.
 */
public class EvictActionTest extends TestCase {
    private static final boolean DEBUG = false;

    private static final int NUM_KEYS = 30;
    private static final int NUM_DUPS = 30;
    protected File envHome = null;
    Environment env = null;

    public EvictActionTest() {
        envHome = new File(System.getProperty("testdestdir"));
    }

    public void setUp()
        throws IOException {

        TestUtils.removeLogFiles("Setup", envHome, false);
    }
    
    public void tearDown()
        throws Exception {

        TestUtils.removeLogFiles("TearDown", envHome, false);
    }

    public void testEvict()
        throws Throwable {
        doEvict(25, 10, 50, 50000, true);
    }

    public void testManyBatches()
        throws Throwable {
        doEvict(5, 5, 50, 5000, true);
    }

    public void testNoNeedToEvict()
        throws Throwable {
        doEvict(100, 100, 80, 500000, false);
    }

    /**
     * Evict in very controlled circumstances. Check that we first strip
     * BINs and later evict BINS.
     */
    private void doEvict(int batchPercentage,
                         int nodeScanPercentage,
                         int floor,
                         int maxMem,
                         boolean shouldEvict) 
        throws Throwable {

        Database db = null;
        try {
            /* Make a non-txnal env w/no daemons and small nodes. */
            EnvironmentConfig envConfig = new EnvironmentConfig();
            envConfig.setAllowCreate(true);
            envConfig.setTxnNoSync(Boolean.getBoolean(TestUtils.NO_SYNC));
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_EVICTOR.getName(), "false");
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_INCOMPRESSOR.getName(), "false");
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_CLEANER.getName(), "false");
            envConfig.setConfigParam(EnvironmentParams.
                                     ENV_RUN_CHECKPOINTER.getName(), "false");

            envConfig.setConfigParam(EnvironmentParams.
                             EVICTOR_EVICTION_BATCH_PERCENTAGE.getName(),
                             (new Integer(batchPercentage)).toString());
            envConfig.setConfigParam(EnvironmentParams.
                             EVICTOR_NODE_SCAN_PERCENTAGE.getName(),
                             (new Integer(nodeScanPercentage)).toString());
            envConfig.setConfigParam(EnvironmentParams.
                                     EVICTOR_USEMEM_FLOOR.getName(),
                                     (new Integer(floor)).toString());
            envConfig.setConfigParam(EnvironmentParams.
                                     MAX_MEMORY.getName(),
                                     new Integer(maxMem).toString());

            /* Make small nodes */
            envConfig.setConfigParam(EnvironmentParams.
                                     NODE_MAX.getName(), "6");
            env = new Environment(envHome, envConfig);


            /* Make a database, insert data. */
            DatabaseConfig dbConfig = new DatabaseConfig();
            dbConfig.setAllowCreate(true);
            dbConfig.setSortedDuplicates(true);
            db = env.openDatabase(null, "foo", dbConfig);
            EntryBinding intBinding =
                TupleBinding.getPrimitiveBinding(Integer.class);

            insertData(db, intBinding);

            /* Evict once after insert. */
            evictAndCheck(env, db, intBinding, shouldEvict);

            /* Evict again after verification. */
            evictAndCheck(env, db, intBinding, shouldEvict);

        } catch (Throwable t) {
            t.printStackTrace();
            throw t;
        } finally {
            if (db != null) {
                db.close();
            }

            if (env != null) {
                env.close();
            }
        }
    }

    private void insertData(Database db, EntryBinding binding) 
        throws Exception {
        DatabaseEntry key = new DatabaseEntry();
        DatabaseEntry data = new DatabaseEntry();
        for (int i = 0; i < NUM_KEYS; i++) {

            binding.objectToEntry(new Integer(i), key);

            if ((i % 5) == 0) {
                for (int j = 10; j < (NUM_DUPS + 10); j++) {
                    binding.objectToEntry(new Integer(j), data);
                    db.put(null, key, data);
                }
            } else {
                binding.objectToEntry(new Integer(i+1), data);
                db.put(null, key, data);
            }
        }
    }

    private void verifyData(Database db, EntryBinding binding)
        throws Exception {

        /* Full scan of data, make sure we can bring everything back in. */
        Cursor cursor = db.openCursor(null, null);
        DatabaseEntry data = new DatabaseEntry();
        DatabaseEntry key = new DatabaseEntry();
        
        for (int i = 0; i < NUM_KEYS; i++) {
            if ((i % 5) ==0) {
                for (int j = 10; j < (NUM_DUPS + 10); j++) {
                    assertEquals(OperationStatus.SUCCESS,
                                 cursor.getNext(key, data, LockMode.DEFAULT));
                    assertEquals(new Integer(i),
                                 binding.entryToObject(key));
                    assertEquals(new Integer(j),
                             binding.entryToObject(data));

                }
            } else {
                assertEquals(OperationStatus.SUCCESS,
                             cursor.getNext(key, data, LockMode.DEFAULT));
                assertEquals(new Integer(i),
                             binding.entryToObject(key));
                assertEquals(new Integer(i+1),
                             binding.entryToObject(data));
            }
        }

        assertEquals(OperationStatus.NOTFOUND,
                     cursor.getNext(key, data, LockMode.DEFAULT));
        cursor.close();
    }

    private void evictAndCheck(Environment env,
                               Database db,
                               EntryBinding binding,
                               boolean shouldEvict) 
        throws Throwable {
        EnvironmentImpl envImpl = DbInternal.envGetEnvironmentImpl(env);
        MemoryBudget mb = envImpl.getMemoryBudget();

        /* 
         * The following batches are run in a single evictMemory() call:
         * 1st eviction will strip DBINs.
         * 2nd will evict DBINs
         * 3rd will evict DINs
         * 4th will strip BINs
         * 5th will evict BINs
         * 6th will evict INs
         * 7th will evict INs
         */
        long preEvictMem = mb.getCacheMemoryUsage();
        TestUtils.validateMemoryUsage(envImpl, true);
        env.evictMemory();
        long postEvictMem = mb.getCacheMemoryUsage();

        TestUtils.validateMemoryUsage(envImpl, true);
        if (DEBUG) {
            System.out.println("preEvict=" + preEvictMem +
                               " postEvict=" + postEvictMem);
        }


        if (shouldEvict) {
            assertTrue("preEvict=" + preEvictMem +
                       " postEvict=" + postEvictMem,
                       (preEvictMem > postEvictMem));
        } else {
            assertTrue("preEvict=" + preEvictMem +
                       " postEvict=" + postEvictMem,
                       (preEvictMem == postEvictMem));
        }

        verifyData(db, binding);
        TestUtils.validateMemoryUsage(envImpl, true);
    }
}
