/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2004
*      Sleepycat Software.  All rights reserved.
*
* $Id: TruncateTest.java,v 1.7 2004/06/02 15:00:44 cwl Exp $
*/

package com.sleepycat.je;

import java.io.File;
import java.io.IOException;

import junit.framework.TestCase;

import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.util.TestUtils;

/**
 * Basic database operations, excluding configuration testing.
 */
public class TruncateTest extends TestCase {
    private static final int NUM_RECS = 257;

    private File envHome;
    private Environment env;

    public TruncateTest() {
        envHome = new File(System.getProperty("testdestdir"));
    }

    public void setUp()
        throws IOException {

        TestUtils.removeLogFiles("Setup", envHome, false);
    }
    
    public void tearDown()
        throws Exception {

        try {
            /* Close in case we hit an exception and didn't close. */
            env.close();
        } catch (DatabaseException e) {
            /* Ok if already closed */
        }
        env = null; // for JUNIT, to reduce memory usage when run in a suite.
        TestUtils.removeLogFiles("TearDown", envHome, false);
    }

    public void testTruncateCommit()
        throws Throwable {

        doTruncate(false, false, true);
    }

    public void testTruncateCommitAutoTxn()
        throws Throwable {

        doTruncate(false, true, true);
    }

    public void testTruncateAbort()
        throws Throwable {

        doTruncate(true, false, true);
    }

    public void testTruncateCommitNoCount()
        throws Throwable {

        doTruncate(false, false, false);
    }

    public void testTruncateCommitAutoTxnNoCount()
        throws Throwable {

        doTruncate(false, true, false);
    }

    public void testTruncateAbortNoCount()
        throws Throwable {

        doTruncate(true, false, false);
    }

    /**
     * 1. Populate a database.
     * 2. Truncate.
     * 3. Commit or abort.
     * 4. Check that database has the right amount of records.
     */
    private void doTruncate(boolean abort,
			    boolean useAutoTxn,
                            boolean countRecords)
        throws Throwable {

        try {
            int numRecsAfterTruncate =
                useAutoTxn ? 0 : ((abort) ? NUM_RECS : 0);
            Database myDb = initEnvAndDb(true);
            DatabaseEntry key = new DatabaseEntry();
            DatabaseEntry data = new DatabaseEntry();

            /* Populate database. */
            for (int i = NUM_RECS; i > 0; i--) {
                key.setData(TestUtils.getTestArray(i));
                data.setData(TestUtils.getTestArray(i));
                assertEquals(OperationStatus.SUCCESS,
			     myDb.put(null, key, data));
            }

            /* Truncate, check the count, commit. */
            int truncateCount = 0;
            if (useAutoTxn) {
                truncateCount = myDb.truncate(null, countRecords);
            } else {
                Transaction txn = env.beginTransaction(null, null);
                truncateCount = myDb.truncate(txn, countRecords);
            
                if (abort) {
                    txn.abort();

                    /* 
                     * The handle is now invalid after the abort. Close the
                     * database handle and re-open.
                     */
                    myDb.close();
                    DatabaseConfig dbConfig = new DatabaseConfig();
                    dbConfig.setSortedDuplicates(true);
                    myDb = env.openDatabase(null, "testDB", dbConfig);
                } else {
                    txn.commit();
                }
            }
            if (countRecords) {
                assertEquals(NUM_RECS, truncateCount);
            } else {
                assertEquals(-1, truncateCount);
            }

            /* Do a cursor read, make sure there's the right amount of data. */
            int count = 0;
            Cursor cursor = myDb.openCursor(null, null);
            while (cursor.getNext(key, data, LockMode.DEFAULT) ==
                   OperationStatus.SUCCESS) {
                count++;
            }
            assertEquals(numRecsAfterTruncate, count);
	    cursor.close();

            /* Recover the database. */
            myDb.close();
            env.close();
            myDb = initEnvAndDb(true);

            /* Check data after recovery. */
            count = 0;
            cursor = myDb.openCursor(null, null);
            while (cursor.getNext(key, data, LockMode.DEFAULT) ==
                   OperationStatus.SUCCESS) {
                count++;
            }
            assertEquals(numRecsAfterTruncate, count);
	    cursor.close();
            myDb.close();
            env.close();
        } catch (Throwable t) {
            t.printStackTrace();
            throw t;
        }
    }

    /**
     * Set up the environment and db.
     */
    private Database initEnvAndDb(boolean isTransactional)
        throws DatabaseException {

        EnvironmentConfig envConfig = new EnvironmentConfig();
        envConfig.setTransactional(isTransactional);
        envConfig.setConfigParam
            (EnvironmentParams.ENV_CHECK_LEAKS.getName(), "false");
        envConfig.setConfigParam(EnvironmentParams.NODE_MAX.getName(), "6");
        envConfig.setAllowCreate(true);
        env = new Environment(envHome, envConfig);

        /* Make a db and open it. */
        DatabaseConfig dbConfig = new DatabaseConfig();
        dbConfig.setTransactional(isTransactional);
        dbConfig.setSortedDuplicates(true);
        dbConfig.setAllowCreate(true);
        Database myDb = env.openDatabase(null, "testDB", dbConfig);
        return myDb;
    }
}
