package dbxmltest;
import com.sleepycat.db.DatabaseException;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.FileNotFoundException;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.sleepycat.dbxml.*;

public class NodeHandleTest{
    // Container name;
    private static final String CON_NAME = "testData.dbxml";
    private static String docString = "<root>"
	+ "<a_node atr1='test' atr2='test2'><b_node/>"
	+ "<c_node>Other text</c_node><d_node>dtext</d_node>atext</a_node>"
	+"</root>";

    private TestConfig hp = null;
    private XmlManager mgr = null;
    private XmlContainer cont = null;

    @BeforeClass
	public static void setupClass() {
	System.out.println("Begin test Node Handle");
	TestConfig.fileRemove(XmlTestRunner.getEnvironmentPath());
    }

    @Before
	public void setUp() throws Throwable {
	if(hp != null)
	    TestConfig.fileRemove(XmlTestRunner.getEnvironmentPath());
	hp = new TestConfig(XmlTestRunner.getEnvironmentType(),
			    XmlTestRunner.isNodeContainer(), XmlTestRunner.getEnvironmentPath());
	mgr = hp.createManager(hp.getEnvironmentPath(), null);
	cont = hp.createContainer(CON_NAME, mgr);
	cont.addAlias("container");
	// use the same document for all tests so put it here
	cont.putDocument("test", docString);
    }

    @After
	public void tearDown() throws Throwable {
	hp.closeContainer(cont);
	hp.closeManager(mgr);
	hp.closeEnvironment();
	TestConfig.fileRemove(XmlTestRunner.getEnvironmentPath());
    }

    @AfterClass
	public static void tearDownClass() {
	System.out.println("Finished test Node Handle");
    }

    @Test
	public void NodeHandleContainer()throws XmlException {
	// transactions are not relevant to this case so ignore them
	XmlUpdateContext uc = mgr.createUpdateContext();
	String contextQuery = "collection('testData.dbxml')/root/a_node";
	String elemQuery="declare namespace ma = 'http://www.xxx.ag/xquery/extensions';"+ 
	    "declare namespace dbxml = 'http://www.sleepycat.com/2002/dbxml';"+
	    "declare variable $handle external;"+
	    "declare function ma:start($node){"+
	      "for $nextnode in $node/* return "+
	        "dbxml:node-to-handle($nextnode)"+ 
	    "};"+ 
	    "ma:start(dbxml:handle-to-node('container', $handle))";
	String attrQuery="declare namespace ma = 'http://www.xxx.ag/xquery/extensions';"+ 
	    "declare namespace dbxml = 'http://www.sleepycat.com/2002/dbxml';"+
	    "declare variable $handle external;"+
	    "declare function ma:start($node){"+
	      "for $nextnode in $node/@* return "+
	        "dbxml:node-to-handle($nextnode)"+ 
	    "};"+ 
	    "ma:start(dbxml:handle-to-node('container', $handle))";
	String textQuery="declare namespace ma = 'http://www.xxx.ag/xquery/extensions';"+ 
	    "declare namespace dbxml = 'http://www.sleepycat.com/2002/dbxml';"+
	    "declare variable $handle external;"+
	    "declare function ma:start($node){"+
	      "for $nextnode in $node/text() return "+
	        "dbxml:node-to-handle($nextnode)"+ 
	    "};"+ 
	    "ma:start(dbxml:handle-to-node('container', $handle))";

	XmlQueryContext context = mgr.createQueryContext();
	XmlResults results = null;
	XmlResults res = null;
	try{
	    results = mgr.query(contextQuery, context);
	    assertTrue(results.size() == 1);
	    XmlValue v = results.next();
	    String h = v.getNodeHandle();
	    context.setVariableValue("handle", new XmlValue(h));

	    res = mgr.query(elemQuery, context);
	    assertTrue(res.size() == 3); // a_node has 3 child elements
	    res.delete();

	    res = mgr.query(attrQuery, context);
	    assertTrue(res.size() == 2); // a_node has 2 attributes
	    res.delete();

	    res = mgr.query(textQuery, context);
	    assertTrue(res.size() == 1); // a_node has 1 text child
	    res.delete();
	    results.delete();
	}catch(XmlException e){
	    throw e;
	} finally {
	    if (results != null)
		results.delete();
	    if (res != null)
		res.delete();
	}
    }

    @Test
	public void NodeHandleConstructed()throws XmlException {
	// transactions are not relevant to this case so ignore them
	XmlUpdateContext uc = mgr.createUpdateContext();
	String elemQuery="declare namespace ma = 'http://www.xxx.ag/xquery/extensions';"+ 
	    "declare namespace dbxml = 'http://www.sleepycat.com/2002/dbxml';"+
	    "declare function ma:start($node){"+
	      "for $nextnode in $node/* return "+
	        "dbxml:node-to-handle($nextnode)"+ 
	    "};"+ 
	    "ma:start(<a><b/></a>)";
	String attrQuery="declare namespace ma = 'http://www.xxx.ag/xquery/extensions';"+ 
	    "declare namespace dbxml = 'http://www.sleepycat.com/2002/dbxml';"+
	    "declare function ma:start($node){"+
	      "for $nextnode in $node/@* return "+
	        "dbxml:node-to-handle($nextnode)"+ 
	    "};"+ 
	    "ma:start(<a a1='a1' a2='a2'><b/></a>)";
	String textQuery="declare namespace ma = 'http://www.xxx.ag/xquery/extensions';"+ 
	    "declare namespace dbxml = 'http://www.sleepycat.com/2002/dbxml';"+
	    "declare function ma:start($node){"+
	      "for $nextnode in $node/text() return "+
	        "dbxml:node-to-handle($nextnode)"+ 
	    "};"+ 
	    "ma:start(<a a1='a1' a2='a2'>text1<b/>text2</a>)";

	XmlQueryContext context = mgr.createQueryContext();
	XmlResults res = null;
	try{
	    res = mgr.query(elemQuery, context);
	    res.delete();
	    fail("Query for transient node handles should have failed");
	}catch(XmlException e){
	    //ignore error
	}
	try{
	    res = mgr.query(attrQuery, context);
	    res.delete();
	    fail("Query for transient node handles should have failed");
	}catch(XmlException e){
	    //ignore error
	}
	try{
	    res = mgr.query(textQuery, context);
	    res.delete();
	    fail("Query for transient node handles should have failed");
	}catch(XmlException e){
	    //ignore error
	}
	finally {
	    if (res != null)
		res.delete();
	}

    }
};
