/*-
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 2007,2009 Oracle.  All rights reserved.
 *
 */

package dbxmltest;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import com.sleepycat.dbxml.XmlContainer;
import com.sleepycat.dbxml.XmlContainerConfig;
import com.sleepycat.dbxml.XmlDocument;
import com.sleepycat.dbxml.XmlDocumentConfig;
import com.sleepycat.dbxml.XmlException;
import com.sleepycat.dbxml.XmlIndexLookup;
import com.sleepycat.dbxml.XmlIndexSpecification;
import com.sleepycat.dbxml.XmlManager;
import com.sleepycat.dbxml.XmlQueryContext;
import com.sleepycat.dbxml.XmlResults;
import com.sleepycat.dbxml.XmlUpdateContext;

public class IndexLookupTest {
    private static final String cname = "testData.dbxml";
    private static String docName = "testDoc.xml";

    private TestConfig hp = null;
    private XmlManager mgr = null;
    private XmlContainer cont = null;

    @BeforeClass
	public static void setupClass() {
	System.out.println("Begin test XmlIndexLookup");
    }

    @AfterClass
	public static void tearDownClass() {
	System.out.println("Finished test XmlIndexLookup");
    }

    @Before
	public void setUp() throws Throwable {
	TestConfig.fileRemove(XmlTestRunner.getEnvironmentPath());
	hp = new TestConfig(XmlTestRunner.getEnvironmentType(),
			    XmlTestRunner.isNodeContainer(), 
			    XmlTestRunner.getEnvironmentPath());
	mgr = hp.createManager();
	cont = hp.createContainer(cname, mgr);
    }

    @After
	public void tearDown() throws Throwable {
	hp.closeContainer(cont);
	hp.closeManager(mgr);
	hp.closeEnvironment();
	TestConfig.fileRemove(null);
    }

    /*
     * Test for execute(XmlQueryContext context)
     * 
     */
    @Test
	public void testExecute() throws Throwable {

	XmlContainerConfig xcc = new XmlContainerConfig();
	xcc.setContainerType(XmlContainer.NodeContainer);
	XmlDocument doc = mgr.createDocument();
	XmlUpdateContext uc = mgr.createUpdateContext();
	XmlQueryContext xqc = mgr.createQueryContext();
	String docString = "<names><name>Jack</name><name>Joe"
	    + "</name><name>Mike</name><my><name>Kate</name></my></names>";
	doc.setContent(docString);
	doc.setName(docName);

	cont.putDocument(doc, uc);

	XmlIndexSpecification is = cont.getIndexSpecification();
	is.addIndex("", "name", "node-element-equality-string");
	is.addIndex("", "name", "node-element-substring-string");
	is.addIndex("", "name", "edge-element-equality-string");

	// create index
	cont.setIndexSpecification(is, uc);

	String uri = "";
	String name = "name";
	String idxStrategy = "node-element-equality-string";

	XmlIndexLookup xil =
	    mgr.createIndexLookup(cont, uri, name, idxStrategy);
	XmlResults xr = null;

	try {
	    xr = xil.execute(xqc);
	    assertEquals(xr.size(), 4);
	} catch (XmlException e) {
	    throw e;
	}finally{
	    is.delete();
	    xil.delete();
	    if(xr != null) xr.delete();
	    xr = null;
	}

	// EvaluationType Lazy
	xil = mgr.createIndexLookup(cont, uri, name, idxStrategy);
	try {
	    xqc.setEvaluationType(XmlQueryContext.Lazy);
	    xr = xil.execute(xqc);
	    assertTrue(xr.hasNext());
	} catch (XmlException e) {
	    throw e;
	}finally{
	    xil.delete();
	    if(xr != null) xr.delete();
	    xr = null;
	}

	// set the queryContext to eager
	xqc.setEvaluationType(XmlQueryContext.Eager);
	// Unknown index specification
	xil = mgr.createIndexLookup(cont, uri, name,
				    "node-string-equality-string");
	try {
	    xr = xil.execute(xqc);
	    fail("Index lookup succeeded on non-existent index.");
	} catch (XmlException e) {
	    assertNotNull(e.getMessage());
	    assertEquals(e.getErrorCode(), XmlException.UNKNOWN_INDEX);
	}finally{
	    xil.delete();
	    if(xr != null) xr.delete();
	    xr = null;
	}

	// substring not available
	xil = mgr.createIndexLookup(cont, uri, name,
				    "node-element-substring-string");
	try {
	    xr = xil.execute(xqc);
	    fail("Index lookup succeeded on unknown index.");
	} catch (XmlException e) {
	    assertNotNull(e.getMessage());
	    assertEquals(e.getErrorCode(), XmlException.UNKNOWN_INDEX);
	}finally{
	    xil.delete();
	    if(xr != null) xr.delete();
	    xr = null;
	}

	// set parent
	xil = mgr.createIndexLookup(cont, "", "name",
				    "edge-element-equality-string");
	xil.setParent("", "my");
	try {
	    xr = xil.execute(xqc);
	    assertEquals(xr.size(), 1);
	} catch (XmlException e) {
	    throw e;
	}finally{
	    xil.delete();
	    if(xr != null) xr.delete();
	    xr = null;
	}
    }

    /*
     * Test for execute(XmlQueryContext context, XmlDocumentConfig config)
     * 
     */
    @Test
	public void testExecute_Conf() throws Throwable {

	XmlDocument doc = mgr.createDocument();
	XmlUpdateContext uc = mgr.createUpdateContext();
	XmlQueryContext xqc = mgr.createQueryContext();
	String docString = "<names><name>Jack</name><name>Joe"
	    + "</name><name>Mike</name></names>";
	doc.setContent(docString);
	doc.setName(docName);

	cont.putDocument(doc, uc);

	XmlIndexSpecification is = cont.getIndexSpecification();
	is.addIndex("", "name", "node-element-equality-string");

	cont.setIndexSpecification(is, uc);

	String uri = "";
	String name = "name";
	String idxStrategy = "node-element-equality-string";

	XmlIndexLookup xil =
	    mgr.createIndexLookup(cont, uri, name, idxStrategy);
	XmlDocumentConfig xdc = new XmlDocumentConfig();
	xdc.setLazyDocs(true);

	XmlResults xr = null;
	try {
	    xr = xil.execute(xqc, xdc);
	    assertEquals(xr.size(), 3);
	} catch (XmlException e) {
	    throw e;
	}finally{
	    is.delete();
	    xil.delete();
	    if(xr != null) xr.delete();
	    xr = null;
	}

	xil = mgr.createIndexLookup(cont, uri, name, idxStrategy);
	xdc = new XmlDocumentConfig();
	xdc.setDocumentProjection(true);

	try {
	    xr = xil.execute(xqc, xdc);
	    fail("Execute succeeded with an invalid configuration.");
	} catch (XmlException e) {
	    assertEquals(e.getErrorCode(), XmlException.INVALID_VALUE);
	}finally{
	    xil.delete();
	    if(xr != null) xr.delete();
	    xr = null;
	}
    }

    /*
     * Test for getContainer()
     * 
     * 
     */
    @Test
	public void testGetContainer() throws Throwable {
	XmlDocument doc = mgr.createDocument();
	XmlUpdateContext uc = mgr.createUpdateContext();

	String docString = "<names><name>Jack</name><name>Joe"
	    + "</name><name>Mike</name></names>";
	doc.setContent(docString);
	doc.setName(docName);

	cont.putDocument(doc, uc);

	XmlIndexSpecification is = cont.getIndexSpecification();
	is.addIndex("", "name", "node-element-equality-string");

	cont.setIndexSpecification(is, uc);

	String uri = "";
	String name = "name";
	String idxStrategy = "node-element-equality-string";

	XmlIndexLookup xil =
	    mgr.createIndexLookup(cont, uri, name, idxStrategy);

	XmlContainer myCont = null;

	//test getContainer return the right container
	try {
	    myCont = xil.getContainer();
	    assertEquals(myCont.getName(), "testData.dbxml");
	} catch (XmlException e) {
	    throw e;
	}finally{
	    is.delete();
	    xil.delete();
	}
    }
    
    /*
     * Test getIndex()
     * 
     */
    @Test
	public void testGetIndex() throws Throwable{
	XmlDocument doc = mgr.createDocument();
	XmlUpdateContext uc = mgr.createUpdateContext();

	String docString = "<names><name>Jack</name><name>Joe"
	    + "</name><name>Mike</name></names>";
	doc.setContent(docString);
	doc.setName(docName);

	cont.putDocument(doc, uc);

	XmlIndexSpecification is = cont.getIndexSpecification();
	is.addIndex("", "name", "node-element-equality-string");

	cont.setIndexSpecification(is, uc);

	String uri = "";
	String name = "name";
	String idxStrategy = "node-element-equality-string";

	XmlIndexLookup xil =
	    mgr.createIndexLookup(cont, uri, name, idxStrategy);

	//test getContainer return the right container
	try {
	    String myIdx = xil.getIndex();
	    assertEquals(myIdx, idxStrategy);
	} catch (XmlException e) {
	    throw e;
	}finally{
	    is.delete();
	    xil.delete();
	}
    }
    
    /*
     * Test setIndex()
     * 
     */
    @Test
	public void testSetIndex() throws Throwable{
	XmlDocument doc = mgr.createDocument();
	XmlUpdateContext uc = mgr.createUpdateContext();

	String docString = "<names><name>Jack</name><name>Joe"
	    + "</name><name>Mike</name></names>";
	doc.setContent(docString);
	doc.setName(docName);

	cont.putDocument(doc, uc);

	XmlIndexSpecification is = cont.getIndexSpecification();
	is.addIndex("", "name", "node-element-equality-string");

	cont.setIndexSpecification(is, uc);

	String uri = "";
	String name = "name";
	String idxStrategy = "node-element-equality-string";

	XmlIndexLookup xil =
	    mgr.createIndexLookup(cont, uri, name, idxStrategy);

	//test getContainer return the right container
	try {
	    xil.setIndex(idxStrategy);
	    assertEquals(xil.getIndex(), idxStrategy);
	} catch (XmlException e) {
	    throw e;
	}finally{
	    is.delete();
	    xil.delete();
	}
    }

}
