/*
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1998,2006 Oracle.  All rights reserved.
 *
 * $Id: perf_test.cpp,v 1.9 2006/10/30 17:46:17 bostic Exp $
 */
#include "perf_workload.hpp"
#include "DbXmlWorkload.hpp"
#include "QueryWorkload.hpp"
#include "PutAndQueriesWorkload.hpp"
#include "PutWorkload.hpp"
#include <iostream>
#include <cstdio>
#include <dbxml/DbXml.hpp>
#ifdef HAVE_GETOPT
#include <unistd.h>
#else
extern "C" int getopt(int argc, char * const argv[], const char *optstring);
extern "C" char *optarg;
extern "C" int optind;
#endif

using namespace DbXml;
using namespace std;

static void usage(const string &progname, int exitCode)
{
	string::size_type pos = progname.rfind('/');
	if(pos == string::npos) {
		pos = progname.rfind('\\');
	}
	if(pos == string::npos) {
		pos = 0;
	}
	else {
		++pos;
	}

	PerfParams default_params;

	cerr << "Usage: " << progname.substr(pos) << " [OPTIONS] <workload_name>" << endl;
	cerr << endl;
	cerr << "Options:" << endl;
	cerr << "-C <cachesize>\tset the cache size to the specified value, in Mb. (default: " << (default_params.cacheSize / (1024 * 1024)) << ")" << endl;
	cerr << "-c <container>\tuse the specified container name. (default: " << default_params.containerName << ")" << endl;
	cerr << "-h <home>\tuse specified directory as a Berkeley DB environment. (default: " << default_params.envDir << ")" << endl;
	cerr << "-i <iterations>\tperform the specified number of iterations per thread. (default: 1000)" << endl;
	cerr << "-n <number>\tset the number of operations per transaction. (default: " << default_params.opsPerTransaction << ")" << endl;
	cerr << "-s <storage>\tuse the specified storage model, including DLS, DLS+, NLS and NLS+. (default: NLS)" << endl;
	cerr << "-t <threads>\tuse the specified number of threads. (default: 2)" << endl;
	cerr << "-V\t\tprint software version" << endl;
	cerr << "-v\t\tverbose. Using the switch a second time increases verbosity" << endl;
	cerr << "-?\t\thelp -- print this message" << endl;
	cerr << endl;
	cerr << "Workloads:" << endl;
	cerr << "query\t\tPerforms concurrent presence based index lookup queries" << endl;
	cerr << "query\t\tPerforms concurrent value based index lookup queries" << endl;
	cerr << "put\t\tPerforms concurrent putDocument operations" << endl;
	cerr << "putandqueries\tPerforms concurrent database queries while also calling putDocument" << endl;
	cerr << endl;
	exit(exitCode);
}

DbXmlWorkload *constructWorkload(const string &name, struct perf_globals *globals, PerfParams &params)
{
	DbXmlWorkload *workload = 0;
	if(name == "query") {
		workload = new QueryWorkload(globals, params);
	}
	else if(name == "query2") {
		workload = new Query2Workload(globals, params);
	}
	else if(name == "putandqueries") {
		workload = new PutAndQueriesWorkload(globals, params);
	}
	else if(name == "put") {
		workload = new PutWorkload(globals, params);
	}
	return workload;
}

/*
 * Structure of a PerfWorkload test program:
 *  o implement subclass of PerfWorkload
 *  o in main:
 *    - call perf_init_globals with configuration parameters,
 *    such as number of threads and iterations
 *    - construct an instance of your subclass
 *    - call its run() method
 *    - optionally dump out run information such as timers.
 *
 * See interfaces in:
 *   perf_workload.hpp
 *   perf_work.h
 *
 * perf_work.h and perf_work.c are in C to allow C-based
 * programs to be written.
 * 
 * PerfWorkload is a simple C++ wrapper to simplify C++
 * use of this framework.
 */

// TBD (make parameterized):
//   multiple ops/txn
//   cleaning env and container before/after
//   specify indexes

int
main(int argc, char **argv)
{
	const char *progName = argv[0];

	int verbose = 0;

	int iters = 1000;
	int threads = 2;

	PerfParams params;

	int ch;
	while ((ch = getopt(argc, argv, "?i:t:vVh:n:C:c:s:")) != -1) {
		switch (ch) {
		case 'i': {
			iters = atoi(optarg);
			break;
		}
		case 't': {
			threads = atoi(optarg);
			break;
		}
		case 'n': {
			params.opsPerTransaction = atoi(optarg);
			break;
		}
		case 'C': {
			params.cacheSize = atoi(optarg) * 1024 * 1024;
			break;
		}
		case 'v': {
			++verbose;
			break;
		}
		case 'V': {
			printf("%s\n", DbXml::dbxml_version(NULL, NULL, NULL));
			printf("%s\n", DbEnv::version(NULL, NULL, NULL));
			exit(0);
		}
		case 's': {
			string storage = optarg;
			if(storage == "DLS") {
				params.nodeStorage = false;
				params.nodeIndexes = false;
			}
			else if(storage == "DLS+") {
				params.nodeStorage = false;
				params.nodeIndexes = true;
			}
			else if(storage == "NLS") {
				params.nodeStorage = true;
				params.nodeIndexes = false;
			}
			else if(storage == "NLS+") {
				params.nodeStorage = true;
				params.nodeIndexes = true;
			}
			else {
				printf("Unknown storage type: %s\n", storage.c_str());
				usage(progName, 1);
			}
			break;
		}
		case 'h': {
			params.envDir = optarg;
			break;
		}
		case 'c': {
			params.containerName = optarg;
			break;
		}
		case '?': {
			usage(progName, 0);
			break;
		}
		default: {
			usage(progName, 1);
			break;
		}
		}
	}

	if(argc - optind > 1) {
		printf("Too many arguments\n");
		usage(progName, 1);
	}

	if(argc == optind) {
		printf("No workload specified\n");
		usage(progName, 1);
	}

	// Turn on logging if extra verbose is specified
	if(verbose > 1) {
		setLogLevel(LEVEL_ALL, true);
		setLogCategory(CATEGORY_ALL, true);
		setLogCategory(CATEGORY_NODESTORE, verbose > 2);
	}
	params.verbose = verbose > 0;

	struct perf_globals globs;
	perf_init_globals(&globs, iters, threads, true);

	// construct the workload
	DbXmlWorkload *workload = constructWorkload(argv[optind], &globs, params);
	if(workload == 0) {
		printf("Unknown workload: %s\n", argv[optind]);
		usage(progName, 1);
	}

	// run workload
	try {
		workload->run();
	} catch (XmlException &e) {
		cout << "XmlException: " << e.what() << endl;
	}

	perf_dump_timers(&globs);
	delete workload;
	perf_uninit_globals(&globs);
	return 0;
}
