/*
 * See the file LICENSE for redistribution information.
 *
 * Copyright (c) 1998,2006 Oracle.  All rights reserved.
 *
 * $Id: DbXmlWorkload.cpp,v 1.6 2006/10/30 17:46:17 bostic Exp $
 */

#include "DbXmlWorkload.hpp"
#include <dbxml/DbXml.hpp>
#include <unistd.h>

using namespace std;
using namespace DbXml;

DbXmlWorkload::~DbXmlWorkload()
{
	if(dbenv_ != 0) {
		cleanup();
	}
}

int DbXmlWorkload::setup()
{
	try {
		u_int32_t mgrflags = DBXML_ALLOW_EXTERNAL_ACCESS;
		u_int32_t cflags = DB_CREATE;
		XmlContainer::ContainerType ctype =
			(params_.nodeStorage ? XmlContainer::NodeContainer :
				XmlContainer::WholedocContainer);
		if(params_.nodeIndexes) cflags |= DBXML_INDEX_NODES;
		if (params_.envDir.length()) {
			dbenv_ = new DbEnv(0);
			u_int32_t dbflags = DB_CREATE|DB_INIT_MPOOL|DB_THREAD;
			dbenv_->set_error_stream(&cerr);
			dbenv_->set_cachesize(0, params_.cacheSize, 1);
			dbenv_->set_lk_max_lockers(50000);
			dbenv_->set_lk_max_locks(50000);
			dbenv_->set_lk_max_objects(50000);
			if(params_.opsPerTransaction) {
				dbflags |= DB_INIT_TXN|DB_INIT_LOCK|DB_RECOVER|DB_INIT_LOG;
				cflags |= DBXML_TRANSACTIONAL;
				dbenv_->set_lk_detect(DB_LOCK_DEFAULT);
			}
			dbenv_->open(params_.envDir.c_str(), dbflags, 0);
			mgrflags |= DBXML_ADOPT_DBENV;
		}
		if(dbenv_) mgr_ = new XmlManager(dbenv_, mgrflags);
		else mgr_ = new XmlManager(mgrflags);

		if(params_.containerName.length()) {
			cont_ = new XmlContainer(mgr_->openContainer(params_.containerName,
							 cflags, ctype, 0));
			cont_->addAlias("cont");
		}

		threadGlobals_.clear();
		for(int id = 0; id < pwGlobals->g_nthreads; ++id) {
			ThreadGlobal *tglob = createThreadGlobal(id);
			threadGlobals_.push_back(tglob);
		}

	} catch (XmlException &xe) {
		cout << "XmlException: " << xe.what() << endl;
		return 1;
	} catch (DbException &de) {
		cout << "DbException: " << de.what() << endl;
		return 1;
	}
	return 0;
}

int DbXmlWorkload::work(int id, int *iteration)
{
	int ret = 0;
	try {
		ThreadGlobal *tglob = threadGlobals_[id];

		// Deadlock retry loop
		bool retry = true;
		do {
			// Do we need to create a transaction?
			if(params_.opsPerTransaction &&
				(*iteration % params_.opsPerTransaction) == 0) {
				tglob->txn = mgr_->createTransaction();
				if(params_.verbose)
					cerr << "id: " << id << ", transaction created" << endl;
			}

			if(params_.verbose)
				cerr << "id: " << id << ", work iteration: " << *iteration << endl;

			try {
				ret = work(id, iteration, tglob);
				retry = false;
			} catch (XmlException &xe) {
				if(xe.getExceptionCode() != XmlException::DATABASE_ERROR ||
					xe.getDbErrno() != DB_LOCK_DEADLOCK) {
					throw xe;
				}
				// else abort and retry
				if(params_.verbose)
					cerr << "id: " << id << ", DEADLOCK! " << endl;
				if(params_.opsPerTransaction) {
					tglob->txn.abort();
					usleep(100000);
					// Force the aborted workload to be re-run
					*iteration = ((int)(*iteration / params_.opsPerTransaction)) * params_.opsPerTransaction;
				}
			}
		} while(retry);

		// Do we need to commit the transaction?
		if(params_.opsPerTransaction &&
			((*iteration + 1) % params_.opsPerTransaction) == 0) {
			tglob->txn.commit();
			tglob->txn = 0;
			if(params_.verbose)
				cerr << "id: " << id << ", transaction commited" << endl;
		}
	} catch (XmlException &xe) {
		cout << "XmlException: " << xe.what() << endl;
		throw;
		return 1;
	} catch (DbException &de) {
		cout << "DbException: " << de.what() << endl;
		return 1;
	}

	return ret;
}

int DbXmlWorkload::cleanup()
{
	for(int id = 0; id < pwGlobals->g_nthreads; ++id) {
		delete threadGlobals_[id];
	}
	threadGlobals_.clear();

	delete cont_;
	cont_ = 0;

	delete mgr_;
	mgr_ = 0;

	// dbenv is deleted by XmlManager
	dbenv_ = 0;

	return 0;
}

ThreadGlobal *DbXmlWorkload::createThreadGlobal(int id)
{
	return new ThreadGlobal();
}
