(:
 : XQuery script to create a benchmark report
 :)

declare boundary-space preserve;

declare function local:differences($stats)
{
  for $n in distinct-values($stats/entry/@name)
  let $n_entries := $stats/entry[@name = $n]

  for $d in distinct-values($stats/entry/data_type)
  let $d_entries := $n_entries[data_type = $d]

  for $q in distinct-values($stats/entry/query_index)
  let $q_entries := $d_entries[query_index = $q]

  let $entries := $q_entries[not(indexes = 0)]
  let $total := sum($entries/time)

  let $d_time  := sum($entries[storage = "document"]/time)
  let $dp_time := sum($entries[storage = "documentplus"]/time)
  let $n_time  := sum($entries[storage = "node"]/time)
  let $np_time := sum($entries[storage = "nodeplus"]/time)

  return if($entries) then <diff
    name="{ $n }"
    data_type="{ $d }"
    query_index="{ $q }"
    total_time="{ $total }"
    d_time="{ $d_time }"
    dp_time="{ $dp_time }"
    n_time="{ $n_time }"
    np_time="{ $np_time }"
    d_perc="{ round(($d_time div $total) * 100 * 100) div 100 }"
    dp_perc="{ round(($dp_time div $total) * 100 * 100) div 100 }"
    n_perc="{ round(($n_time div $total) * 100 * 100) div 100 }"
    np_perc="{ round(($np_time div $total) * 100 * 100) div 100 }"
  >{ $entries[1]/description, $entries[1]/action }</diff> else ()
};

declare function local:tableEntry($percent)
{
  let $bar_width := 400
  let $width := round($percent * ($bar_width div 100))
  return (
    <td>
      <table>
        <tr>
          <td bgcolor="#003366" style="color: white" align="right" width="{$width}">{ if($percent >= 50) then ($percent, "%") else () }</td>
          <td bgcolor="#CCCCCC" style="color: black" align="left" width="{$bar_width - $width}">{ if($percent >= 50) then () else ($percent, "%") }</td>
        </tr>
      </table>
    </td>
  )
};

declare function local:rankingTable($diffs, $type)
{
  <table border="1">
  <tr>
    <th>Position</th><th>Percent of Total Time</th><th>Time taken/s</th><th>Query</th>
  </tr>
  {
    for $d at $pos in $diffs
    return (
    <tr>
      <td>{ $pos }</td>
      {
        if($type eq "document") then
          (local:tableEntry(number($d/@d_perc)), <td>{ number($d/@d_time) }</td>)
        else if($type eq "documentplus") then
          (local:tableEntry(number($d/@dp_perc)), <td>{ number($d/@dp_time) }</td>)
        else if($type eq "node") then
          (local:tableEntry(number($d/@n_perc)), <td>{ number($d/@n_time) }</td>)
        else
          (local:tableEntry(number($d/@np_perc)), <td>{ number($d/@np_time) }</td>)
      }
      <td><a href="#query{string($d/@data_type)}{string($d/@query_index)}">{string($d/@data_type), " ", string($d/@query_index)}</a></td>
    </tr>
    )
  }
  </table>
};

let $docElem := /statistics
return <html>
  <head><title>Storage Comparison</title></head>
  <body>
    <h1>Storage Comparison</h1>
    {
      if($docElem/@version) then (
        <p>for version { string($docElem//@version) }</p>
      ) else ()
    }
    <div id="contents">
      <ul>
        <li><a href="#document">Document Storage Queries</a>
          <ul>
            <li><a href="#document_fastest">Fastest</a></li>
            <li><a href="#document_slowest">Slowest</a></li>
          </ul>
        </li>
        <li><a href="#documentplus">Document Plus Storage Queries</a>
          <ul>
            <li><a href="#documentplus_fastest">Fastest</a></li>
            <li><a href="#documentplus_slowest">Slowest</a></li>
          </ul>
        </li>
        <li><a href="#node">Node Storage Queries</a>
          <ul>
            <li><a href="#node_fastest">Fastest</a></li>
            <li><a href="#node_slowest">Slowest</a></li>
          </ul>
        </li>
        <li><a href="#nodeplus">Node+ Storage Queries</a>
          <ul>
            <li><a href="#nodeplus_fastest">Fastest</a></li>
            <li><a href="#nodeplus_slowest">Slowest</a></li>
          </ul>
        </li>
        <li><a href="#queries">Queries</a></li>
      </ul>
    </div>
    <hr/>
    {
      let $diffs := local:differences($docElem)
      return (
        <div id="document">
          <h2>Document Storage Queries</h2>
          <div id="document_fastest">
            <h3>Fastest</h3>
            <p>Queries that are fastest using Document storage, ranked by the minimum percentage difference to the other storage formats.</p>
            {
              let $d_fastest :=
                for $a in $diffs
                let $diff1 := $a/@d_perc - $a/@np_perc
                let $diff2 := $a/@d_perc - $a/@n_perc
                let $diff3 := $a/@d_perc - $a/@dp_perc
                where $diff1 < 0 and $diff2 < 0 and $diff3 < 0
                order by min((abs($diff1), abs($diff2), abs($diff3))) descending
                return $a
              return local:rankingTable($d_fastest, "document")
            }
          </div>
          <div id="document_slowest">
            <h3>Slowest</h3>
            <p>Queries that are slowest using Document storage, ranked by the minimum percentage difference to the other storage formats.</p>
            {
              let $d_slowest :=
                for $a in $diffs
                let $diff1 := $a/@d_perc - $a/@np_perc
                let $diff2 := $a/@d_perc - $a/@n_perc
                let $diff3 := $a/@d_perc - $a/@dp_perc
                where $diff1 > 0 and $diff2 > 0 and $diff3 > 0
                order by min((abs($diff1), abs($diff2), abs($diff3))) descending
                return $a
              return local:rankingTable($d_slowest, "document")
            }
          </div>
        </div>,<hr/>,
        <div id="documentplus">
          <h2>Document+ Storage Queries</h2>
          <div id="documentplus_fastest">
            <h3>Fastest</h3>
            <p>Queries that are fastest using Document+ storage, ranked by the minimum percentage difference to the other storage formats.</p>
            {
              let $dp_fastest :=
                for $a in $diffs
                let $diff1 := $a/@dp_perc - $a/@np_perc
                let $diff2 := $a/@dp_perc - $a/@n_perc
                let $diff3 := $a/@dp_perc - $a/@d_perc
                where $diff1 < 0 and $diff2 < 0 and $diff3 < 0
                order by min((abs($diff1), abs($diff2), abs($diff3))) descending
                return $a
              return local:rankingTable($dp_fastest, "documentplus")
            }
          </div>
          <div id="documentplus_slowest">
            <h3>Slowest</h3>
            <p>Queries that are slowest using Document+ storage, ranked by the minimum percentage difference to the other storage formats.</p>
            {
              let $dp_slowest :=
                for $a in $diffs
                let $diff1 := $a/@dp_perc - $a/@np_perc
                let $diff2 := $a/@dp_perc - $a/@n_perc
                let $diff3 := $a/@dp_perc - $a/@d_perc
                where $diff1 > 0 and $diff2 > 0 and $diff3 > 0
                order by min((abs($diff1), abs($diff2), abs($diff3))) descending
                return $a
              return local:rankingTable($dp_slowest, "documentplus")
            }
          </div>
        </div>,<hr/>,
        <div id="node">
          <h2>Node Storage Queries</h2>
          <div id="node_fastest">
            <h3>Fastest</h3>
            <p>Queries that are fastest using Node storage, ranked by the minimum percentage difference to the other storage formats.</p>
            {
              let $n_fastest :=
                for $a in $diffs
                let $diff1 := $a/@n_perc - $a/@np_perc
                let $diff2 := $a/@n_perc - $a/@d_perc
                let $diff3 := $a/@n_perc - $a/@dp_perc
                where $diff1 < 0 and $diff2 < 0 and $diff3 < 0
                order by min((abs($diff1), abs($diff2), abs($diff3))) descending
                return $a
              return local:rankingTable($n_fastest, "node")
            }
          </div>
          <div id="node_slowest">
            <h3>Slowest</h3>
            <p>Queries that are slowest using Node storage, ranked by the minimum percentage difference to the other storage formats.</p>
            {
              let $n_slowest :=
                for $a in $diffs
                let $diff1 := $a/@n_perc - $a/@np_perc
                let $diff2 := $a/@n_perc - $a/@d_perc
                let $diff3 := $a/@n_perc - $a/@dp_perc
                where $diff1 > 0 and $diff2 > 0 and $diff3 > 0
                order by min((abs($diff1), abs($diff2), abs($diff3))) descending
                return $a
              return local:rankingTable($n_slowest, "node")
            }
          </div>
        </div>,<hr/>,
        <div id="nodeplus">
          <h2>Node+ Storage Queries</h2>
          <div id="nodeplus_fastest">
            <h3>Fastest</h3>
            <p>Queries that are fastest using Node+ storage, ranked by the minimum percentage difference to the other storage formats.</p>
            {
              let $np_fastest :=
                for $a in $diffs
                let $diff1 := $a/@np_perc - $a/@n_perc
                let $diff2 := $a/@np_perc - $a/@d_perc
                let $diff3 := $a/@np_perc - $a/@dp_perc
                where $diff1 < 0 and $diff2 < 0 and $diff3 < 0
                order by min((abs($diff1), abs($diff2), abs($diff3))) descending
                return $a
              return local:rankingTable($np_fastest, "nodeplus")
            }
          </div>
          <div id="nodeplus_slowest">
            <h3>Slowest</h3>
            <p>Queries that are slowest using Node+ storage, ranked by the minimum percentage difference to the other storage formats.</p>
            {
              let $np_slowest :=
                for $a in $diffs
                let $diff1 := $a/@np_perc - $a/@n_perc
                let $diff2 := $a/@np_perc - $a/@d_perc
                let $diff3 := $a/@np_perc - $a/@dp_perc
                where $diff1 > 0 and $diff2 > 0 and $diff3 > 0
                order by min((abs($diff1), abs($diff2), abs($diff3))) descending
                return $a
              return local:rankingTable($np_slowest, "nodeplus")
            }
          </div>
        </div>,<hr/>,
        <div id="queries">
          <h2>Queries</h2>
          {
            for $q in $diffs
            return (<div id="query{string($q/@data_type)}{string($q/@query_index)}">
              <h4>{ string($q/@data_type), " ", string($q/@query_index) }</h4>
              <p>{string($q/description)}</p>
              <pre style="background-color: #CCCCCC;">{string($q/action)}</pre>
              <table>
                <tr><th>Category</th><th>Percent of Total Time</th><th>Time Taken/s</th></tr>
                <tr><td>Document Storage</td>{ local:tableEntry(number($q/@d_perc)) }<td>{ number($q/@d_time) }</td></tr>
                <tr><td>Document+ Storage</td>{ local:tableEntry(number($q/@dp_perc)) }<td>{ number($q/@dp_time) }</td></tr>
                <tr><td>Node Storage</td>{ local:tableEntry(number($q/@n_perc)) }<td>{ number($q/@n_time) }</td></tr>
                <tr><td>Node+ Storage</td>{ local:tableEntry(number($q/@np_perc)) }<td>{ number($q/@np_time) }</td></tr>
                <tr><td>All Types</td>{ local:tableEntry(100) }<td>{ number($q/@total_time) }</td></tr>
              </table>
            </div>, <hr/>)
          }
        </div>,<hr/>
      )
    }
  </body>
</html>
