(:
 : XQuery script to create a benchmark report
 :)

declare boundary-space preserve;

declare function local:two-digits($num as xs:integer) as xs:string
{
  let $result := fn:string($num)
  let $length := fn:string-length($result)
  return if($length > 2) then fn:substring($result, $length - 1)
    else if($length = 1) then fn:concat("0", $result)
    else if($length = 0) then "00" else $result
};

declare function local:timestamp($date as xs:dateTime) as xs:string
{
  let $day := local:two-digits(fn:day-from-dateTime($date))
  let $month := fn:month-from-dateTime($date)
  let $year := string(fn:year-from-dateTime($date))
  let $hours := local:two-digits(fn:hours-from-dateTime($date))
  let $minutes := local:two-digits(fn:minutes-from-dateTime($date))
  let $seconds := local:two-digits(xs:integer(fn:round(fn:seconds-from-dateTime($date))))
  let $monthNames := ("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
  return fn:concat(
    $day, " ",
    $monthNames[$month], " ",
    $year, " ",
    $hours, ":",
    $minutes, ":",
    $seconds
  )
};

declare function local:timestampLink($date as xs:dateTime) as xs:string
{
  let $day := local:two-digits(fn:day-from-dateTime($date))
  let $month := local:two-digits(fn:month-from-dateTime($date))
  let $year := string(fn:year-from-dateTime($date))
  let $hours := local:two-digits(fn:hours-from-dateTime($date))
  let $minutes := local:two-digits(fn:minutes-from-dateTime($date))
  let $seconds := local:two-digits(xs:integer(fn:round(fn:seconds-from-dateTime($date))))
  return fn:concat(
    $year,
    $month,
    $day,
    $hours,
    $minutes,
    $seconds
  )  
};

declare function local:tableEntry($value, $max)
{
  let $bar_width := 400
  let $width := round(($value div $max) * $bar_width)
  let $increase := round((($value - $max) div $max) * 100 * 100) div 100
  return (
    <td>
      <table>
        <tr>
          <td bgcolor="#003366" style="color: white" align="right" width="{$width}">{ if($width >= ($bar_width div 2)) then ($increase, "%") else () }</td>
          <td bgcolor="#CCCCCC" style="color: black" align="left" width="{$bar_width - $width}">{ if($width >= ($bar_width div 2)) then () else ($increase, "%") }</td>
        </tr>
      </table>
    </td>,
    <td>{$value}</td>
  )
};

declare function local:benchName($n)
{
  if($n eq "load") then "Document Loading"
  else if($n eq "query") then "Querying"
  else if($n eq "prepare") then "Preparing"
  else error(concat("Unknown benchmark name: ", $n))
};

declare function local:statsDiv($entries)
{
  if(exists($entries)) then

  let $n := string($entries[1]/@name)
  let $s := string($entries[1]/storage)
  let $indexes := $entries[1]/indexes
  return

  <div id="{ $n }_{ $s }_{ if($indexes = 0) then 'notindexed' else 'indexed' }">
    <h2>{ local:benchName($n) }<br/>
    <small><small>
      Storage: { $s }<br/>
      Indexes: { string($indexes) }<br/>
      Average Data Set Size: { round($entries[1]/size div 1024) }K
    </small></small></h2>
    <table>
      <tr><th>Version</th><th>Timestamp</th><th>Percent Difference from Maximum</th><th>Operations / Second</th><th>Note</th></tr>
      {
      let $max := max($entries/per_second/average)
      for $stat in $entries/..
      for $entry in $entries[.. is $stat]
      order by xs:dateTime($stat/@timestamp)
      return <tr>
        <td>{ string($stat/@version) }</td>
        <td><small><a href="statistics{ local:timestampLink($stat/@timestamp) }.html#{ $n }">{ local:timestamp($stat/@timestamp) }</a></small></td>
        { local:tableEntry($entry/per_second/average, $max) }
        <td><small>{ $stat/note/node() }</small></td>
      </tr>
      }
    </table>
    <hr/>
  </div>

  else ()
};

declare function local:statsLink($entries)
{
  if(exists($entries)) then

    let $n := string($entries[1]/@name)
    let $s := string($entries[1]/storage)
    let $indexes := $entries[1]/indexes
    return <li><a href="#{ $n }_{ $s }_{ if($indexes = 0) then 'notindexed' else 'indexed' }">{ if($indexes = 0) then 'Not Indexed' else 'Indexed' }</a></li>
  else ()
};


let $docElem := /summary
let $stats := $docElem/statistics
return <html>
  <head><title>Benchmark Summary</title></head>
  <body>
    <h1>Benchmark Summary</h1>
    <div id="contents">
      <ul>
      {
        for $n in distinct-values($stats/entry/@name)
        return <li>{ local:benchName($n) }
          <ul>
          {
            for $s in distinct-values($stats/entry/storage)
            return <li>Storage: { $s }
              <ul>
              {
                let $indexedEntries := $stats/entry[@name = $n][storage = $s][not(indexes = 0)]
                let $notIndexedEntries := $stats/entry[@name = $n][storage = $s][indexes = 0]
                return (local:statsLink($indexedEntries), local:statsLink($notIndexedEntries))
              }
              </ul>
            </li>
          }
          </ul>
        </li>
      }
      </ul>
    </div>
    <hr/>
    {
      for $n in distinct-values($stats/entry/@name)
      for $s in distinct-values($stats/entry/storage)
      let $indexedEntries := $stats/entry[@name = $n][storage = $s][not(indexes = 0)]
      let $notIndexedEntries := $stats/entry[@name = $n][storage = $s][indexes = 0]
      return (local:statsDiv($indexedEntries), local:statsDiv($notIndexedEntries))
    }
  </body>
</html>
