(:
 : XQuery script to create a benchmark report
 :)

declare boundary-space preserve;

declare function local:two-digits($num as xs:integer) as xs:string
{
  let $result := fn:string($num)
  let $length := fn:string-length($result)
  return if($length > 2) then fn:substring($result, $length - 1)
    else if($length = 1) then fn:concat("0", $result)
    else if($length = 0) then "00" else $result
};

declare function local:timestamp($date as xs:dateTime) as xs:string
{
  let $day := local:two-digits(fn:day-from-dateTime($date))
  let $month := fn:month-from-dateTime($date)
  let $year := string(fn:year-from-dateTime($date))
  let $hours := local:two-digits(fn:hours-from-dateTime($date))
  let $minutes := local:two-digits(fn:minutes-from-dateTime($date))
  let $seconds := local:two-digits(xs:integer(fn:round(fn:seconds-from-dateTime($date))))
  let $monthNames := ("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
  return fn:concat(
    $day, " ",
    $monthNames[$month], " ",
    $year, " ",
    $hours, ":",
    $minutes, ":",
    $seconds
  )
};

declare function local:tableEntry($name, $value, $total)
{
  let $bar_width := 400
  let $percent := round(($value div $total) * 100 * 100) div 100
  let $width := round($percent * ($bar_width div 100))
  return (
    <td>{$name}</td>,
    <td>
      <table>
        <tr>
          <td bgcolor="#003366" style="color: white" align="right" width="{$width}">{ if($percent >= 50) then ($percent, "%") else () }</td>
          <td bgcolor="#CCCCCC" style="color: black" align="left" width="{$bar_width - $width}">{ if($percent >= 50) then () else ($percent, "%") }</td>
        </tr>
      </table>
    </td>,
    <td>{$value}</td>
  )
};

declare function local:categoryTable($entries)
{
  let $totalTime := sum($entries/time)
  return
  <table>
    <tr>
      <th>Category</th>
      <th>Percent of Total Time</th>
      <th>Time Taken/s</th>
    </tr>
    <tr>{ local:tableEntry("Document Storage", sum($entries[storage = "document"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Document+ Storage", sum($entries[storage = "documentplus"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Node Storage", sum($entries[storage = "node"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Node+ Storage", sum($entries[storage = "nodeplus"]/time), $totalTime) }</tr>
    {
      if(empty($entries/query_index)) then (
        <tr>{ local:tableEntry("Indexed", sum($entries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr>{ local:tableEntry("Not Indexed", sum($entries[indexes = 0]/time), $totalTime) }</tr>
      ) else ()
    }
    <tr>{ local:tableEntry("Text Centric", sum($entries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Data Centric", sum($entries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Multiple Document", sum($entries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Single Document", sum($entries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("All Types", $totalTime, $totalTime) }</tr>
  </table>
};

declare function local:storageComparisonTable($entries)
{
  let $totalTime := sum($entries/time)
  let $documentEntries := $entries[storage = "document"]
  let $documentplusEntries := $entries[storage = "documentplus"]
  let $nodeEntries := $entries[storage = "node"]
  let $nodeplusEntries := $entries[storage = "nodeplus"]
  return
  <table>
    <tr>
      <th style="width: 8em">Storage</th>
      <th>Category</th>
      <th>Percent of Total Time</th>
      <th>Time Taken/s</th>
    </tr>
    {
      if(empty($entries/query_index)) then (
        <tr><td>Document</td>{ local:tableEntry("Indexed", sum($documentEntries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr><td>Document+</td>{ local:tableEntry("Indexed", sum($documentplusEntries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr><td>Node</td>{ local:tableEntry("Indexed", sum($nodeplusEntries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr><td>Node+</td>{ local:tableEntry("Indexed", sum($nodeEntries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr><td>Document</td>{ local:tableEntry("Not Indexed", sum($documentEntries[indexes = 0]/time), $totalTime) }</tr>,
        <tr><td>Document+</td>{ local:tableEntry("Not Indexed", sum($documentplusEntries[indexes = 0]/time), $totalTime) }</tr>,
        <tr><td>Node</td>{ local:tableEntry("Not Indexed", sum($nodeEntries[indexes = 0]/time), $totalTime) }</tr>,
        <tr><td>Node+</td>{ local:tableEntry("Not Indexed", sum($nodeplusEntries[indexes = 0]/time), $totalTime) }</tr>
      ) else ()
    }
    <tr><td>Document</td>{ local:tableEntry("Text Centric", sum($documentEntries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr><td>Document+</td>{ local:tableEntry("Text Centric", sum($documentplusEntries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr><td>Node</td>{ local:tableEntry("Text Centric", sum($nodeEntries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr><td>Node+</td>{ local:tableEntry("Text Centric", sum($nodeplusEntries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr><td>Document</td>{ local:tableEntry("Data Centric", sum($documentEntries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr><td>Document+</td>{ local:tableEntry("Data Centric", sum($documentplusEntries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr><td>Node</td>{ local:tableEntry("Data Centric", sum($nodeEntries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr><td>Node+</td>{ local:tableEntry("Data Centric", sum($nodeplusEntries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr><td>Document</td>{ local:tableEntry("Multiple Document", sum($documentEntries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr><td>Document+</td>{ local:tableEntry("Multiple Document", sum($documentplusEntries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr><td>Node</td>{ local:tableEntry("Multiple Document", sum($nodeEntries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr><td>Node+</td>{ local:tableEntry("Multiple Document", sum($nodeplusEntries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr><td>Document</td>{ local:tableEntry("Single Document", sum($documentEntries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr><td>Document+</td>{ local:tableEntry("Single Document", sum($documentplusEntries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr><td>Node</td>{ local:tableEntry("Single Document", sum($nodeEntries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr><td>Node+</td>{ local:tableEntry("Single Document", sum($nodeplusEntries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr><td/>{ local:tableEntry("All Types", $totalTime, $totalTime) }</tr>
  </table>
};

declare function local:indexComparisonTable($entries)
{
  let $totalTime := sum($entries/time)
  let $indexedEntries := $entries[not(indexes = 0)]
  let $notIndexedEntries := $entries[indexes = 0]
  return
  <table>
    <tr>
      <th style="width: 8em">Storage</th>
      <th>Category</th>
      <th>Percent of Total Time</th>
      <th>Time Taken/s</th>
    </tr>
    <tr><td>Indexed</td>{ local:tableEntry("Text Centric", sum($indexedEntries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr><td>Not Indexed</td>{ local:tableEntry("Text Centric", sum($notIndexedEntries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr><td>Indexed</td>{ local:tableEntry("Data Centric", sum($indexedEntries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr><td>Not Indexed</td>{ local:tableEntry("Data Centric", sum($notIndexedEntries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr><td>Indexed</td>{ local:tableEntry("Multiple Document", sum($indexedEntries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr><td>Not Indexed</td>{ local:tableEntry("Multiple Document", sum($notIndexedEntries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr><td>Indexed</td>{ local:tableEntry("Single Document", sum($indexedEntries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr><td>Not Indexed</td>{ local:tableEntry("Single Document", sum($notIndexedEntries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr><td/>{ local:tableEntry("All Types", $totalTime, $totalTime) }</tr>
  </table>
};

declare function local:documentTypeComparisonTable($entries)
{
  let $totalTime := sum($entries/time)
  let $textEntries := $entries[starts-with(data_type, "TC")]
  let $dataEntries := $entries[starts-with(data_type, "DC")]
  return
  <table>
    <tr>
      <th>Document Type</th>
      <th>Category</th>
      <th>Percent of Total Time</th>
      <th>Time Taken/s</th>
    </tr>
    <tr><td>Text Centric</td>{ local:tableEntry("Document Storage", sum($textEntries[storage = "document"]/time), $totalTime) }</tr>
    <tr><td>Data Centric</td>{ local:tableEntry("Document Storage", sum($dataEntries[storage = "document"]/time), $totalTime) }</tr>
    <tr><td>Text Centric</td>{ local:tableEntry("Document+ Storage", sum($textEntries[storage = "documentplus"]/time), $totalTime) }</tr>
    <tr><td>Data Centric</td>{ local:tableEntry("Document+ Storage", sum($dataEntries[storage = "documentplus"]/time), $totalTime) }</tr>
    <tr><td>Text Centric</td>{ local:tableEntry("Node Storage", sum($textEntries[storage = "node"]/time), $totalTime) }</tr>
    <tr><td>Data Centric</td>{ local:tableEntry("Node Storage", sum($dataEntries[storage = "node"]/time), $totalTime) }</tr>
    <tr><td>Text Centric</td>{ local:tableEntry("Node+ Storage", sum($textEntries[storage = "nodeplus"]/time), $totalTime) }</tr>
    <tr><td>Data Centric</td>{ local:tableEntry("Node+ Storage", sum($dataEntries[storage = "nodeplus"]/time), $totalTime) }</tr>
    {
      if(empty($entries/query_index)) then (
        <tr><td>Text Centric</td>{ local:tableEntry("Indexed", sum($textEntries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr><td>Data Centric</td>{ local:tableEntry("Indexed", sum($dataEntries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr><td>Text Centric</td>{ local:tableEntry("Not Indexed", sum($textEntries[indexes = 0]/time), $totalTime) }</tr>,
        <tr><td>Data Centric</td>{ local:tableEntry("Not Indexed", sum($dataEntries[indexes = 0]/time), $totalTime) }</tr>
      ) else ()
    }
    <tr><td>Text Centric</td>{ local:tableEntry("Multiple Document", sum($textEntries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr><td>Data Centric</td>{ local:tableEntry("Multiple Document", sum($dataEntries[ends-with(data_type, "MD")]/time), $totalTime) }</tr>
    <tr><td>Text Centric</td>{ local:tableEntry("Single Document", sum($textEntries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr><td>Data Centric</td>{ local:tableEntry("Single Document", sum($dataEntries[ends-with(data_type, "SD")]/time), $totalTime) }</tr>
    <tr><td/>{ local:tableEntry("All Types", $totalTime, $totalTime) }</tr>
  </table>
};

declare function local:numDocComparisonTable($entries)
{
  let $totalTime := sum($entries/time)
  let $multipleEntries := $entries[ends-with(data_type, "MD")]
  let $singleEntries := $entries[ends-with(data_type, "SD")]
  return
  <table>
    <tr>
      <th>Single / Multiple Documents</th>
      <th>Category</th>
      <th>Percent of Total Time</th>
      <th>Time Taken/s</th>
    </tr>
    <tr><td>Multiple</td>{ local:tableEntry("Document Storage", sum($multipleEntries[storage = "document"]/time), $totalTime) }</tr>
    <tr><td>Single</td>{ local:tableEntry("Document Storage", sum($singleEntries[storage = "document"]/time), $totalTime) }</tr>
    <tr><td>Multiple</td>{ local:tableEntry("Document+ Storage", sum($multipleEntries[storage = "documentplus"]/time), $totalTime) }</tr>
    <tr><td>Single</td>{ local:tableEntry("Document+ Storage", sum($singleEntries[storage = "documentplus"]/time), $totalTime) }</tr>
    <tr><td>Multiple</td>{ local:tableEntry("Node Storage", sum($multipleEntries[storage = "node"]/time), $totalTime) }</tr>
    <tr><td>Single</td>{ local:tableEntry("Node Storage", sum($singleEntries[storage = "node"]/time), $totalTime) }</tr>
    <tr><td>Multiple</td>{ local:tableEntry("Node+ Storage", sum($multipleEntries[storage = "nodeplus"]/time), $totalTime) }</tr>
    <tr><td>Single</td>{ local:tableEntry("Node+ Storage", sum($singleEntries[storage = "nodeplus"]/time), $totalTime) }</tr>
    {
      if(empty($entries/query_index)) then (
        <tr><td>Multiple</td>{ local:tableEntry("Indexed", sum($multipleEntries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr><td>Single</td>{ local:tableEntry("Indexed", sum($singleEntries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr><td>Multiple</td>{ local:tableEntry("Not Indexed", sum($multipleEntries[indexes = 0]/time), $totalTime) }</tr>,
        <tr><td>Single</td>{ local:tableEntry("Not Indexed", sum($singleEntries[indexes = 0]/time), $totalTime) }</tr>
      ) else ()
    }
    <tr><td>Multiple</td>{ local:tableEntry("Text Centric", sum($multipleEntries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr><td>Single</td>{ local:tableEntry("Text Centric", sum($singleEntries[starts-with(data_type, "TC")]/time), $totalTime) }</tr>
    <tr><td>Multiple</td>{ local:tableEntry("Data Centric", sum($multipleEntries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr><td>Single</td>{ local:tableEntry("Data Centric", sum($singleEntries[starts-with(data_type, "DC")]/time), $totalTime) }</tr>
    <tr><td/>{ local:tableEntry("All Types", $totalTime, $totalTime) }</tr>
  </table>
};

declare function local:storageTable($entries)
{
  let $totalTime := sum($entries/time)
  return
  <table>
    <tr>
      <th>Category</th>
      <th>Percent of Total Time</th>
      <th>Time Taken/s</th>
    </tr>
    <tr>{ local:tableEntry("Document Storage", sum($entries[storage = "document"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Document+ Storage", sum($entries[storage = "documentplus"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Node Storage", sum($entries[storage = "node"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Node+ Storage", sum($entries[storage = "nodeplus"]/time), $totalTime) }</tr>
    {
      if(empty($entries/query_index)) then (
        <tr>{ local:tableEntry("Indexed", sum($entries[not(indexes = 0)]/time), $totalTime) }</tr>,
        <tr>{ local:tableEntry("Not Indexed", sum($entries[indexes = 0]/time), $totalTime) }</tr>
      ) else ()
    }
    <tr>{ local:tableEntry("All Types", $totalTime, $totalTime) }</tr>
  </table>
};

declare function local:benchName($n)
{
  if($n eq "load") then "Document Loading"
  else if($n eq "query") then "Querying"
  else if($n eq "prepare") then "Preparing"
  else error(concat("Unknown benchmark name: ", $n))
};



let $docElem := /statistics
let $totalTime := sum($docElem/entry/time)
let $note := $docElem/note
return <html>
  <head><title>{concat("Benchmark Statistics - ", local:timestamp($docElem/@timestamp))}</title></head>
  <body>
    <h1>Full Benchmark Report</h1>
    <h3>{local:timestamp($docElem/@timestamp)}</h3>
    <div id="contents">
      <ul>
        <li><a href="#load">Document Loading Statistics</a>
          <ul>
            <li><a href="#load_category_totals">Category Totals</a></li>
            <li><a href="#load_storage_comparison">Storage Method Comparison</a></li>
            <li><a href="#load_doctype_comparison">Document Type Comparison</a></li>
            <li><a href="#load_numdoc_comparison">Number of Documents Comparison</a></li>
            <li><a href="#load_index_comparison">Index Scheme Comparison</a></li>
          </ul>
        </li>
        <li><a href="#query">Preparing Statistics</a>
          <ul>
            <li><a href="#prepare_category_totals">Category Totals</a></li>
            <li><a href="#prepare_storage_comparison">Storage Method Comparison</a></li>
            <li><a href="#prepare_doctype_comparison">Document Type Comparison</a></li>
            <li><a href="#prepare_numdoc_comparison">Number of Documents Comparison</a></li>
          </ul>
        </li>
        <li><a href="#query">Querying Statistics</a>
          <ul>
            <li><a href="#query_category_totals">Category Totals</a></li>
            <li><a href="#query_storage_comparison">Storage Method Comparison</a></li>
            <li><a href="#query_doctype_comparison">Document Type Comparison</a></li>
            <li><a href="#query_numdoc_comparison">Number of Documents Comparison</a></li>
            <li><a href="#query_query_totals">Query Totals</a></li>
            <li><a href="#query_queries">Queries</a></li>
          </ul>
        </li>
      </ul>
    </div>
    {
      if(exists($note)) then <div id="note">
        <p>{$note/node()}</p>
      </div>
      else ()
    }
    <hr/>
    {
    for $n in distinct-values($docElem/entry/@name)
    let $entries := $docElem/entry[@name = $n]
    return <div id="{ $n }">
    <h1>{ local:benchName($n) } Statistics</h1>
      <ul>
        <li><a href="#{ $n }_category_totals">Category Totals</a></li>
        <li><a href="#{ $n }_storage_comparison">Storage Method Comparison</a></li>
        <li><a href="#{ $n }_doctype_comparison">Document Type Comparison</a></li>
        <li><a href="#{ $n }_numdoc_comparison">Number of Documents Comparison</a></li>
        {
        if($n eq "query") then (
        <li><a href="#{ $n }_query_totals">Query Totals</a></li>,
        <li><a href="#{ $n }_queries">Queries</a></li>
        )
        else (
        <li><a href="#{ $n }_index_comparison">Index Scheme Comparison</a></li>
        )
        }
      </ul>

    <div id="{ $n }_category_totals">
      <h2>{ local:benchName($n) } Category Totals</h2>
      {
        local:categoryTable($entries)
      }
    </div>
    <hr/>
    <div id="{ $n }_storage_comparison">
      <h2>{ local:benchName($n) } Storage Method Comparison</h2>
      {
        local:storageComparisonTable($entries)
      }
    </div>
    <hr/>
    <div id="{ $n }_doctype_comparison">
      <h2>{ local:benchName($n) } Document Type Comparison</h2>
      {
        local:documentTypeComparisonTable($entries)
      }
    </div>
    <hr/>
    <div id="{ $n }_numdoc_comparison">
      <h2>{ local:benchName($n) } Number of Documents Comparison</h2>
      {
        local:numDocComparisonTable($entries)
      }
    </div>
    <hr/>
    { if($n eq "query") then (
    <div id="{ $n }_query_totals">
      <h2>Query Totals</h2>
      <table>
      <tr><th>Query</th><th>Percent of Total Time</th><th>Time Taken/s</th></tr>
      {
        for $datatype in distinct-values($entries/data_type)
        let $typeEntries := $entries[data_type = $datatype]
        for $query in distinct-values($typeEntries/query_index)
        let $queryEntries := $typeEntries[query_index = $query]
        let $total := sum($queryEntries/time)
        order by $total descending
        return <tr>{ local:tableEntry(<a href="#query{$datatype}{$query}">{$datatype, " ", $query}</a>, $total, $totalTime) }</tr>
      }
      </table>
    </div>,
    <hr/>,
    <div id="{ $n }_queries">
      <h2>Queries</h2>
      {
        for $datatype in distinct-values($entries/data_type)
        let $typeEntries := $entries[data_type = $datatype]
        return <div>
          <h3>Type: {$datatype}</h3>
          {
            for $query in distinct-values($typeEntries/query_index)
            let $queryEntries := $typeEntries[query_index = $query]
            let $queryEntry := $queryEntries[1]
            return (<div id="query{$datatype}{$query}">
              <h4>{$datatype, " ", $query}</h4>
              <p>{string($queryEntry/description)}</p>
              <pre style="background-color: #CCCCCC;">{string($queryEntry/action)}</pre>
              {
                local:storageTable($queryEntries)
              }
            </div>, <hr/>)
          }
        </div>
      }
    </div>
    ) else (
    <div id="{ $n }_index_comparison">
      <h2>{ local:benchName($n) } Index Scheme Comparison</h2>
      {
        local:indexComparisonTable($entries)
      }
    </div>,
    <hr/>
    ) }
    </div>
    }
  </body>
</html>