(:
 : XQuery script to create a detailed benchmark report
 :)

declare boundary-space preserve;

declare function local:size($size)
{
  if($size eq "miniscule") then (20 * 1024)
  else if($size eq "tiny") then (1024 * 1024)
  else if($size eq "small") then (10 * 1024 * 1024)
  else error(concat("Unknown size: ", $size))
};

declare function local:two-digits($num as xs:integer) as xs:string
{
  let $result := fn:string($num)
  let $length := fn:string-length($result)
  return if($length > 2) then fn:substring($result, $length - 1)
    else if($length = 1) then fn:concat("0", $result)
    else if($length = 0) then "00" else $result
};

declare function local:timestamp($date as xs:dateTime) as xs:string
{
  let $day := local:two-digits(fn:day-from-dateTime($date))
  let $month := fn:month-from-dateTime($date)
  let $year := string(fn:year-from-dateTime($date))
  let $hours := local:two-digits(fn:hours-from-dateTime($date))
  let $minutes := local:two-digits(fn:minutes-from-dateTime($date))
  let $seconds := local:two-digits(xs:integer(fn:round(fn:seconds-from-dateTime($date))))
  let $monthNames := ("Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec")
  return fn:concat(
    $day, " ",
    $monthNames[$month], " ",
    $year, " ",
    $hours, ":",
    $minutes, ":",
    $seconds
  )
};

declare function local:tableEntry($name, $value, $total)
{
  let $bar_width := 400
  let $percent := round(($value div $total) * 100 * 100) div 100
  let $width := round($percent * ($bar_width div 100))
  return (
    <td>{$name}</td>,
    <td>
      <table>
        <tr>
          <td bgcolor="#003366" style="color: white" align="right" width="{$width}">{ if($percent >= 50) then ($percent, "%") else () }</td>
          <td bgcolor="#CCCCCC" style="color: black" align="left" width="{$bar_width - $width}">{ if($percent >= 50) then () else ($percent, "%") }</td>
        </tr>
      </table>
    </td>,
    <td>{$value}</td>
  )
};

declare function local:benchName($n)
{
  if($n eq "load") then "Document Loading"
  else if($n eq "query") then "Querying"
  else if($n eq "prepare") then "Preparing"
  else error(concat("Unknown benchmark name: ", $n))
};

declare function local:statsDiv($entries)
{
  if(exists($entries)) then

  let $n := string($entries[1]/@name)
  let $sz := string($entries[1]/size)
  let $d := string($entries[1]/data_type)
  let $s := string($entries[1]/storage)
  let $indexes := $entries[1]/indexes
  return

  <div id="{ $n }_{ $sz }_{ $d }_{ $s }_{ if($indexes = 0) then 'notindexed' else 'indexed' }">
    <h2>{ local:benchName($n) }<br/>
    <small><small>
      Data Set Size: { round(local:size($sz) div 1024) }K ({ $sz })<br/>
      Data Type: { $d }<br/>
      Storage: { $s }<br/>
      Indexes: { string($indexes) }<br/>
    </small></small></h2>
    Average time taken: { round((sum($entries/time) div sum($entries/count)) * 10000) div 10000 }<br/>
    Average operations / second: { round((sum($entries/count) div (sum($entries/time))) * 1000) div 1000 }
    { if($n eq "query") then (
      <h3>Query Totals</h3>,
      <table>
      <tr><th>Query</th><th>Percent of Total Time</th><th>Time Taken/s</th><th>Operations / second</th></tr>
      {
        let $total := sum($entries/time)
        for $query in distinct-values($entries/query_index)
        let $queryEntries := $entries[query_index = $query]
        let $time := sum($queryEntries/time)
(:         order by number($query) ascending :)
        order by $time descending
        return <tr>
          { local:tableEntry(<a href="#query{$d}{$query}">{$d, " ", $query}</a>, $time, $total) }
          <td>{ round((sum($queryEntries/count) div ($time)) * 1000) div 1000 }</td>
        </tr>
      }
      </table>) else ()
    }
    <hr/>
  </div>

  else ()
};



let $docElem := /statistics
let $totalTime := sum($docElem/entry/time)
let $note := $docElem/note
return <html>
  <head><title>{concat("Benchmark Statistics - ", local:timestamp($docElem/@timestamp))}</title></head>
  <body>
    <h1>Detailed Benchmark Report</h1>
    {
      if(exists($docElem/@version)) then 
      <h3>Version: { string($docElem/@version) }</h3>
      else ()
    }
    <h3>{local:timestamp($docElem/@timestamp)}</h3>
    <div id="contents">
      <ul>
        <li><a href="#load">Document Loading Statistics</a>
        </li>
        <li><a href="#query">Querying Statistics</a>
        </li>
      </ul>
    </div>
    {
      if(exists($note)) then <div id="note">
        <p>{$note/node()}</p>
      </div>
      else ()
    }
    <hr/>
    {
    for $n in distinct-values($docElem/entry/@name)
    let $entries := $docElem/entry[@name = $n]
    return <div id="{ $n }">
    <h1>{ local:benchName($n) } Statistics</h1>
    <div id="{ $n }_contents">
    <ul>
    {
      for $size in distinct-values($entries/size)
      let $sizeEntries := $entries[size = $size]
      return <li>Data Set Size: { round(local:size($size) div 1024) }K ({ $size })
      <ul>
      {
        for $data in distinct-values($sizeEntries/data_type)
        let $dataEntries := $sizeEntries[data_type = $data]
        return <li>Data Type: { $data }
        <ul>
        {
          for $storage in distinct-values($dataEntries/storage)
          let $storageEntries := $dataEntries[storage = $storage]
          return (
            if($n eq "query") then () else (<li><a href="#{ $n }_{ $size }_{ $data }_{ $storage }_notindexed">Storage: { $storage }, Not Indexed</a></li>),
            <li><a href="#{ $n }_{ $size }_{ $data }_{ $storage }_indexed">Storage: { $storage }, Indexed</a></li>
          )
        }
        </ul>
        </li>
      }
      </ul>
      </li>
    }
    </ul>
    </div>
    {
      for $size in distinct-values($entries/size)
      let $sizeEntries := $entries[size = $size]

      for $data in distinct-values($sizeEntries/data_type)
      let $dataEntries := $sizeEntries[data_type = $data]

      for $storage in distinct-values($dataEntries/storage)
      let $storageEntries := $dataEntries[storage = $storage]
      return (
        local:statsDiv($storageEntries[indexes = 0]),
        local:statsDiv($storageEntries[not(indexes = 0)])
      )
    }
    { if($n eq "query") then (
    <div id="{ $n }_queries">
      <h2>Queries</h2>
      {
        for $datatype in distinct-values($entries/data_type)
        let $typeEntries := $entries[data_type = $datatype]
        return <div>
          <h3>Type: {$datatype}</h3>
          {
            for $query in distinct-values($typeEntries/query_index)
            let $queryEntries := $typeEntries[query_index = $query]
            let $queryEntry := $queryEntries[1]
            return (<div id="query{$datatype}{$query}">
              <h4>{$datatype, " ", $query}</h4>
              <p>{string($queryEntry/description)}</p>
              <pre style="background-color: #CCCCCC;">{string($queryEntry/action)}</pre>
            </div>, <hr/>)
          }
        </div>
      }
    </div>,
    <hr/>) else ()
    }
    </div>
    }
  </body>
</html>