(:
 : XQuery script to create a benchmark report
 :)

declare boundary-space preserve;

declare function local:tableEntry($name, $value, $total)
{
  let $bar_width := 400
  let $percent := round(($value div $total) * 100 * 100) div 100
  let $width := round($percent * ($bar_width div 100))
  return (
    <td>{$name}</td>,
    <td>
      <table>
        <tr>
          <td bgcolor="#003366" style="color: white" align="right" width="{$width}">{ if($percent >= 50) then ($percent, "%") else () }</td>
          <td bgcolor="#CCCCCC" style="color: black" align="left" width="{$bar_width - $width}">{ if($percent >= 50) then () else ($percent, "%") }</td>
        </tr>
      </table>
    </td>,
    <td>{$value}</td>
  )
};

declare function local:storageTable($entries1, $entries2)
{
  let $time1 := sum($entries1/time)
  let $time2 := sum($entries2/time)
  let $totalTime := $time1 + $time2
  return
  <table>
    <tr>
      <th>Category</th>
      <th>Percent of Total Time</th>
      <th>Time Taken/s</th>
    </tr>
    <tr>{ local:tableEntry("Document Storage Before", sum($entries1[storage = "document"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Document Storage After", sum($entries2[storage = "document"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Document+ Storage Before", sum($entries1[storage = "documentplus"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Document+ Storage After", sum($entries2[storage = "documentplus"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Node Storage Before", sum($entries1[storage = "node"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Node Storage After", sum($entries2[storage = "node"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Node+ Storage Before", sum($entries1[storage = "nodeplus"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("Node+ Storage After", sum($entries2[storage = "nodeplus"]/time), $totalTime) }</tr>
    <tr>{ local:tableEntry("All Types Before", $time1, $totalTime) }</tr>
    <tr>{ local:tableEntry("All Types After", $time2, $totalTime) }</tr>
  </table>
};

declare function local:differences($stats1, $stats2)
{
  for $n in distinct-values($stats1/entry/@name)
  let $n1 := $stats1/entry[@name = $n]
  let $n2 := $stats2/entry[@name = $n]

  for $s in distinct-values($stats1/entry/storage)
  let $s1 := $n1[storage = $s]
  let $s2 := $n2[storage = $s]

  for $d in distinct-values($stats1/entry/data_type)
  let $d1 := $s1[data_type = $d]
  let $d2 := $s2[data_type = $d]

  for $q in distinct-values($stats1/entry/query_index)
  let $q1 := $d1[query_index = $q]
  let $q2 := $d2[query_index = $q]

  let $entries1 := $q1[not(indexes = 0)]
  let $entries2 := $q2[not(indexes = 0)]
  let $diff := sum($entries1/time) - sum($entries2/time)
  order by abs($diff) descending

  return if($entries1) then <diff
    name="{ $n }"
    storage="{ $s }"
    data_type="{ $d }"
    query_index="{ $q }"
    diff="{ $diff }"
    original="{ sum($entries1/time) }"
    new="{ sum($entries2/time) }"/> else ()
};

declare function local:rankingTable($diffs)
{
  <table border="1">
  <tr>
    <th>Position</th><th>Absolute Difference/s</th><th>Percent Difference</th>
    <th>Original Time/s</th><th>New Time/s</th><th>Storage</th><th>Query</th>
  </tr>
  {
    for $d at $pos in $diffs
    return <tr>
      <td>{ $pos }</td>
      <td>{ round($d/@diff * 1000) div 1000 }</td>
      <td>{ round((abs(number($d/@diff)) div number($d/@original)) * 100 * 100) div 100 }</td>
      <td>{ string($d/@original) }</td>
      <td>{ string($d/@new) }</td>
      <td>{ string($d/@storage) }</td>
      <td><a href="#query{string($d/@data_type)}{string($d/@query_index)}">{string($d/@data_type), " ", string($d/@query_index)}</a></td>
    </tr>
  }
  </table>
};

let $docElem := /compare
let $stats1 := $docElem/statistics[1]
let $stats2 := $docElem/statistics[2]
return <html>
  <head><title>Benchmark Comparison</title></head>
  <body>
    <h1>Benchmark Comparison</h1>
    {
      if($stats1/@version and $stats2/@version) then (
        <p>Comparing version { string($stats1/@version) } to version { string($stats2/@version) }</p>
      ) else ()
    }
    <div id="contents">
      <ul>
        <li><a href="#faster">Top 20 Faster Queries</a></li>
        <li><a href="#slower">Top 20 Slower Queries</a></li>
        <li><a href="#queries">Queries</a></li>
      </ul>
    </div>
    <hr/>
    {
      let $diffs := local:differences($stats1, $stats2)
      return (
        <div id="faster">
          <h2>Top 20 Faster Queries</h2>
          { local:rankingTable(subsequence($diffs[@diff > 0], 0, 20)) }
        </div>,<hr/>,
        <div id="slower">
          <h2>Top 20 Slower Queries</h2>
          { local:rankingTable(subsequence($diffs[@diff < 0], 0, 20)) }
        </div>
      )
    }
    <hr/>
    <div id="queries">
      <h2>Queries</h2>
      {
        for $datatype in distinct-values($stats1/entry/data_type)
        let $typeEntries1 := $stats1/entry[data_type = $datatype]
        let $typeEntries2 := $stats2/entry[data_type = $datatype]
        return <div>
          <h3>Type: {$datatype}</h3>
          {
            for $query in distinct-values($typeEntries1/query_index)
            let $queryEntries1 := $typeEntries1[query_index = $query]
            let $queryEntries2 := $typeEntries2[query_index = $query]
            let $queryEntry := $queryEntries1[1]
            return (<div id="query{$datatype}{$query}">
              <h4>{$datatype, " ", $query}</h4>
              <p>{string($queryEntry/description)}</p>
              <pre style="background-color: #CCCCCC;">{string($queryEntry/action)}</pre>
              {
                local:storageTable($queryEntries1, $queryEntries2)
              }
            </div>, <hr/>)
          }
        </div>
      }
    </div>
  </body>
</html>
