//
// See the file LICENSE for redistribution information.
//
// Copyright (c) 2004-2005
//	Sleepycat Software.  All rights reserved.
// Copyright (c) 2004-2005
//	Progress Software Corporation.  All rights reserved.
//
// $Id: XQDOMConstructor.cpp,v 1.24 2005/04/05 16:44:56 bostic Exp $
//

//////////////////////////////////////////////////////////////////////
// XQDOMConstructor.cpp: implementation of the XQDOMConstructor class.
//////////////////////////////////////////////////////////////////////

#include "xquery/XQEngine.hpp"
#include "xquery/dataItem/XQDOMConstructor.hpp"
#include "xquery/XQContext.hpp"
#include <xquery/context/XQDynamicContextImpl.hpp>
#include "xquery/dataItem/XQDebugHook.hpp"
#include "xquery/dom-extensions/XQScopedNamespace.hpp"
#include "xquery/XQDebugCallback.hpp"
#include <pathan/exceptions/DataItemException.hpp>
#include <pathan/exceptions/NamespaceLookupException.hpp>
#include <pathan/exceptions/StaticAnalysisException.hpp>
#include <pathan/SequenceType.hpp>
#include <pathan/XPath2Utils.hpp>
#include <pathan/XPath2NSUtils.hpp>
#include <pathan/Node.hpp>
#include <pathan/dataItem/StaticResolutionContext.hpp>
#include <pathan/dataItem/DataItemSequence.hpp>
#include <pathan/dataItem/DataItemLiteral.hpp>
#include <pathan/internal/factory/DatatypeFactory.hpp>

#include <xercesc/dom/DOM.hpp>
#include <xercesc/validators/schema/SchemaSymbols.hpp>
#include <xercesc/framework/XMLBuffer.hpp>
#include <xercesc/util/XMLUniDefs.hpp>
#include <assert.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

XQDOMConstructor::XQDOMConstructor(short nodeType, DataItem* name, VectorOfDataItems* attrList, VectorOfDataItems* children, XPath2MemoryManager* expr) :
  DataItemImpl(expr),
  m_name(name),
  m_attrList(attrList),
  m_children(children)
{
  m_nodeType=nodeType;
  setType((DataItem::whichType)XQContext::DOM_CONSTRUCTOR);
}

Sequence XQDOMConstructor::collapseTreeInternal(DynamicContext *context, int flags) const 
{
    XQDynamicContext* xqcontext=CAST_TO_DYNAMIC(context);
    XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *result = 0;
#ifndef _DEBUG
    // in debug builds, I'd rather get the exception
    try
    {
#endif
        switch(m_nodeType)
        {
        case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE:
        {
            XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument* newNode = context->createNewDocument();
            for (VectorOfDataItems::const_iterator itCont = m_children->begin(); itCont != m_children->end (); ++itCont) 
            {
                DataItem* childItem=(*itCont);
                Result oneChild = childItem->collapseTree(context);
                Item::Ptr child;
                while((child = oneChild.next(context)) != NULLRCP)
                {
                    if(child->isNode())
                    {
                        const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* sourceNode=((Node*)(const Item*)child)->getDOMNode();
                        assert(sourceNode!=NULL);
                        // If the content sequence contains an attribute node, a type error is raised.[err:XP0004][err:XP0006]
                        if(sourceNode->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE)
                            DSLthrow(DataItemException,X("DOM Constructor"),X("An attribute node cannot be a child of a document [err:XP0006]"));
                        // If the content sequence contains a document node, the document node is replaced in the content 
                        // sequence by its children.
                        else if(sourceNode->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE)
                        {
                            XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* childNode=sourceNode->getFirstChild();
                            while(childNode)
                            {
                                XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* imported=newNode->importNode(const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(childNode),true);
                                newNode->appendChild(imported); 
                                if(xqcontext->getDebugCallback()) xqcontext->getDebugCallback()->ReportClonedNode(context, childNode, imported);
                                childNode=childNode->getNextSibling();
                            }
                        }
                        // Adjacent text nodes in the content sequence are merged into a single text node by concatenating 
                        // their contents, with no intervening blanks. After concatenation, any text node whose content is 
                        // a zero-length string is deleted from the content sequence.
                        else if(sourceNode->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE && 
                                newNode->getLastChild()!=NULL &&
                                newNode->getLastChild()->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE)
                        {
                            const XMLCh* buff=XPath2Utils::concatStrings(newNode->getLastChild()->getNodeValue(),sourceNode->getNodeValue(),context->getMemoryManager());
                            newNode->getLastChild()->setNodeValue(buff);
                        }
                        else
                        {
                            XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* imported=newNode->importNode(const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(sourceNode),true);
                            newNode->appendChild(imported); 
                            if(xqcontext->getDebugCallback()) xqcontext->getDebugCallback()->ReportClonedNode(context,sourceNode, imported);
                        }
                    }
                    else
                    {
                        XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* last=newNode->getLastChild();
                        if(last!=NULL && last->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE)
                        {
                            const XMLCh* buff=XPath2Utils::concatStrings(last->getNodeValue(),child->asString(context),context->getMemoryManager());
                            last->setNodeValue(buff);
                        }
                        else
                        {
                            XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* imported=newNode->createTextNode(child->asString(context));
                            newNode->appendChild(imported); 
                        }
                    }
                }
            }
            result = newNode;
            break;
        }
        case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE:
        {
            Result resName=m_name->collapseTree(context).atomize(context);
            AnyAtomicType::Ptr itemName=resName.next(context);
            if(itemName==NULLRCP || resName.next(context)!=NULLRCP)
                DSLthrow(DataItemException,X("DOM Constructor"),X("The name for the element must be a single xs:QName, xs:string or xs:untypedAtomic item [err:XP0006]"));

            // Add a new scope for the namespace definitions, before we try to assign a URI to the name of the element
            const XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNSResolver* oldNsResolver=context->getNSResolver();
            XQScopedNamespace newNSScope(context->getMemoryManager(), oldNsResolver);
            context->setNSResolver(&newNSScope);
            std::vector<XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*, PathanAllocator<XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*> > attrList(PathanAllocator<XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*>(context->getMemoryManager()));
            for (VectorOfDataItems::const_iterator itAttr = m_attrList->begin(); itAttr != m_attrList->end (); ++itAttr) 
            {
                DataItem* attrItem=(*itAttr);
                static SequenceType nodeSequence(new SequenceType::ItemType(SequenceType::ItemType::TEST_NODE), SequenceType::STAR);
                Result oneAttribute=attrItem->collapseTree(context).convertFunctionArg(&nodeSequence,context);
                Item::Ptr attr;
                while((attr = oneAttribute.next(context)) != NULLRCP) 
                {
                    if(attr->isNode()) 
                    {
                        const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* node=((Node*)(const Item*)attr)->getDOMNode();
                        if(node->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE) 
                        {
                            XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr* pAttr=const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*>(static_cast<const XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*>(node));
                            // check for namespace nodes
                            if(XPath2Utils::equals(pAttr->getName(),XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXMLNSString))
                                xqcontext->setNamespaceBinding(XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString,pAttr->getValue());
                            else if(pAttr->getPrefix() && XPath2Utils::equals(pAttr->getPrefix(),XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXMLNSString))
                                xqcontext->setNamespaceBinding(pAttr->getLocalName(),pAttr->getValue());

                            // TODO: this is according to the spec, but in the Use Cases, literal namespace attributes
                            //       are expected to show up in the element they are attached to, so for the moment
                            //       I keep adding them to the list of attributes
                            //else // if it's not a namespace node, add it to the list of attributes of the element
                            attrList.push_back(pAttr);
                        }
                    }
                }
            }

            const XMLCh* nodeUri=NULL, *nodeName=NULL;
            if(itemName->getPrimitiveTypeIndex()==AnyAtomicType::QNAME)
            {
                const ATQNameOrDerived* pQName=(const ATQNameOrDerived*)(const AnyAtomicType*)itemName;
                nodeUri=pQName->getURI();
                nodeName=pQName->getName();
            }
            else if(itemName->getPrimitiveTypeIndex()==AnyAtomicType::STRING || itemName->getPrimitiveTypeIndex()==AnyAtomicType::UNTYPED_ATOMIC)
            {
                const XMLCh* pString=itemName->asString(context);
                const XMLCh* prefix=XPath2NSUtils::getPrefix(pString, context->getMemoryManager());
                nodeUri=context->getUriBoundToPrefix(prefix);
                // if the prefix was empty and we didn't find a xmlns=".." declaration in the scope, use the default element/type ns
                if((prefix==0 || *prefix==0) && (nodeUri==0 || *nodeUri==0))
                    nodeUri=context->getDefaultElementAndTypeNS();
                // keep the specified prefix in the node name
                nodeName=pString;
            }
            else
                DSLthrow(DataItemException,X("DOM Constructor"),X("The name for the element must be either a xs:QName, xs:string or xs:untypedAtomic [err:XP0006]"));

            // TODO: what error should we return if the string is empty?
            if(XERCES_CPP_NAMESPACE_QUALIFIER XMLString::stringLen(nodeName)==0)
                DSLthrow(DataItemException,X("DOM Constructor"),X("The name for the element is empty"));

            XERCES_CPP_NAMESPACE_QUALIFIER DOMElement* newNode=xqcontext->getOutputDocument()->createElementNS(nodeUri,nodeName);
            for(std::vector<XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*, PathanAllocator<XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*> >::iterator it=attrList.begin();it!=attrList.end();it++)
                newNode->setAttributeNode(*it);

            XQStaticContext::ConstructionMode constrMode=xqcontext->getConstructionMode();
            for (VectorOfDataItems::const_iterator itCont = m_children->begin(); itCont != m_children->end (); ++itCont) 
            {
                DataItem* childItem=(*itCont);
                Result childList=childItem->collapseTree(context);
                Item::Ptr child;
                bool lastWasAtomic = false;
                while((child = childList.next(context)) != NULLRCP)
                {
                    if(child->isNode())
                    {
                        lastWasAtomic = false;
                        const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* sourceNode=((Node*)(const Item*)child)->getDOMNode();
                        assert(sourceNode!=NULL);
                        switch(sourceNode->getNodeType())
                        {
                            case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE:
                            {
                                XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* childNode=sourceNode->getFirstChild();
                                while(childNode)
                                {
                                    XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* imported=newNode->getOwnerDocument()->importNode(const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(childNode),true);
                                    newNode->appendChild(imported); 
                                    if(constrMode==XQStaticContext::CONSTRUCTION_MODE_PRESERVE && childNode->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE)
                                        XPath2Utils::copyElementType(newNode->getOwnerDocument(), (XERCES_CPP_NAMESPACE_QUALIFIER DOMElement*)imported, (XERCES_CPP_NAMESPACE_QUALIFIER DOMElement*)childNode);
                                    if(xqcontext->getDebugCallback()) 
                                        xqcontext->getDebugCallback()->ReportClonedNode(context, childNode, imported);
                                    childNode=childNode->getNextSibling();
                                }
                                break;
                            }
                            case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE:
                            {
                                if(newNode->getFirstChild()!=NULL)
                                    DSLthrow(DataItemException,X("DOM Constructor"),X("Attribute nodes must be created before the other child nodes of an element [err:XQ0024]"));
                                XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr* newAttr=static_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*>(newNode->getOwnerDocument()->importNode(const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(sourceNode),true));
                                if(constrMode==XQStaticContext::CONSTRUCTION_MODE_PRESERVE)
                                    XPath2Utils::copyAttributeType(newNode->getOwnerDocument(), newAttr, (const XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*)sourceNode);
                                newNode->setAttributeNode(newAttr);
                                break;
                            }
                            default:
                            {
                                if(sourceNode->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE && 
                                   newNode->getLastChild()!=NULL &&
                                   newNode->getLastChild()->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE)
                                {
                                    const XMLCh* buff=XPath2Utils::concatStrings(newNode->getLastChild()->getNodeValue(),sourceNode->getNodeValue(),context->getMemoryManager());
                                    newNode->getLastChild()->setNodeValue(buff);
                                }
                                else
                                {
                                    if(newNode->getOwnerDocument()!=sourceNode->getOwnerDocument())
                                    {
                                        XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* imported=newNode->getOwnerDocument()->importNode(const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(sourceNode),true);
                                        newNode->appendChild(imported); 
                                        if(constrMode==XQStaticContext::CONSTRUCTION_MODE_PRESERVE && sourceNode->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE)
                                            XPath2Utils::copyElementType(newNode->getOwnerDocument(), (XERCES_CPP_NAMESPACE_QUALIFIER DOMElement*)imported, (XERCES_CPP_NAMESPACE_QUALIFIER DOMElement*)sourceNode);
                                        if(xqcontext->getDebugCallback()) 
                                            xqcontext->getDebugCallback()->ReportClonedNode(context,sourceNode, imported);
                                    }
                                    else
                                    {
                                        DataItem* pChild=childItem;
                                        if((unsigned int)pChild->getType()==XQContext::DEBUG_HOOK)
                                            pChild=((XQDebugHook*)pChild)->m_impl;
                                        // if the node we should add as a child was generated by a DOM constructor, just import it
                                        // otherwise, make a copy of it
                                        if((unsigned int)pChild->getType()!=XQContext::DOM_CONSTRUCTOR)
                                        {
                                            XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* imported=sourceNode->cloneNode(true);
                                            newNode->appendChild(imported);
                                            if(constrMode==XQStaticContext::CONSTRUCTION_MODE_PRESERVE && sourceNode->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE)
                                                XPath2Utils::copyElementType(newNode->getOwnerDocument(), (XERCES_CPP_NAMESPACE_QUALIFIER DOMElement*)imported, (XERCES_CPP_NAMESPACE_QUALIFIER DOMElement*)sourceNode);
                                            if(xqcontext->getDebugCallback()) 
                                                xqcontext->getDebugCallback()->ReportClonedNode(context,sourceNode, imported);
                                        }
                                        else
                                        {
                                            // no need to copy type info, we are not making a copy of the source node
                                            newNode->appendChild(const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(sourceNode));
                                        }
                                    }
                                }
                            }
                        }
                    }
                    else
                    {
                        const XMLCh* valueStr=child->asString(context);
                        if(lastWasAtomic)
                        {
                            XMLCh space[]={ ' ', 0 };
                            valueStr=XPath2Utils::concatStrings(space,valueStr,context->getMemoryManager());
                        }
                        lastWasAtomic = true;
                        XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* last=newNode->getLastChild();
                        if(last!=NULL && last->getNodeType()==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE)
                        {
                            const XMLCh* buff=XPath2Utils::concatStrings(last->getNodeValue(),valueStr,context->getMemoryManager());
                            last->setNodeValue(buff);
                        }
                        else
                        {
                            XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* imported=xqcontext->getOutputDocument()->createTextNode(valueStr);
                            newNode->appendChild(imported); 
                        }
                    }
                }
            }
            if(constrMode==XQStaticContext::CONSTRUCTION_MODE_PRESERVE)
                XPath2Utils::setElementType(newNode->getOwnerDocument(),newNode,XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgATTVAL_ANYTYPE);
            result = newNode;
            context->setNSResolver(oldNsResolver);
            break;
        }
        case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE:
        {
            Result resName=m_name->collapseTree(context).atomize(context);
            AnyAtomicType::Ptr itemName=resName.next(context);
            if(itemName==NULLRCP || resName.next(context)!=NULLRCP)
                DSLthrow(DataItemException,X("DOM Constructor"),X("The name for the attribute must be a single xs:QName, xs:string or xs:untypedAtomic item [err:XP0006]"));

            const XMLCh* nodeUri=NULL, *nodeName=NULL;
            if(itemName->getPrimitiveTypeIndex()==AnyAtomicType::QNAME)
            {
                const ATQNameOrDerived* pQName=(const ATQNameOrDerived*)(const AnyAtomicType*)itemName;
                nodeUri=pQName->getURI();
                nodeName=pQName->getName();
            }
            else if(itemName->getPrimitiveTypeIndex()==AnyAtomicType::STRING || itemName->getPrimitiveTypeIndex()==AnyAtomicType::UNTYPED_ATOMIC)
            {
                const XMLCh* pString=itemName->asString(context);
                const XMLCh* prefix=XPath2NSUtils::getPrefix(pString, context->getMemoryManager());
                // if the prefix was empty we are in no namespace
                if(prefix==0 || *prefix==0)
                    nodeUri=NULL;
                else
                    nodeUri=context->getUriBoundToPrefix(prefix);
                // keep the specified prefix in the node name
                nodeName=pString;
            }
            else
                DSLthrow(DataItemException,X("DOM Constructor"),X("The name for the attribute must be either a xs:QName, xs:string or xs:untypedAtomic [err:XP0006]"));

            if(nodeUri==NULL && XPath2Utils::equals(nodeName, XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXMLNSString))
                nodeUri=XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXMLNSURIName;
            // TODO: add a flag to distinguish between direct attribute constructors and computed constructor: this test must
            // be done only on computed attribute constuctors
            //if(XPath2Utils::equals(nodeUri, XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXMLNSURIName))
            //    DSLthrow(DataItemException,X("DOM Constructor"),X("A computed attribute constructor cannot create a namespace declaration [err:XQ0044]"));

            // TODO: what error should we return if the string is empty?
            if(XERCES_CPP_NAMESPACE_QUALIFIER XMLString::stringLen(nodeName)==0)
                DSLthrow(DataItemException,X("DOM Constructor"),X("The name for the attribute is empty"));

            XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr* newAttr=xqcontext->getOutputDocument()->createAttributeNS(nodeUri,nodeName);
            XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer value;
            for (VectorOfDataItems::const_iterator itCont = m_children->begin(); itCont != m_children->end (); ++itCont) 
            {
                Result childList=(*itCont)->collapseTree(context).atomize(context);
                Item::Ptr child;
                XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer subValue;
                while((child = childList.next(context)) != NULLRCP)
                {
                    if(!subValue.isEmpty())
                        subValue.append(' ');
                    subValue.append(child->asString(context));
                }
                value.append(subValue.getRawBuffer());
            }
            newAttr->setValue(value.getRawBuffer());
            result = newAttr;
            break;
        }
        case XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNamespace::XPATH_NAMESPACE_NODE:
        {
            const XMLCh* nsPrefix=m_name->collapseTree(context).castAsSingleString(context);
            assert(m_children->size()==1);
            const XMLCh* nsURI=(*m_children)[0]->collapseTree(context).castAsSingleString(context);
            // add the new binding (nsPrefix may be empty)
            xqcontext->setNamespaceBinding(nsPrefix,nsURI);
            break;
        }
        case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::PROCESSING_INSTRUCTION_NODE:
        {
            Result resName=m_name->collapseTree(context).atomize(context);
            AnyAtomicType::Ptr itemName=resName.next(context);
            if(itemName==NULLRCP || resName.next(context)!=NULLRCP)
                DSLthrow(DataItemException,X("DOM Constructor"),X("The target for the processing instruction must be a single xs:NCName, xs:string or xs:untypedAtomic item [err:XP0006]"));

            const XMLCh* nodeName=NULL;
            // the specs specify that a xs:NCName could be returned, but we create a xs:string in that case
            if(itemName->getPrimitiveTypeIndex()==AnyAtomicType::STRING || itemName->getPrimitiveTypeIndex()==AnyAtomicType::UNTYPED_ATOMIC)
            {
                nodeName=itemName->asString(context);
                if(XERCES_CPP_NAMESPACE_QUALIFIER XMLString::indexOf(nodeName,XERCES_CPP_NAMESPACE_QUALIFIER chColon)!=-1)
                    DSLthrow(DataItemException,X("DOM Constructor"),X("The target for the processing instruction must be a valid xs:NCName [err:XQ0041]"));
            }
            else
                DSLthrow(DataItemException,X("DOM Constructor"),X("The target for the processing instruction must be a single xs:NCName, xs:string or xs:untypedAtomic item [err:XP0006]"));

            if(XERCES_CPP_NAMESPACE_QUALIFIER XMLString::compareIString(nodeName, XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXMLString)==0)
                DSLthrow(DataItemException,X("DOM Constructor"),X("The target for the processing instruction must not be 'XML' [err:XQ0064]"));
            int nIndex=XERCES_CPP_NAMESPACE_QUALIFIER XMLString::indexOf(nodeName, XERCES_CPP_NAMESPACE_QUALIFIER chQuestion);
            int nTargetLen=XERCES_CPP_NAMESPACE_QUALIFIER XMLString::stringLen(nodeName);
            while(nIndex!=-1)
            {
                if((nIndex+1)<nTargetLen && nodeName[nIndex+1]==XERCES_CPP_NAMESPACE_QUALIFIER chCloseAngle)
                    DSLthrow(DataItemException,X("DOM Constructor"),X("The target for the processing instruction must not contain the string '?>' [err:XQ0026]"));
                nIndex=XERCES_CPP_NAMESPACE_QUALIFIER XMLString::indexOf(nodeName, XERCES_CPP_NAMESPACE_QUALIFIER chQuestion, nIndex+1);
            }

            if(XERCES_CPP_NAMESPACE_QUALIFIER XMLString::stringLen(nodeName)==0)
                DSLthrow(DataItemException,X("DOM Constructor"),X("The name for the processing instruction is empty"));
            XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer value;
            for (VectorOfDataItems::const_iterator itCont = m_children->begin(); itCont != m_children->end (); ++itCont) 
            {
                Result childList=(*itCont)->collapseTree(context).atomize(context);
                Item::Ptr child;
                while((child = childList.next(context)) != NULLRCP)
                {
                    if(!value.isEmpty())
                        value.append(' ');
                    const XMLCh* toAppend=child->asString(context);
                    int nIndex=XERCES_CPP_NAMESPACE_QUALIFIER XMLString::indexOf(toAppend, XERCES_CPP_NAMESPACE_QUALIFIER chQuestion);
                    int nTargetLen=XERCES_CPP_NAMESPACE_QUALIFIER XMLString::stringLen(toAppend);
                    while(nIndex!=-1)
                    {
                        if((nIndex+1)<nTargetLen && toAppend[nIndex+1]==XERCES_CPP_NAMESPACE_QUALIFIER chCloseAngle)
                            DSLthrow(DataItemException,X("DOM Constructor"),X("The content for the processing instruction must not contain the string '?>' [err:XQ0026]"));
                        nIndex=XERCES_CPP_NAMESPACE_QUALIFIER XMLString::indexOf(toAppend, XERCES_CPP_NAMESPACE_QUALIFIER chQuestion, nIndex+1);
                    }
                    value.append(toAppend);
                }
            }
            XERCES_CPP_NAMESPACE_QUALIFIER DOMProcessingInstruction* newPI=xqcontext->getOutputDocument()->createProcessingInstruction(nodeName,value.getRawBuffer());
            result = newPI;
            break;
        }
        case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::COMMENT_NODE:
        {
            Result commentValue=m_name->collapseTree(context).atomize(context);
            XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer value;
            Item::Ptr child;
            while((child = commentValue.next(context)) != NULLRCP)
            {
                if(!value.isEmpty())
                    value.append(' ');
                const XMLCh* toAppend=child->asString(context);
                int nTargetLen=XERCES_CPP_NAMESPACE_QUALIFIER XMLString::stringLen(toAppend);
                // avoid that a '--' appears in the final string
                for(int i=0; i<nTargetLen; i++)
                {
                    if(!value.isEmpty() && value.getRawBuffer()[value.getLen()-1]==XERCES_CPP_NAMESPACE_QUALIFIER chDash && toAppend[i]==XERCES_CPP_NAMESPACE_QUALIFIER chDash)
                        value.append(XERCES_CPP_NAMESPACE_QUALIFIER chSpace);
                    value.append(toAppend[i]);
                }
            }
            // avoid that the comment ends with a '-'
            if(!value.isEmpty() && value.getRawBuffer()[value.getLen()-1]==XERCES_CPP_NAMESPACE_QUALIFIER chDash)
                value.append(XERCES_CPP_NAMESPACE_QUALIFIER chSpace);
            XERCES_CPP_NAMESPACE_QUALIFIER DOMComment* newComment=xqcontext->getOutputDocument()->createComment(value.getRawBuffer());
            result = newComment;
            break;
        }
        case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE:
        {
            XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer value;
            for (VectorOfDataItems::const_iterator itCont = m_children->begin(); itCont != m_children->end (); ++itCont) 
            {
                Result childList=(*itCont)->collapseTree(context).atomize(context);
                Item::Ptr child;
                while((child = childList.next(context)) != NULLRCP)
                {
                    if(!value.isEmpty())
                        value.append(' ');
                    value.append(child->asString(context));
                }
            }
            if(!value.isEmpty())
            {
                XERCES_CPP_NAMESPACE_QUALIFIER DOMText* newText=xqcontext->getOutputDocument()->createTextNode(value.getRawBuffer());
                result = newText;
            }
            break;
        }
        default:
            assert(false);
            break;
        }
#ifndef _DEBUG
    }
    catch(XERCES_CPP_NAMESPACE_QUALIFIER DOMException& e) {
        DSLthrow(DataItemException,X("DOM Constructor"),X("Unexpected DOM exception"));
    }
#endif
    if(result)
        return Sequence(DatatypeFactory::POD2AT::createNode(result, context), context->getMemoryManager());
    return Sequence(context->getMemoryManager());
}

DataItem* XQDOMConstructor::staticResolution(StaticContext *context, StaticResolutionContext * src)
{
  XQContext* xqcontext = CAST_TO_XQCONTEXT(context);

  StaticResolutionContext newSrc(context->getMemoryManager());

  if(m_nodeType==XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNamespace::XPATH_NAMESPACE_NODE) {
    AutoRelease<DynamicContext> dContext(context->createDynamicContext());
    dContext->setMemoryManager(context->getMemoryManager());
    // m_name and m_children[0] are always literals, no need to test for isConstant
    const XMLCh* nsPrefix=m_name->collapseTree(dContext).castAsSingleString(dContext);
    assert(m_children->size()==1);
    const XMLCh* nsURI=(*m_children)[0]->collapseTree(dContext).castAsSingleString(dContext);
    xqcontext->setNamespaceBinding(nsPrefix, nsURI);
    return this;    // no predicates, no costant fold
  }

  m_name = m_name->staticResolution(context, &newSrc);

  unsigned int i;
  // Add a new scope for the namespace definitions
  const XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNSResolver* oldNsResolver=context->getNSResolver();
  XQScopedNamespace newNSScope(context->getMemoryManager(), oldNsResolver);
  context->setNSResolver(&newNSScope);
  for (i=0;i<m_attrList->size();i++) 
    (*m_attrList)[i]=(*m_attrList)[i]->staticResolution(context, &newSrc);
  for (i=0;i<m_children->size();i++) 
    (*m_children)[i]=(*m_children)[i]->staticResolution(context, &newSrc);
  context->setNSResolver(oldNsResolver);

  if(m_nodeType==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE && m_name->isConstant()) {
    AutoRelease<DynamicContext> dContext(context->createDynamicContext());
    dContext->setMemoryManager(context->getMemoryManager());

    Item::Ptr item;
    if(m_name->getType() == DataItem::LITERAL) {
      item = ((DataItemLiteral*)m_name)->getItemConstructor()->createItem(dContext);
    }
    else if(m_name->getType() == DataItem::SEQUENCE) {
      const ItemConstructor::Vector &ics = ((DataItemSequence*)m_name)->getItemConstructors();
      if(ics.size() == 1) {
        item = ics[0]->createItem(dContext);
      }
    }

    if(item != NULLRCP) {
      const XMLCh* strName = item->asString(dContext);
      const XMLCh* XMLNSPrefix=NULL;
      if(XPath2Utils::equals(strName,XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXMLNSString)) {
        XMLNSPrefix=XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString;
      }
      else if(XPath2Utils::equals(XPath2NSUtils::getPrefix(strName, context->getMemoryManager()),
                                                               XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXMLNSString)) {
        XMLNSPrefix=XPath2NSUtils::getLocalName(strName);
      }

      if(XMLNSPrefix!=NULL) {
        // we are a namespace attribute: check that we have a constant value
        if(m_children->size()==0) { // TODO: supporting Namespace 1.1 would mean unsetting the binding...
          xqcontext->setNamespaceBinding(XMLNSPrefix, XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString);
        }
        else if(m_children->size()>1 || (*m_children)[0]->getType()!=DataItem::SEQUENCE) {
          DSLthrow(StaticAnalysisException,X("DOM Constructor"),X("The value of a namespace declaration attribute must be a literal string [err:XQ0022]"));
        }
        else {
          DataItem *child = (*m_children)[0];
          Item::Ptr nsUri;
          if(child->getType() == DataItem::LITERAL) {
            nsUri = ((DataItemLiteral*)child)->getItemConstructor()->createItem(dContext);
          }
          else if(child->getType() == DataItem::SEQUENCE) {
            const ItemConstructor::Vector &ics = ((DataItemSequence*)child)->getItemConstructors();
            if(ics.size() == 1) {
              nsUri = ics[0]->createItem(dContext);
            }
          }

          if(nsUri == NULLRCP)
            DSLthrow(StaticAnalysisException,X("DOM Constructor"),X("The value of a namespace declaration attribute must be a literal string [err:XQ0022]"));
          xqcontext->setNamespaceBinding(XMLNSPrefix, nsUri->asString(dContext));
        }
      }
    }
  }

  src->add(&newSrc);
  src->forceNoFolding(true);
  return resolvePredicates(context, src); // Never constant fold
}

short XQDOMConstructor::getNodeType() const
{
  return m_nodeType;
}

const DataItem *XQDOMConstructor::getName() const
{
  return m_name;
}

const VectorOfDataItems *XQDOMConstructor::getAttributes() const
{
  return m_attrList;
}

const VectorOfDataItems *XQDOMConstructor::getChildren() const
{
  return m_children;
}

void XQDOMConstructor::setName(DataItem *name)
{
  m_name = name;
}
