/*
 * Copyright (c) 2001, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../config/pathan_config.h"
#include <assert.h>
#include <sstream>

#include <pathan/internal/utils/PrintDataItemTree.hpp>
#include <pathan/internal/navigation/NavStepImpl.hpp>
#include <xercesc/dom/DOMNodeList.hpp>
#include <xercesc/dom/DOMXPathNamespace.hpp>
#include <xercesc/util/XMLString.hpp>

#include <pathan/XPath2Utils.hpp>
#include <pathan/exceptions/NavigationException.hpp>
#include <pathan/Node.hpp>
#include <pathan/internal/DOMutils/XStr.hpp>
#include <pathan/internal/factory/DatatypeFactory.hpp>

NavStepImpl::NavStepImpl(void) {
  _name = 0; 
  _uri = 0;
  _prefix = 0;
  _type = 0;
  _wildcardName = false;
  _wildcardNamespace = false;
  _wildcardType = false;
  _axisOrientation = FORWARD;
  _usePrefix = false;
  _itemType = 0;
}

NavStepImpl::~NavStepImpl(void) {
  delete _itemType;
}

bool NavStepImpl::filterNode(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node, DynamicContext* context) const
{
    //std::cerr << "itemType=" << _itemType << " - Testing a node " << node->getNodeType() << " (" <<
    //          XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(node->getNodeName()) << "," <<
    //          (node->getNodeValue()?XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(node->getNodeValue()):"#NULL") << ")\n";
    if((_itemType!=NULL && _itemType->matches(node,context)) ||
       (_itemType==NULL && checkNodeType(node) && checkNodeName(node) && checkNodeNamespace(node, context)))
    {
      //std::cerr << "we passed the tests" << std::endl;
      return true;
    }//if
    return false;
}

bool NavStepImpl::checkNodeType(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node) const
{

	assert(node != 0);

	if(_wildcardType) {
		return true;
	}

  int nodeType = node->getNodeType();

  if(nodeType == _type) {
    return true;
  }

  //CDATA nodes are the same as Text, so we need to keep that in mind
  if((nodeType == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::CDATA_SECTION_NODE) && (_type == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE)) {
    return true;
  }//if

  return false;
}//checkNodeType

bool NavStepImpl::checkNodeName(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node) const
{
  if(_wildcardName) {
    return true;
  }
  else {
    assert(_name != 0);
    if(_type == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE || _type == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE || _type == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::PROCESSING_INSTRUCTION_NODE || XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNamespace::XPATH_NAMESPACE_NODE) {
      //to make sure we are ok with node created with DOM level 1
      const XMLCh* nodeName = node->getLocalName();
      if(nodeName == NULL) {
        nodeName = node->getNodeName();
      }
      return XPath2Utils::equals(nodeName, _name);
    } else {
      return true;
    }
  }
}



bool NavStepImpl::checkNodeNamespace(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node, DynamicContext* context) const
{
  if(_wildcardNamespace) {
    return true;
  }
  else if(node->getNodeType() == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE) {
    return (XPath2Utils::equals(node->getNamespaceURI(), _uri));
  }
  else if(node->getNodeType() == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE) {
    if(_uri == 0) {
      return (XPath2Utils::equals(node->getNamespaceURI(), context->getDefaultElementAndTypeNS()));
    }
    else {
      return (XPath2Utils::equals(node->getNamespaceURI(), _uri));
    }
  }
  else {
    return true;
  }
}

void NavStepImpl::setAxisOrientation(const AxisOrientation ori)
{
  _axisOrientation = ori;
}


void NavStepImpl::setNodeName(const XMLCh* name)
{
  _name = name;
}

void NavStepImpl::setNodePrefix(const XMLCh* prefix)
{
  _prefix=prefix;
  _usePrefix=true;
}

void NavStepImpl::setNodeUri(const XMLCh* uri)
{
  _uri = uri;
  _usePrefix=false;
}

void NavStepImpl::setNodeType(int type)
{
  _type = type;
}

void NavStepImpl::setNameWildcard(void)
{
  _wildcardName = true;
}

void NavStepImpl::setNamespaceWildcard(void)
{
  _wildcardNamespace = true;
}

void NavStepImpl::setTypeWildcard(void)
{
  _wildcardType = true;
}


int NavStepImpl::getNodeType(void) const
{
  return _type;
}

const XMLCh* NavStepImpl::getNodeUri(void) const
{
  return _uri;
}

const XMLCh* NavStepImpl::getNodePrefix(void) const
{
  return _prefix;
}

const XMLCh* NavStepImpl::getNodeName(void) const
{
  return _name;
}

bool NavStepImpl::getNameWildcard(void) const
{
  return _wildcardName;
}

bool NavStepImpl::getNamespaceWildcard(void) const
{
  return _wildcardNamespace;
}

bool NavStepImpl::getTypeWildcard(void) const
{
  return _wildcardType;
}


bool NavStepImpl::getTypeSet(void) const
{
  return _type != 0;
}

void NavStepImpl::staticResolution(StaticContext *context)
{
  if(getPrefixSet()) {
    setNodeUri(context->getUriBoundToPrefix(getNodePrefix()));
  }
}

Result NavStepImpl::createResult(DynamicContext* context) const {
  return Sequence(context->getMemoryManager());
}

int NavStepImpl::getAxisOrientation() const {
	return _axisOrientation;
}

bool NavStepImpl::getPrefixSet(void) const {
	return _usePrefix;
}

SequenceType::ItemType* NavStepImpl::getType(void) const {
    return _itemType;
}

void NavStepImpl::setType(SequenceType::ItemType* type) {
    _itemType=type;
}

NavStepImpl::Axis NavStepImpl::getAxis() const {
  return NavStepImpl::UNKNOWN;
}

/////////////////////////////////////
// AxisResult
/////////////////////////////////////

NavStepImpl::AxisResult::AxisResult(const NavStepImpl *step, DynamicContext *context)
  : ResultImpl(context),
    contextNode_(0),
    step_(step)
{
}

Item::Ptr NavStepImpl::AxisResult::next(DynamicContext *context)
{
  if(contextNode_ == 0) {
    const Item::Ptr item = context->getContextItem();

    if(item == NULLRCP || !item->isNode()) {
      DSLthrow(NavigationException,X("NavStepImpl::AxisStep::next"), X("An attempt was made to perform an axis step when the Context Item was not a node [err:XP0020]"));
    }

    contextNode_ = ((Node*)(const Item*)item)->getDOMNode();
    if(contextNode_==NULL) {
      DSLthrow(NavigationException,X("NavStepImpl::AxisStep::next"), X("An attempt was made to perform an axis step when the Context Item was not a node [err:XP0020]"));
    }
  }

  const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *result = 0;
  while((result = stepNext(context)) != 0 &&
        (result->getNodeType() == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ENTITY_REFERENCE_NODE ||
         !step_->filterNode(result, context))) {}
  return result == 0 ? (const Item::Ptr)0 : (const Item::Ptr)DatatypeFactory::POD2AT::createNode(result, context);
}

std::string NavStepImpl::AxisResult::asString(DynamicContext *context, int indent) const
{
  std::ostringstream oss;
  std::string in(getIndent(indent));

  oss << in << "<axis name=\"" << PrintDataItemTree::getAxisName(step_->getAxis()) << "\"/>" << std::endl;

  return oss.str();
}
