/*
 * Copyright (c) 2004, Parthenon Computing Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../config/pathan_config.h"
#include <pathan/internal/StringPool.hpp>
#include <pathan/XPath2Utils.hpp>

#include <xercesc/util/XMLUni.hpp>

#if defined(XERCES_HAS_CPP_NAMESPACE)
XERCES_CPP_NAMESPACE_USE
#endif

static const unsigned int initialModulus = 233;
static const unsigned int lengthThreshold = 100;

StringPool::StringPool(MemoryManager *mm)
  : _mm(mm),
    _bucketList(0),
    _modulus(initialModulus),
    _count(0),
    _hits(0),
    _misses(0),
    _toobig(0)
{
  _bucketList = (Bucket**)_mm->allocate(_modulus * sizeof(Bucket*));
  memset(_bucketList, 0, _modulus * sizeof(Bucket*));
}

StringPool::~StringPool()
{
  Bucket *bucket, *next;
  for(unsigned int i = 0; i < _modulus; ++i) {
    bucket = _bucketList[i];
    while(bucket != 0) {
      next = bucket->next;
      _mm->deallocate((void*)bucket->value);
      _mm->deallocate(bucket);
      bucket = next;
    }
  }
  _mm->deallocate(_bucketList);
}

const XMLCh *StringPool::getPooledString(const XMLCh *src)
{
  if(src == 0) return 0;
  if(*src == 0) return XMLUni::fgZeroLenString;

  // strings longer than lengthThreshold bytes are not pooled, as it is not probable they can be recycled
  const XMLCh* pszTmp = src + 1;
  while(*pszTmp) ++pszTmp;
  unsigned int length = (unsigned int)(pszTmp - src);

  if(length > lengthThreshold) {
    ++_toobig;
    return replicate(src, length);
  }

  unsigned int hashVal = hash(pszTmp);
  unsigned int modHashVal = hashVal % _modulus;

  const Bucket *bucket = _bucketList[modHashVal];
  while(bucket) {
    if(bucket->length == length &&
       XPath2Utils::equals(bucket->value, src)) {
      break;
    }
    bucket = bucket->next;
  }

  if(bucket) {
    ++_hits;
    return bucket->value;
  }
  else {
    ++_misses;
    if(_count >= (_modulus * 3 / 4)) {
      resize();
      modHashVal = hashVal % _modulus;
    }

    const XMLCh *result = replicate(src, length);
    _bucketList[modHashVal] = new (_mm->allocate(sizeof(Bucket)))
      Bucket(result, length, hashVal, _bucketList[modHashVal]);
    ++_count;

    return result;
  }
}

const XMLCh *StringPool::getPooledString(const char *src)
{
  if(src == 0) return 0;
  if(*src == 0) return XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString;

  XMLCh *transcoded = XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(src, _mm);
  if(!transcoded) return 0;

  // strings longer than lengthThreshold bytes are not pooled, as it is not probable they can be recycled
  const XMLCh* pszTmp = transcoded + 1;
  while(*pszTmp) ++pszTmp;
  unsigned int length = (unsigned int)(pszTmp - transcoded);

  if(length > lengthThreshold) {
    ++_toobig;
    return transcoded;
  }

  unsigned int hashVal = hash(pszTmp);
  unsigned int modHashVal = hashVal % _modulus;

  const Bucket *bucket = _bucketList[modHashVal];
  while(bucket) {
    if(bucket->length == length &&
       XPath2Utils::equals(bucket->value, transcoded)) {
      break;
    }
    bucket = bucket->next;
  }

  if(bucket) {
    ++_hits;
    _mm->deallocate(transcoded);
    return bucket->value;
  }
  else {
    ++_misses;
    if(_count >= (_modulus * 3 / 4)) {
      resize();
      modHashVal = hashVal % _modulus;
    }

    _bucketList[modHashVal] = new (_mm->allocate(sizeof(Bucket)))
      Bucket(transcoded, length, hashVal, _bucketList[modHashVal]);
    ++_count;

    return transcoded;
  }
}

void StringPool::resize()
{
  unsigned int new_modulus  = (_modulus << 1) + 1;
  Bucket **new_bucketList = (Bucket**)_mm->allocate(new_modulus * sizeof(Bucket*));
  memset(new_bucketList, 0, new_modulus * sizeof(Bucket*));

  unsigned int modHashVal;
  Bucket *bucket, *next;
  for(unsigned int i = 0; i < _modulus; ++i) {
    bucket = _bucketList[i];
    while(bucket != 0) {
      next = bucket->next;

      modHashVal = bucket->hashValue % new_modulus;
      bucket->next = new_bucketList[modHashVal];
      new_bucketList[modHashVal] = bucket;
      
      bucket = next;
    }
  }

  _mm->deallocate(_bucketList);
  _bucketList = new_bucketList;
  _modulus = new_modulus;
}
