/*
 * Copyright (c) 2003, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../../config/pathan_config.h"
#include "NodeImpl.hpp"
#include <pathan/DynamicContext.hpp>
#include <pathan/XPath2MemoryManager.hpp>
#include <pathan/Sequence.hpp>
#include <pathan/dataItem/Result.hpp>
#include <pathan/exceptions/ItemException.hpp>
#include "../../exceptions/XPath2TypeCastException.hpp"
#include <pathan/ATAnyURIOrDerived.hpp>
#include <pathan/ATStringOrDerived.hpp>
#include <pathan/ATQNameOrDerived.hpp>
#include <pathan/ATUntypedAtomic.hpp>
#include <pathan/XPath2NSUtils.hpp>
#include <pathan/PathanException.hpp>
#include <pathan/functions/FunctionRoot.hpp>
#include <pathan/functions/FunctionConstructor.hpp>
#include <pathan/internal/factory/DatatypeFactory.hpp>
#include <pathan/DocumentCache.hpp>
#include <pathan/XPath2Utils.hpp>
#include <pathan/internal/context/DocumentCacheImpl.hpp>

#include <xercesc/dom/DOM.hpp>
#include <xercesc/dom/DOMPSVITypeInfo.hpp>
#include <xercesc/validators/schema/SchemaSymbols.hpp>
#include <xercesc/validators/schema/SchemaElementDecl.hpp>
#include <xercesc/validators/schema/ComplexTypeInfo.hpp>
#include <xercesc/validators/datatype/ListDatatypeValidator.hpp>
#include <assert.h>

const XMLCh NodeImpl::ls_string[] =
{
    XERCES_CPP_NAMESPACE_QUALIFIER chLatin_L, XERCES_CPP_NAMESPACE_QUALIFIER chLatin_S,
    XERCES_CPP_NAMESPACE_QUALIFIER chNull
};

NodeImpl::NodeImpl(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node, const DynamicContext *context) : 
    Node(), 
    fSerializer(0), 
    fNode(node),
    _context(context)
{  
  assert(node!=0);
  _context->getDocumentCache()->incrementDocumentRefCount(XPath2Utils::getOwnerDoc(fNode));
}

NodeImpl::~NodeImpl()
{
  _context->getDocumentCache()->decrementDocumentRefCount(XPath2Utils::getOwnerDoc(fNode), _context);
}

bool NodeImpl::isNode() const
{
    return true;
}

bool NodeImpl::isAtomicValue() const
{
    return false;
}

const XMLCh* NodeImpl::asString(const DynamicContext* context) const
{
  if(!fSerializer) {
    XERCES_CPP_NAMESPACE_QUALIFIER DOMImplementation *impl = XERCES_CPP_NAMESPACE_QUALIFIER DOMImplementationRegistry::getDOMImplementation(ls_string);
    ((NodeImpl*)this)->fSerializer = ((XERCES_CPP_NAMESPACE_QUALIFIER DOMImplementationLS*)impl)->createDOMWriter(context->getMemoryManager());
  }
  return fSerializer->writeToString(*fNode);
}

/** check if the underlying type of this node is instance of a certain type */
bool NodeImpl::hasInstanceOfType(const XMLCh* typeURI, const XMLCh* typeName, const DynamicContext* context) const {
    const XMLCh* uri, *name;
    getTypeUriAndName(uri,name);
    return context->isTypeOrDerivedFromType(uri, name, typeURI, typeName);
}


/* Accessor functions */

Sequence NodeImpl::dmBaseURI(const DynamicContext* context) const {
  
	switch (fNode->getNodeType()) {
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE : 
        {
            XERCES_CPP_NAMESPACE_QUALIFIER DOMElement* pNode=((XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*)fNode)->getOwnerElement();
            if(pNode)
            {
                const XMLCh *tmp = pNode->getBaseURI();
                if(tmp && tmp[0]) 
                {
                    const ATAnyURIOrDerived::Ptr tempURI = DatatypeFactory::STR2AT::createAnyURI(tmp, context);
                    return Sequence(tempURI, context->getMemoryManager());
                }
            }
        }
        break;
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::COMMENT_NODE : 
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE : 
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE : 
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE :
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::PROCESSING_INSTRUCTION_NODE : 
        {
            const XMLCh *tmp = fNode->getBaseURI();
            if(tmp && tmp[0]) 
            {
                const ATAnyURIOrDerived::Ptr tempURI = DatatypeFactory::STR2AT::createAnyURI(tmp, context);
                return Sequence(tempURI, context->getMemoryManager());
            }
        }
	}
    return Sequence(context->getMemoryManager());
}

// The dm:node-kind accessor returns a string value representing the
// node's kind: either "document", "element", "attribute", "text",
// "namespace", "processing-instruction", or "comment".

const XMLCh* NodeImpl::dmNodeKind(void) const {
  
    switch(fNode->getNodeType()) {
        
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE:
        return document_string;
        
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE:
        return element_string;
        
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE:
        return attribute_string;

    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE:
        return text_string;
        
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::PROCESSING_INSTRUCTION_NODE:
        return processing_instruction_string;
        
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::COMMENT_NODE:
        return comment_string;
        
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNamespace::XPATH_NAMESPACE_NODE:
        return namespace_string;
	}
    
  DSLthrow(ItemException, X("NodeImpl::dmNodeKind"), X("Unknown node type."));
}


Sequence NodeImpl::dmNodeName(const DynamicContext* context) const {
  
	switch(fNode->getNodeType())
	{
	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE:					
        return Sequence(DatatypeFactory::STR2AT::createQName(fNode->getNamespaceURI(), fNode->getLocalName(), context), context->getMemoryManager());

	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE:				
        return Sequence(DatatypeFactory::STR2AT::createQName(fNode->getNamespaceURI(), fNode->getLocalName(), context), context->getMemoryManager());

	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::PROCESSING_INSTRUCTION_NODE:	
        return Sequence(DatatypeFactory::STR2AT::createQName(XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString,fNode->getNodeName(), context), context->getMemoryManager());

	case XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNamespace::XPATH_NAMESPACE_NODE:  
        {
            const XMLCh* prefix = fNode->getPrefix();
            if(prefix)         
                return Sequence(DatatypeFactory::STR2AT::createQName(XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString, prefix, context), context->getMemoryManager());
        }
	}
	return Sequence(context->getMemoryManager());
}


void NodeImpl::addStringValueToBuffer(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* node, XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer& buffer) const
{
    short nodeType=node->getNodeType();
    if(nodeType==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE ||
       nodeType==XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::CDATA_SECTION_NODE)
    {
        buffer.append(node->getNodeValue());
    }
    else
    {
        for(XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* child=node->getFirstChild();child!=NULL;child=child->getNextSibling())
            addStringValueToBuffer(child,buffer);
    }
}

//The dm:string-value accessor returns the node's string
//representation. For some kinds of nodes, this is part of the node;
//for other kinds of nodes, it is computed from the dm:string-value of
//its descendant nodes.


const XMLCh* NodeImpl::dmStringValue(const DynamicContext* context) const {

	switch(fNode->getNodeType())
	{
	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE:
		{
			// Returns the concatenation of the string-values of all its text node descendants in document order
			XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer str(1023, context->getMemoryManager());
			addStringValueToBuffer(fNode,str);
		    return context->getMemoryManager()->getPooledString(str.getRawBuffer());
		}
        break;

	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE:
		{
			XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer str(1023, context->getMemoryManager());
            const XMLCh* typeUri, *typeName;
            getTypeUriAndName(typeUri,typeName);
		    //Returns the string value calculated as follows:
		    // - If the element has a type of xdt:untyped, a complex type with complex content, or a complex type with mixed content, 
		    //   returns the concatenation of the string-values of all its text node descendants in document order. 
		    //   It returns "" if the element has no text node descendants.
		    // - If the element has a complex type with empty content, returns "".
		    if(XPath2Utils::equals(typeName, DocumentCacheParser::g_szUntyped) && XPath2Utils::equals(typeUri, FunctionConstructor::XMLChXPath2DatatypesURI)
		       || context->getDocumentCache()->getComplexTypeInfo(typeUri, typeName)!=NULL)
			    addStringValueToBuffer(fNode,str);
		    else
		    {
			    // - If the element has a simple type or a complex type with simple content:
			    //     - If the element type is xs:string, or a type derived from xs:string, returns that string.
			    if(context->getDocumentCache()->isTypeOrDerivedFromType(typeUri,typeName,XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_STRING))
				    str.set(fNode->getTextContent());
			    //     - If the element type is xs:anyURI, returns the characters of the URI.
			    else if(XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYURI) &&
					    XPath2Utils::equals(typeUri, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA))
				    str.set(fNode->getTextContent());
			    //     - If the element type is xs:QName returns the value calculated as follows:
			    //          - If the value has no namespace URI and the in-scope namespaces map the default namespace to any 
			    //            namespace URI, then an error is raised ("default namespace is defined").
			    //          - If the value has a namespace URI, then there must be at least one prefix mapped to that URI in 
			    //            the in-scope namespaces. If there is no such prefix, an error is raised ("no prefix defined for namespace"). 
			    //            If there is more than one such prefix, the one that is chosen is implementation dependent.
			    //          - If no error occurs, returns a string with the lexical form of a xs:QName using the prefix chosen 
			    //            as described above, and the local name of the value.
			    else if(XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_QNAME) &&
					    XPath2Utils::equals(typeUri, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA))
			    {
				    // REVISIT: the rules assume that a QName has been expanded upon loading, but we don't do that...
				    str.set(fNode->getTextContent());
			    }
			    //     - If the element type is xs:dateTime, xs:date, or xs:time, returns the original lexical representation of 
			    //       the typed value recovered as follows: if an explicit timezone was present, the normalized value is adjusted 
			    //       using the explicit timezone; if an explicit timezone was not present, the Z is dropped from the normalized value. 
			    //       The normalized value and the explicit timezone, if present, are converted separately to xs:string and concatenated 
			    //       to yield the string value.
			    else if(XPath2Utils::equals(typeUri, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA) &&
					    (XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DATETIME) ||
					     XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DATE) ||
					     XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_TIME)))
			    {
				    const AnyAtomicType::Ptr item = DatatypeFactory::STR2AT::createDerivedFromAtomicType(typeUri, typeName, fNode->getTextContent(), context);
				    str.set(item->asString(context));
			    }
			    //     - In all other cases, returns the concatenation of the string-values of all its text node descendants in document order.
			    else
				    addStringValueToBuffer(fNode,str);
		    }
		    return context->getMemoryManager()->getPooledString(str.getRawBuffer());
		}
        break;

	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE:
        {
            const XMLCh* typeUri, *typeName;
            getTypeUriAndName(typeUri,typeName);
            // Returns the value calculated as follows:
            //  - If the attribute type is xdt:untypedAtomic, xs:string, or a type derived from xs:string, returns that string.
            if(XPath2Utils::equals(typeName, ATUntypedAtomic::fgDT_UNTYPEDATOMIC) &&
               XPath2Utils::equals(typeUri, FunctionConstructor::XMLChXPath2DatatypesURI) ||
               context->getDocumentCache()->isTypeOrDerivedFromType(typeUri,typeName,XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_STRING))
                return fNode->getTextContent();
            //  - If the attribute type is xs:anyURI, returns the characters of the URI.
            else if(XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYURI) &&
                    XPath2Utils::equals(typeUri, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA))
                return fNode->getTextContent();
            //  - If the attribute type is xs:QName returns the value calculated as follows:
            //      - If the value has no namespace URI, then an error is raised ("default namespace is defined") if the in-scope 
            //        namespaces map the default namespace to any namespace URI.
            //      - If the value has a namespace URI, then there must be at least one prefix mapped to that URI in the in-scope 
            //        namespaces. If there is no such prefix, an error is raised ("no prefix defined for namespace"). 
            //        If there is more than one such prefix, the one that is chosen is implementation dependent.
            //      - If no error occurs, returns a string with the lexical form of a xs:QName using the prefix chosen as described 
            //        above, and the local name of the value.
            else if(XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_QNAME) &&
                    XPath2Utils::equals(typeUri, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA))
            {
                // REVISIT: the rules assume that a QName has been expanded upon loading, but we don't do that...
                return fNode->getTextContent();
            }
            //  - If the attribute type is xs:dateTime, xs:date, or xs:time, returns the original lexical representation recovered as follows: 
            //      - if an explicit timezone was present, the normalized value is adjusted using the explicit timezone; 
            //      - if an explicit timezone was not present, the Z is dropped from the normalized value. 
            //        The normalized value and the explicit timezone, if present, are converted separately to xs:string and concatenated to yield the string value.
            else if(XPath2Utils::equals(typeUri, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA) &&
                    (XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DATETIME) ||
                     XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DATE) ||
                     XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_TIME)))
            {
                const AnyAtomicType::Ptr item = DatatypeFactory::STR2AT::createDerivedFromAtomicType(typeUri, typeName, fNode->getTextContent(), context);
                return item->asString(context);
            }
            else
                return fNode->getTextContent();
        }
        break;
	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::PROCESSING_INSTRUCTION_NODE:
	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::COMMENT_NODE:
	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE:
	case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::CDATA_SECTION_NODE:
        // Returns the value of the content property.
        return fNode->getTextContent();
        break;

	case XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNamespace::XPATH_NAMESPACE_NODE:
        // Returns the value of the uri property.
        return fNode->getTextContent();
        break;

	}
	return XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString;
}



Sequence NodeImpl::getListTypeTypedValue(XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator *dtv, const DynamicContext* context) const {

    const XMLCh *stringVal = dmStringValue(context);

    //the actual type we want
    XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator* theItemTypeDTV = ((XERCES_CPP_NAMESPACE_QUALIFIER ListDatatypeValidator*)dtv)->getItemTypeDTV();
    XERCES_CPP_NAMESPACE_QUALIFIER BaseRefVectorOf<XMLCh>* tokenVector = XERCES_CPP_NAMESPACE_QUALIFIER XMLString::tokenizeString(stringVal);


    const XMLCh* itemTypeDTVName = theItemTypeDTV->getTypeLocalName();
    const XMLCh* itemTypeDTVUri = theItemTypeDTV->getTypeUri();
    Sequence s(tokenVector->size(), context->getMemoryManager());


    for ( unsigned int j = 0; j < tokenVector->size(); j++ )
        s.addItem(DatatypeFactory::STR2AT::createDerivedFromAtomicType(itemTypeDTVUri, itemTypeDTVName, tokenVector->elementAt(j), context));

    return s;
 
}


Sequence NodeImpl::dmTypedValue(const DynamicContext* context) const {
    /*
    cerr<<"NodeImpl::dmTypedValue getTypeName(): "<<XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(getTypeName())<<endl;
    cerr<<"NodeImpl::dmTypedValue getTypeURI():  "<<XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(getTypeURI())<<endl;

    cerr<<"NodeImpl::dmTypedValue nodenaem:  "<<XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(fNode->getNodeName())<<endl;


    cerr<<"NodeImpl::dmTypedValue uri:  "<<XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(fNode->getNamespaceURI())<<endl;

    cerr << "\n\n" << endl;
    */
    switch(fNode->getNodeType())
    {
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE:
        {
            // Data Model,  7.3.3 and 7.3.4
            const XMLCh* typeUri, *typeName;
            getMemberTypeUriAndName(typeUri,typeName);
            Item::Ptr item = 0;

            //cerr<<"NodeImpl::dmTypedValue getTypeName(): "<<XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(typeName)<<endl;
            //cerr<<"NodeImpl::dmTypedValue getTypeURI():  "<<XERCES_CPP_NAMESPACE_QUALIFIER XMLString::transcode(typeUri)<<endl;

            // If the attribute is of type xdt:untypedAtomic: its typed-value is its dm:string-value as an xdt:untypedAtomic
            if(XPath2Utils::equals(typeName, ATUntypedAtomic::fgDT_UNTYPEDATOMIC) &&
               XPath2Utils::equals(typeUri, FunctionConstructor::XMLChXPath2DatatypesURI))
            {
                item = (const Item::Ptr)DatatypeFactory::STR2AT::createUntypedAtomic(dmStringValue(context), context);
                return Sequence(item, context->getMemoryManager()); 
            } 
            else 
            {
                // Otherwise: its typed-value is a sequence of zero or more atomic values derived from the string-value of 
                // the node and its type in a way that is consistent with XML Schema validation. The type of each atomic value 
                // is assigned as described in 3.3.1.2 Atomic Value Type Names
                XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator *dtv = context->getDocumentCache()->getDatatypeValidator(typeUri, typeName);

                if(!dtv) 
                {
                    assert(false);
                    XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer msg(1023, context->getMemoryManager());
                    msg.set(X("The datatype {"));
                    msg.append(typeUri);
                    msg.append(X("}"));
                    msg.append(typeName);
                    msg.append(X(" is unknown"));
                    DSLthrow(XPath2TypeCastException,X("NodeImpl::dmTypedValue"), msg.getRawBuffer());
                } 

                if(dtv->getType() == XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator::List)
                    return getListTypeTypedValue(dtv, context);
                else 
                {
                    item = (const Item::Ptr)DatatypeFactory::STR2AT::createDerivedFromAtomicType(typeUri, typeName, dmStringValue(context), context);
                    return Sequence(item, context->getMemoryManager()); 
                }
            }
        }
        break;
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE:
        {
            // Data Model,  7.2.3 and 7.2.4
            // If the nilled property is true, its typed-value is ().
            if(((const ATBooleanOrDerived*)(const Item*)(dmNilled(context).first()))->isTrue())
                return Sequence(context->getMemoryManager());

            const XMLCh* typeUri, *typeName;
            getMemberTypeUriAndName(typeUri,typeName);

            // If the element is of type xdt:untyped or xs:anyType, its typed-value is its dm:string-value as an xdt:untypedAtomic.
            if((XPath2Utils::equals(typeName, DocumentCacheParser::g_szUntyped) && XPath2Utils::equals(typeUri, FunctionConstructor::XMLChXPath2DatatypesURI)) ||
               (XPath2Utils::equals(typeName, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgATTVAL_ANYTYPE) && XPath2Utils::equals(typeUri, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA))
              )
            {
                const AnyAtomicType::Ptr item = DatatypeFactory::STR2AT::createUntypedAtomic(dmStringValue(context), context);
                return Sequence(item, context->getMemoryManager());
            }

            // If the element has a simple type or a complex type with simple content: its typed value is compute 
            // as described in 3.3.1.2 Atomic Value Type Names. The result is a sequence of zero or more atomic values 
            // derived from the string-value of the node and its type in a way that is consistent with XML Schema validation.
            XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator *dtv = context->getDocumentCache()->getDatatypeValidator(typeUri, typeName);
            if(dtv) 
            {
                if(dtv->getType() == XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator::List)
                    return getListTypeTypedValue(dtv, context);
                else 
                {
                    const AnyAtomicType::Ptr item = DatatypeFactory::STR2AT::createDerivedFromAtomicType(typeUri, typeName, dmStringValue(context), context);
                    return Sequence(item, context->getMemoryManager()); 
                }
            }
            XERCES_CPP_NAMESPACE_QUALIFIER ComplexTypeInfo *cti = context->getDocumentCache()->getComplexTypeInfo(typeUri, typeName);
            if(cti) 
            {
                if(cti->getContentType() == XERCES_CPP_NAMESPACE_QUALIFIER SchemaElementDecl::Simple) 
                {
                    XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator *dtv = cti->getDatatypeValidator();
                    assert(dtv != 0);

                    if(dtv->getType() == XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator::List)
                        return getListTypeTypedValue(dtv, context);
                    else 
                    {
                        const AnyAtomicType::Ptr item = DatatypeFactory::STR2AT::createDerivedFromAtomicType(dtv->getTypeUri(), dtv->getTypeLocalName(), dmStringValue(context), context);
                        return Sequence(item, context->getMemoryManager());
                    }
                } 

                // If the element is empty: its typed-value is the empty sequence, ().
                if(cti->getContentType() == XERCES_CPP_NAMESPACE_QUALIFIER SchemaElementDecl::Empty)
                    return Sequence(context->getMemoryManager());
        
                // If the element has a complex type with mixed content, its typed-value is its dm:string-value as an xdt:untypedAtomic.
                if(cti->getContentType() == XERCES_CPP_NAMESPACE_QUALIFIER SchemaElementDecl::Mixed_Simple || 
                   cti->getContentType() == XERCES_CPP_NAMESPACE_QUALIFIER SchemaElementDecl::Mixed_Complex)
                {
                    const AnyAtomicType::Ptr item = DatatypeFactory::STR2AT::createUntypedAtomic(dmStringValue(context), context);
                    return Sequence(item, context->getMemoryManager());
                }

                // Otherwise, the element must be a complex type with element-only content. 
                // The typed-value of such an element is undefined. Attempting to access this property with the dm:typed-value 
                // accessor always raises an error.
                if(cti->getContentType() == XERCES_CPP_NAMESPACE_QUALIFIER SchemaElementDecl::Children) 
                    throw PathanException(XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathException::TYPE_ERR, X("Attempt to get typed value from a complex type with non-mixed complex content"));
            }
            else 
            {
                //throw error here?? Should not happen now, but if we stop giving anytype to invalid in DOMTypeInfo
                //then we will get here if its null.
                assert(false);
            }
        }
        break;
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE:
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE:
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::CDATA_SECTION_NODE:
        {
            const XMLCh *stringValue = dmStringValue(context);
            const AnyAtomicType::Ptr untypedAtomic = DatatypeFactory::STR2AT::createUntypedAtomic(stringValue, context);
            return Sequence(untypedAtomic, context->getMemoryManager()); 
        }
        break;
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNamespace::XPATH_NAMESPACE_NODE:
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::COMMENT_NODE:
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::PROCESSING_INSTRUCTION_NODE:
        {
            const XMLCh *stringValue = dmStringValue(context);
            const AnyAtomicType::Ptr untypedAtomic = DatatypeFactory::STR2AT::createString(stringValue, context);
            return Sequence(untypedAtomic, context->getMemoryManager());
        }
        break;
    }
    return Sequence(context->getMemoryManager());
}

Sequence NodeImpl::dmDocumentURI(const DynamicContext* context) const {
  const XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument* doc;
	const XMLCh* docURI;
  const XMLCh* prefix;
	switch (fNode->getNodeType()) {
    case XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE :
      doc = (const XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *)fNode;
		  docURI = doc->getDocumentURI();
      prefix = XPath2NSUtils::getPrefix(docURI, context->getMemoryManager());
      if (!XPath2Utils::equals(prefix, XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString)) {
        return Sequence(DatatypeFactory::STR2AT::createString(docURI, context), context->getMemoryManager());
      } else {
  			return Sequence(context->getMemoryManager());
      }
		default :
			return Sequence(context->getMemoryManager());
	}
}

Sequence NodeImpl::dmTypeName(const DynamicContext* context) const {
    short nodeType=fNode->getNodeType();
    if(nodeType != XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE &&
       nodeType != XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE &&
       nodeType != XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE &&
       nodeType != XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::CDATA_SECTION_NODE) {
        return Sequence(context->getMemoryManager());
    }
 

  //otherwise return the Qname for the node 
  const XMLCh* typeUri, *typeName;
  getTypeUriAndName(typeUri,typeName);
  return Sequence(DatatypeFactory::STR2AT::createQName(typeName, typeUri, context), context->getMemoryManager());
}

Sequence NodeImpl::dmNilled(const DynamicContext* context) const
{
    if(fNode->getNodeType() != XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE) {
        return Sequence(context->getMemoryManager());
    }

    // If the [validity] property exists on an Element Node and is "valid" then if the [nil] property exists and is true, 
    // then nilled property is "true". In all other cases, including all cases where schema validity assessment was not 
    // attempted or did not succeed, the nilled property is "false".
    try
    {
        XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo* psviType=(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo*)const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(fNode)->getInterface(XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXercescInterfacePSVITypeInfo);
        if(psviType && psviType->getNumericProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Validity)==XERCES_CPP_NAMESPACE_QUALIFIER PSVIItem::VALIDITY_VALID)
        {
            bool isNil=(psviType->getNumericProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Nil)!=0);
            return Sequence(DatatypeFactory::POD2AT::createBoolean(isNil, context), context->getMemoryManager());
        }
    } catch(XERCES_CPP_NAMESPACE_QUALIFIER DOMException&) {
        // ignore it; the implementation of getInterface for Xerces < 2.6 will throw it
    }
    return Sequence(DatatypeFactory::POD2AT::createBoolean(false, context), context->getMemoryManager());
}

bool NodeImpl::lessThan(const Node::Ptr &other) const {

  const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* otherNode = other->getDOMNode();
  short pos = const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *>(fNode)->compareTreePosition(const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *>(otherNode));
  // compareTreePosition returns the position of the other node, compared to my position; so, if it sets the bit
  // TREE_POSITION_FOLLOWING, it means that we come before it
  if(pos & XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TREE_POSITION_FOLLOWING)
    return true;
  if(pos & XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TREE_POSITION_PRECEDING)
    return false;
  if(pos & XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TREE_POSITION_SAME_NODE)
	  return false;
  // if the two nodes are attributes or namespaces, we get that they are equivalent; we need a stable ordering, so
  // we resort to comparing their pointers (we could even decide to sort them on their names...)
  if(pos & XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TREE_POSITION_EQUIVALENT)
    return fNode < otherNode;
  // if we get they are disconnected, it could be they belong to different documents; in this case, order them according
  // to the pointer of their documents (all the nodes in document A must come before or after all the nodes in document B,
  // regardless of the chosen ordering criteria)
  // If they belong to the same document, they are floating, or maybe just one of them is floating; let's state we consider
  // the one connected coming before the disconnected one, and, if both are disconnected, we compare the two roots
  if(pos == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TREE_POSITION_DISCONNECTED)
  {
    if(fNode->getOwnerDocument()!=otherNode->getOwnerDocument())
      return fNode->getOwnerDocument()<otherNode->getOwnerDocument();
    else
    {
      const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* root1=FunctionRoot::root(fNode);
      const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* root2=FunctionRoot::root(otherNode);
      if(root1==fNode->getOwnerDocument())
        return true;
      else
        return root1<root2;
    }
  }
  assert(false);
  return true;
}

bool NodeImpl::equals(const Node::Ptr &other) const {
  return fNode->isSameNode(other->getDOMNode());
}

const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* NodeImpl::getDOMNode() const {
  return fNode;
}

void NodeImpl::getMemberTypeUriAndName(const XMLCh*& uri, const XMLCh*& name) const
{
    short nodeType=fNode->getNodeType();
    try
    {
        if (nodeType == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE ||
            nodeType == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE) 
        {
            // check if we have PSVI info
            XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo* psviType=(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo*)const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(fNode)->getInterface(XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXercescInterfacePSVITypeInfo);
            if(psviType)
            {
                // check if we have PSVI info for the content of a union-validated type
                const XMLCh* memberName=psviType->getStringProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Member_Type_Definition_Name);
                if(memberName)
                {
                    uri=psviType->getStringProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Member_Type_Definition_Namespace);
                    name=memberName;
                    return;
                }
            }
        }
    } catch(XERCES_CPP_NAMESPACE_QUALIFIER DOMException&) {
        // ignore it; the implementation of getInterface for Xerces < 2.6 will throw it
    }
    // we are not a union, return the type of the node
    getTypeUriAndName(uri, name);
}

void NodeImpl::getTypeUriAndName(const XMLCh*& uri, const XMLCh*& name) const
{
    short nodeType=fNode->getNodeType();
    if (nodeType == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ELEMENT_NODE) {
        // check if we have PSVI info
        try
        {
            XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo* psviType=(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo*)const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(fNode)->getInterface(XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXercescInterfacePSVITypeInfo);
            if(psviType && psviType->getNumericProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Validity)==XERCES_CPP_NAMESPACE_QUALIFIER PSVIItem::VALIDITY_VALID)
            {
                uri=psviType->getStringProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Type_Definition_Namespace);
                name=psviType->getStringProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Type_Definition_Name);
                return;
            }
        } catch(XERCES_CPP_NAMESPACE_QUALIFIER DOMException&) {
            // ignore it; the implementation of getInterface for Xerces < 2.6 will throw it
        }
        // we are xdt:untyped
        uri=FunctionConstructor::XMLChXPath2DatatypesURI;
        name=DocumentCacheParser::g_szUntyped;
        return;
    }
    else if (nodeType == XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::ATTRIBUTE_NODE) {
        // check if we have PSVI info
        try
        {
            XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo* psviType=(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo*)const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DOMNode*>(fNode)->getInterface(XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgXercescInterfacePSVITypeInfo);
            if(psviType && psviType->getNumericProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Validity)==XERCES_CPP_NAMESPACE_QUALIFIER PSVIItem::VALIDITY_VALID)
            {
                uri=psviType->getStringProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Type_Definition_Namespace);
                name=psviType->getStringProperty(XERCES_CPP_NAMESPACE_QUALIFIER DOMPSVITypeInfo::PSVI_Type_Definition_Name);
                return;
            }
        } catch(XERCES_CPP_NAMESPACE_QUALIFIER DOMException&) {
            // ignore it; the implementation of getInterface for Xerces < 2.6 will throw it
        }
        // check if we have type informations coming from a DTD
        const XERCES_CPP_NAMESPACE_QUALIFIER DOMTypeInfo* pTypeInfo=((XERCES_CPP_NAMESPACE_QUALIFIER DOMAttr*)fNode)->getTypeInfo();
        const XMLCh* szUri=pTypeInfo->getNamespace();
        if(szUri==0 || szUri[0]==0)
        {
            // in these cases, we are xs:*
            const XMLCh* szName=pTypeInfo->getName();
            if(XPath2Utils::equals(szName,XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgIDString) ||
               XPath2Utils::equals(szName,XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgIDRefString) ||
               XPath2Utils::equals(szName,XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgIDRefsString) ||
               XPath2Utils::equals(szName,XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgEntityString) ||
               XPath2Utils::equals(szName,XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgEntitiesString) ||
               XPath2Utils::equals(szName,XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgNmTokenString) ||
               XPath2Utils::equals(szName,XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgNmTokensString))
            {
                uri=XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA;
                name=szName;
                return;
            }
        }
        // we are xdt:untypedAtomic
        uri=FunctionConstructor::XMLChXPath2DatatypesURI;
        name=ATUntypedAtomic::fgDT_UNTYPEDATOMIC;
        return;
    }
    else if(nodeType != XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::TEXT_NODE || 
            nodeType != XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::CDATA_SECTION_NODE) {
        uri=FunctionConstructor::XMLChXPath2DatatypesURI;
        name=ATUntypedAtomic::fgDT_UNTYPEDATOMIC;
        return;
    }
    DSLthrow(ItemException, X("NodeImpl::getTypeUriAndName"), X("Tried to get type informations on Node other than DOMElement, DOMAttribute or DOMText"));
}

const XMLCh* NodeImpl::getTypeName() const {
    const XMLCh* uri, *name;
    getTypeUriAndName(uri,name);
    return name;
}

const XMLCh* NodeImpl::getTypeURI() const {
    const XMLCh* uri, *name;
    getTypeUriAndName(uri,name);
    return uri;
}

