/*
 * Copyright (c) 2001, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../../config/pathan_config.h"
#include "ATQNameOrDerivedImpl.hpp"
#include <pathan/ATStringOrDerived.hpp>
#include <pathan/XPath2Utils.hpp>
#include <pathan/XPath2NSUtils.hpp>
#include <pathan/Sequence.hpp>
#include <pathan/internal/factory/DatatypeFactory.hpp>
#include <pathan/exceptions/IllegalArgumentException.hpp>
#include <pathan/exceptions/NamespaceLookupException.hpp>
#include "../../exceptions/InvalidLexicalSpaceException.hpp"
#include <pathan/exceptions/TypeNotFoundException.hpp>
#include "../../exceptions/XPath2TypeCastException.hpp"
#include <xercesc/util/XMLUni.hpp>
#include <xercesc/util/XMLUniDefs.hpp>
#include <xercesc/framework/XMLBuffer.hpp>
#include <xercesc/validators/schema/SchemaSymbols.hpp>
#include <pathan/internal/DOMutils/XStr.hpp> // defines X() and XMLCh*
#include <pathan/XPath2MemoryManager.hpp>
#include <pathan/DynamicContext.hpp>
#include <xercesc/util/XMLString.hpp>

ATQNameOrDerivedImpl::
ATQNameOrDerivedImpl(const XMLCh* typeURI, const XMLCh* typeName, const XMLCh* value, const StaticContext* context): 
    ATQNameOrDerived(),
    _typeName(typeName),
    _typeURI(typeURI) { 
  
  const XMLCh* prefix = XPath2NSUtils::getPrefix(value, context->getMemoryManager());
  const XMLCh* uri;
  if(XPath2Utils::equals(prefix, XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString)) {
  // If $qname has no prefix
    uri = context->getDefaultElementAndTypeNS();
  }
  // If $qname has a prefix
  else {
    uri = context->getUriBoundToPrefix(prefix);
    if (uri == 0) {
       DSLthrow(IllegalArgumentException, X("ATQNameOrDerivedImpl::ATQNameOrDerivedImpl"),X("No namespace for prefix"));
    }
  }

  // _uri will be null if there is no default NS
  _uri = context->getMemoryManager()->getPooledString(uri);
  _name = context->getMemoryManager()->getPooledString(XPath2NSUtils::getLocalName(value));

}

ATQNameOrDerivedImpl::
ATQNameOrDerivedImpl(const XMLCh* typeURI, const XMLCh* typeName, const XMLCh* uri, const XMLCh* name, const StaticContext* context): 
    ATQNameOrDerived(),
    _typeName(typeName),
    _typeURI(typeURI) { 
  
  // if uri is NULL, so will _uri
  _uri = context->getMemoryManager()->getPooledString(uri);
  _name = context->getMemoryManager()->getPooledString(name);

}


/* Get the name of the primitive type (basic type) of this type
 * (ie "decimal" for xs:decimal) */
const XMLCh* ATQNameOrDerivedImpl::getPrimitiveTypeName() const {
  return this->getPrimitiveName();
}

/* Get the name of this type  (ie "integer" for xs:integer) */
const XMLCh* ATQNameOrDerivedImpl::getTypeName() const {
  return _typeName;
}

/* Get the namespace URI for this type */
const XMLCh* ATQNameOrDerivedImpl::getTypeURI() const {
  return _typeURI; 
}

/* returns the URI */
const XMLCh* ATQNameOrDerivedImpl::getURI(void) const {
  return _uri;
}

/* returns the name */
const XMLCh* ATQNameOrDerivedImpl::getName(void) const {
  return _name;
}


AnyAtomicType::AtomicObjectType ATQNameOrDerivedImpl::getTypeIndex() {
  return AnyAtomicType::QNAME;
} 

/* If possible, cast this type to the target type */
AnyAtomicType::Ptr ATQNameOrDerivedImpl::castAsInternal(const XMLCh* targetURI, const XMLCh* targetType, const DynamicContext* context) const {
  const DatatypeFactory* target = context->getDatatypeFactory(targetURI, targetType); 

  AnyAtomicType::AtomicObjectType targetIndex = target->getPrimitiveTypeIndex();
  switch (targetIndex) {
    case ANY_SIMPLE_TYPE:
    case UNTYPED_ATOMIC:
      //anySimpleType and untypedAtomic follow the same casting rules as string.
    case STRING: {
      return DatatypeFactory::STR2AT::createDerivedFromAtomicType(targetURI, targetType, this->asLexicalString(context), context);
    } 
    case QNAME: {
      return DatatypeFactory::STR2AT::createQNameOrDerived(targetURI, targetType, _uri, _name, context);
    }
    default: return AnyAtomicType::castAsInternal(targetURI, targetType, context);
  }
}

/* returns the XMLCh* (canonical) representation of this type */
const XMLCh* ATQNameOrDerivedImpl::asString(const DynamicContext* context) const {
  XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer buffer(1023, context->getMemoryManager());
  buffer.set(_uri);
  buffer.append(XERCES_CPP_NAMESPACE_QUALIFIER chColon);
  buffer.append(_name);

  return context->getMemoryManager()->getPooledString(buffer.getRawBuffer());
}

/* returns the XMLCh* (lexical => prefix:localname) representation of this type */
const XMLCh* ATQNameOrDerivedImpl::asLexicalString(const DynamicContext* context) const {
  XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer buffer(1023, context->getMemoryManager());
  const XMLCh* prefix;
  if (this->_uri == 0) {
    if(context->getDefaultElementAndTypeNS() != 0) {
      DSLthrow(NamespaceLookupException,X("ATQNameOrDerivedImpl::asLexicalString"), X("default namespace is defined"));
    }
  } 
  else {
    prefix = context->getPrefixBoundToUri(this->_uri);
    if (prefix == 0) {
      DSLthrow(NamespaceLookupException,X("ATQNameOrDerivedImpl::asLexicalString"), X("no prefix defined for namespace"));      
    }
    buffer.set(prefix);
    buffer.append(XERCES_CPP_NAMESPACE_QUALIFIER chColon);
    buffer.append(_name);    
  }
  return context->getMemoryManager()->getPooledString(buffer.getRawBuffer());
}

/* returns true if the two objects' URI are equal (string comparison)
   * false otherwise */
bool ATQNameOrDerivedImpl::equals(const AnyAtomicType::Ptr &target, const DynamicContext* context) const {
  if(this->getPrimitiveTypeIndex() != target->getPrimitiveTypeIndex()) {
    DSLthrow(IllegalArgumentException,X("ATQNameOrDerivedImpl::equals"), X("Equality operator for given types not supported"));
  }

  return XPath2Utils::equals(((ATQNameOrDerivedImpl*)(const AnyAtomicType*)target)->_name, _name) &&
         XPath2Utils::equals(((ATQNameOrDerivedImpl*)(const AnyAtomicType*)target)->_uri, _uri) ;  
  
}

const XMLCh* ATQNameOrDerivedImpl::getPrimitiveName()  {
  return XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_QNAME;
}

AnyAtomicType::AtomicObjectType ATQNameOrDerivedImpl::getPrimitiveTypeIndex() const {
    return this->getTypeIndex();
}
