/*
 * Copyright (c) 2001, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../config/pathan_config.h"
#include <pathan/functions/FunctionMin.hpp>
#include <pathan/Sequence.hpp>
#include <pathan/dataItem/DataItemSequence.hpp>
#include <pathan/Collation.hpp>
#include <pathan/internal/collations/CodepointCollation.hpp>
#include "../operators/TotalOrderComparison.hpp"
#include <pathan/DynamicContext.hpp>
#include <pathan/exceptions/FunctionException.hpp>
#include <pathan/exceptions/XPath2ErrorException.hpp>
#include <pathan/exceptions/IllegalArgumentException.hpp>
#include <pathan/ATStringOrDerived.hpp>
#include <pathan/ATBooleanOrDerived.hpp>
#include <pathan/ATDecimalOrDerived.hpp>
#include <pathan/internal/factory/DatatypeFactory.hpp>
#include <xercesc/validators/schema/SchemaSymbols.hpp>

const XMLCh FunctionMin::name[] = {
XERCES_CPP_NAMESPACE_QUALIFIER chLatin_m, XERCES_CPP_NAMESPACE_QUALIFIER chLatin_i, XERCES_CPP_NAMESPACE_QUALIFIER chLatin_n, XERCES_CPP_NAMESPACE_QUALIFIER chNull };

/**
 * fn:min($arg as xdt:anyAtomicType*) as xdt:anyAtomicType?
 * fn:min($arg as xdt:anyAtomicType*, $collation as string) as xdt:anyAtomicType?
**/

FunctionMin::FunctionMin(const VectorOfDataItems &args, XPath2MemoryManager* memMgr)
  : AggregateFunction(name,1, 2, "anyAtomicType*, string", args, memMgr)
{
}

Sequence FunctionMin::collapseTreeInternal(DynamicContext* context, int flags) const
{
    XPath2MemoryManager* memMgr = context->getMemoryManager();

    Sequence sequence(memMgr);  
    try {
        sequence = validateSequence(getParamNumber(1,context,DataItem::UNORDERED), context);
    } catch (IllegalArgumentException &e) {
        DSLthrow(IllegalArgumentException, X("FunctionMin::collapseTreeInternal"), X("Invalid argument to fn:min() function"));
    }

    // Return the empty sequence if the sequence is empty
    if(sequence.isEmpty()) {
        return Sequence(memMgr);
    }
    if(sequence.getLength()==1 && isNumericNaN(sequence.first()))
        return sequence;

    Collation* collation=NULL;
    if (getNumArgs()>1) {
        Sequence collArg = getParamNumber(2,context);
        const XMLCh* collName = collArg.first()->asString(context);
        try {
            DatatypeFactory::STR2AT::createAnyURI(collName, context);
        } catch(XPath2ErrorException &e) {
            DSLthrow(FunctionException, X("FunctionMin::collapseTreeInternal"), X("Invalid collationURI"));  
        }
        collation=context->getCollation(collName);
        if (collation==NULL)
            DSLthrow(FunctionException,X("FunctionMin::collapseTreeInternal"),X("Collation object is not available"));
    }
    else
        collation=context->getDefaultCollation();
    if(collation==NULL)
        collation=context->getCollation(CodepointCollation::getCodepointCollationName());

    Sequence::iterator i = sequence.begin();
    Item::Ptr minItem = *i;

    if(((AnyAtomicType*)(const Item*)(*i))->getPrimitiveTypeIndex() == AnyAtomicType::STRING) {
        for (++i; i != sequence.end(); i++) {
          if(collation->compare((*i)->asString(context),minItem->asString(context))<0) {
            minItem = *i;
          }
        }
    } else {
        ATBooleanOrDerived::Ptr less;
        for (++i; i != sequence.end(); i++) {
            VectorOfDataItems gtArgs = VectorOfDataItems(PathanAllocator<DataItem*>(memMgr));
            DataItemSequence seq1(*i, context, memMgr);
            gtArgs.push_back(&seq1);
            DataItemSequence seq2(minItem, context, memMgr);
            gtArgs.push_back(&seq2);
            TotalOrderComparison gt(gtArgs, false, memMgr);
            try {
                less = (const ATBooleanOrDerived::Ptr )gt.collapseTree(context).next(context);
            } catch (IllegalArgumentException &e) {
                DSLthrow(IllegalArgumentException, X("FunctionMin::collapseTreeInternal"), X("Invalid argument to fn:min() function"));
            } catch (XPath2ErrorException &e) {
                DSLthrow(IllegalArgumentException, X("FunctionMin::collapseTreeInternal"), X("Invalid argument to fn:min() function"));
            }
            if(((const ATBooleanOrDerived*)less)->isTrue()) {
              minItem = *i;
            }
        }
    }
    return Sequence(minItem, memMgr);
}
