/*
 * Copyright (c) 2001, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../config/pathan_config.h"
#include <pathan/Pathan.hpp>
#include <pathan/functionAPI/DatatypeLookup.hpp>
#include <xercesc/util/XMLUniDefs.hpp>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/validators/datatype/DatatypeValidator.hpp>
#include <xercesc/validators/schema/SchemaSymbols.hpp>
#include <xercesc/framework/XMLBuffer.hpp>

#include <pathan/XPath2Utils.hpp>
#include <pathan/XPath2MemoryManager.hpp>
#include <pathan/internal/factory/DatatypeFactory.hpp>
#include "../factory/DatatypeFactoryTemplate.hpp"
#include <pathan/exceptions/TypeNotFoundException.hpp>
#include "../items/impl/ATAnyURIOrDerivedImpl.hpp"
#include "../items/impl/ATAnySimpleTypeImpl.hpp"
#include "../items/impl/ATBase64BinaryOrDerivedImpl.hpp"
#include "../items/impl/ATBooleanOrDerivedImpl.hpp"
#include "../items/impl/ATDateOrDerivedImpl.hpp"
#include "../items/impl/ATDateTimeOrDerivedImpl.hpp"
#include "../items/impl/ATDecimalOrDerivedImpl.hpp"
#include "../items/impl/ATDoubleOrDerivedImpl.hpp"
#include "../items/impl/ATDurationOrDerivedImpl.hpp"
#include "../items/impl/ATFloatOrDerivedImpl.hpp"
#include "../items/impl/ATGDayOrDerivedImpl.hpp"
#include "../items/impl/ATGMonthOrDerivedImpl.hpp"
#include "../items/impl/ATGMonthDayOrDerivedImpl.hpp"
#include "../items/impl/ATGYearMonthOrDerivedImpl.hpp"
#include "../items/impl/ATGYearOrDerivedImpl.hpp"
#include "../items/impl/ATHexBinaryOrDerivedImpl.hpp"
#include "../items/impl/ATNotationOrDerivedImpl.hpp"
#include "../items/impl/ATQNameOrDerivedImpl.hpp"
#include "../items/impl/ATStringOrDerivedImpl.hpp"
#include "../items/impl/ATTimeOrDerivedImpl.hpp"
#include "../items/impl/ATUntypedAtomicImpl.hpp"
#include <pathan/ATUntypedAtomic.hpp>
#include <pathan/DocumentCache.hpp>

DatatypeLookup::DatatypeLookup(const DocumentCache* dc, XPath2MemoryManager* memMgr) :
    fDatatypeTable(30, false, memMgr),
    fDocumentCache(dc),
    fMemMgr(memMgr)
{
  createTable();
}

DatatypeLookup::~DatatypeLookup()
{
}

void DatatypeLookup::insertDatatype(DatatypeFactory *datatype)
{
  fDatatypeTable.put((void*)datatype->getPrimitiveTypeName(),datatype);
}

const DatatypeFactory* DatatypeLookup::lookupDatatype(const XMLCh* typeURI, const XMLCh* typeName) const
{
  const DatatypeFactory* pFactory=fDatatypeTable.get((void*)typeName);
  
  // in case we're lucky and we were given a primitive type
  if (pFactory) {
    if(XPath2Utils::equals(typeURI, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA) ||
      XPath2Utils::equals(typeName, ATUntypedAtomic::fgDT_UNTYPEDATOMIC) ) {
      return pFactory;
    }
  }
  
  const XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator* validator = fDocumentCache->getDatatypeValidator(typeURI, typeName);

  while(validator) {
    const XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator* tempVal = validator->getBaseValidator();
    if(!tempVal) break;
    else validator = tempVal;
  }

  if(validator) {
    pFactory=fDatatypeTable.get((void*)validator->getTypeLocalName());
    if (pFactory)
      return pFactory;
  }

  XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer buf(1023, fMemMgr);
  buf.append(X("Type "));
  buf.append(typeURI);
  buf.append(XERCES_CPP_NAMESPACE_QUALIFIER chColon);
  buf.append(typeName);
  buf.append(X(" not found"));
  DSLthrow(TypeNotFoundException, X("DatatypeLookup::lookupDatatype"), buf.getRawBuffer());

}


void DatatypeLookup::createTable()
{
  // create a xs:anySimpleType
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATAnySimpleTypeImpl>(fDocumentCache));
  
  // create a xs:anyURI
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATAnyURIOrDerivedImpl>(fDocumentCache));
  
  // create a xs:base64Binary
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATBase64BinaryOrDerivedImpl>(fDocumentCache));

  // create a xs:boolean
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATBooleanOrDerivedImpl>(fDocumentCache));

  // create a xs:date
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATDateOrDerivedImpl>(fDocumentCache));

  // create a xs:dateTime
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATDateTimeOrDerivedImpl>(fDocumentCache));

  // create a xs:decimal
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATDecimalOrDerivedImpl>(fDocumentCache));

  // create a xs:double
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATDoubleOrDerivedImpl>(fDocumentCache));

  // create a xs:duration
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATDurationOrDerivedImpl>(fDocumentCache));

  // create a xs:float
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATFloatOrDerivedImpl>(fDocumentCache));

  // create a xs:gDay
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATGDayOrDerivedImpl>(fDocumentCache));

  // create a xs:gMonth
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATGMonthOrDerivedImpl>(fDocumentCache));

  // create a xs:gMonthDay
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATGMonthDayOrDerivedImpl>(fDocumentCache));

  // create a xs:gYear
   insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATGYearOrDerivedImpl>(fDocumentCache));

  // create a xs:gYearMonth
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATGYearMonthOrDerivedImpl>(fDocumentCache));

  // create a xs:hexBinary
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATHexBinaryOrDerivedImpl>(fDocumentCache));

  // create a xs:NOTATION
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATNotationOrDerivedImpl>(fDocumentCache));

  // create a xs:QName
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATQNameOrDerivedImpl>(fDocumentCache));
  
  // create a xs:string
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATStringOrDerivedImpl>(fDocumentCache));

  // create a xs:time
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATTimeOrDerivedImpl>(fDocumentCache));
  
  // create a xdt:untypedAtomic 
  insertDatatype(new (fMemMgr) DatatypeFactoryTemplate<ATUntypedAtomicImpl>(fDocumentCache));
}



