/*
 * Copyright (c) 2001, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../config/pathan_config.h"
#include <assert.h>

#include <pathan/internal/factory/DatatypeFactory.hpp>
#include <pathan/StaticContext.hpp>
#include <pathan/functions/FunctionConstructor.hpp> // for datatype-uri
#include "../items/impl/ATBooleanOrDerivedImpl.hpp"
#include "../items/impl/ATDecimalOrDerivedImpl.hpp"
#include "../items/impl/ATDoubleOrDerivedImpl.hpp"
#include "../items/impl/ATFloatOrDerivedImpl.hpp"
#include "../items/impl/ATQNameOrDerivedImpl.hpp"
#include "../items/impl/NodeImpl.hpp"
#include "../exceptions/InvalidLexicalSpaceException.hpp"
#include <pathan/ATDurationOrDerived.hpp>
#include <pathan/ATUntypedAtomic.hpp>
#include <pathan/XPath2Utils.hpp>
#include <pathan/exceptions/TypeNotFoundException.hpp>

#include <xercesc/util/XMLString.hpp>
#include <xercesc/validators/schema/SchemaSymbols.hpp>
#include <xercesc/validators/datatype/DatatypeValidator.hpp>

DatatypeFactory::DatatypeFactory(const DocumentCache* dc)
  : fDocumentCache(dc)
{
}

DatatypeFactory::~DatatypeFactory()
{
}

AnyAtomicType::Ptr DatatypeFactory::createInstance(const XMLCh* typeURI,
                                                   const XMLCh* typeName,
                                                   const XMLCh* value,
                                                   const StaticContext* context) const
{
  XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator* validator =
    const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator*>(fDocumentCache->getDatatypeValidator(typeURI, typeName));

  if(validator) {
    try {
      validator->validate(value, 0, context->getMemoryManager());
    } catch (XERCES_CPP_NAMESPACE_QUALIFIER XMLException &e) {
      DSLthrow(InvalidLexicalSpaceException, X("DatatypeFactory::createInstance"), e.getMessage());
    }
  }
  else {
    XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer buf(1023, context->getMemoryManager());
    buf.append(X("Type "));
    buf.append(typeURI);
    buf.append(XERCES_CPP_NAMESPACE_QUALIFIER chColon);
    buf.append(typeName);
    buf.append(X(" not found"));
    DSLthrow(TypeNotFoundException, X("DatatypeFactoryTemplate::createInstance"), buf.getRawBuffer());
  }

  return createInstanceNoCheck(typeURI, typeName, value, context);
}

bool DatatypeFactory::checkInstance(const XMLCh* typeURI,
                                    const XMLCh* typeName,
                                    const XMLCh* value,
                                    const StaticContext* context) const {
      
  XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator* validator =
    const_cast<XERCES_CPP_NAMESPACE_QUALIFIER DatatypeValidator*>(fDocumentCache->getDatatypeValidator(typeURI, typeName));

  if(validator) {
    try {
      validator->validate(value, 0, context->getMemoryManager());
    } catch (XERCES_CPP_NAMESPACE_QUALIFIER XMLException &e) {
      return false;
    }
  }
  else {
    XERCES_CPP_NAMESPACE_QUALIFIER XMLBuffer buf(1023, context->getMemoryManager());
    buf.append(X("Type "));
    buf.append(typeURI);
    buf.append(XERCES_CPP_NAMESPACE_QUALIFIER chColon);
    buf.append(typeName);
    buf.append(X(" not found"));
    DSLthrow(TypeNotFoundException, X("DatatypeFactoryTemplate::createInstance"), buf.getRawBuffer());
  }

  return true;
}

/////////////////////////////
// Helper creation methods //
/////////////////////////////

Node::Ptr DatatypeFactory::POD2AT::createNode(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node, const DynamicContext *context)
{
  return new NodeImpl(node, context);
}

ATQNameOrDerived::Ptr DatatypeFactory::STR2AT::createQName(const XMLCh* uri,
                                                         const XMLCh* name, 
                                                         const StaticContext* context
                                                          ) {
  return STR2AT::createQNameOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_QNAME,
      uri,
      name,
      context
      );
}

ATQNameOrDerived::Ptr DatatypeFactory::STR2AT::createQName(const XMLCh* value, 
							   const StaticContext* context
							   ) {
  return STR2AT::createQNameOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_QNAME,
      value,
      context
      );
}

ATDoubleOrDerived::Ptr DatatypeFactory::POD2AT::createDouble(const MAPM value, const StaticContext* context) {
  return POD2AT::createDoubleOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DOUBLE,
      value, context);
}

ATDoubleOrDerived::Ptr DatatypeFactory::STR2AT::createDouble(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createDoubleOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DOUBLE,
      value, context);
}

ATFloatOrDerived::Ptr DatatypeFactory::POD2AT::createFloat(const MAPM value, const StaticContext* context) {
  return POD2AT::createFloatOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_FLOAT,
      value, context);
}

ATFloatOrDerived::Ptr DatatypeFactory::STR2AT::createFloat(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createFloatOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_FLOAT,
      value, context);
}

ATDecimalOrDerived::Ptr DatatypeFactory::POD2AT::createDecimal(const MAPM value, const StaticContext* context) {
  return POD2AT::createDecimalOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DECIMAL,
      value, context);
}

ATDecimalOrDerived::Ptr DatatypeFactory::STR2AT::createDecimal(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createDecimalOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DECIMAL,
      value, context);
}

ATDecimalOrDerived::Ptr DatatypeFactory::POD2AT::createInteger(const int value, const StaticContext* context) {
  return context->getMemoryManager()->createInteger(value);
}

ATDecimalOrDerived::Ptr DatatypeFactory::POD2AT::createInteger(const MAPM value, const StaticContext* context) {
  return POD2AT::createDecimalOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_INTEGER,
      value, context);
}

ATDecimalOrDerived::Ptr DatatypeFactory::STR2AT::createInteger(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createDecimalOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_INTEGER,
      value, context);
}

ATBooleanOrDerived::Ptr DatatypeFactory::POD2AT::createBoolean(bool value, const StaticContext* context) {
  return POD2AT::createBooleanOrDerived(
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_BOOLEAN,
	  value, context);  
}

ATBooleanOrDerived::Ptr DatatypeFactory::STR2AT::createBoolean(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createBooleanOrDerived(
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_BOOLEAN,
	  value, context);  
}

ATDecimalOrDerived::Ptr DatatypeFactory::POD2AT::createNonNegativeInteger(const MAPM value, const StaticContext* context) {
  return POD2AT::createDecimalOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_NONNEGATIVEINTEGER,
      value, context);
}
      
ATDurationOrDerived::Ptr DatatypeFactory::STR2AT::createDayTimeDuration(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createDurationOrDerived(
      FunctionConstructor::XMLChXPath2DatatypesURI, 
      ATDurationOrDerived::fgDT_DAYTIMEDURATION,
      value, context);
}

ATDurationOrDerived::Ptr DatatypeFactory::STR2AT::createYearMonthDuration(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createDurationOrDerived(
      FunctionConstructor::XMLChXPath2DatatypesURI, 
      ATDurationOrDerived::fgDT_YEARMONTHDURATION,
      value, context);
}

ATDateOrDerived::Ptr DatatypeFactory::STR2AT::createDate(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createDateOrDerived(
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DATE,
	  value, context);  
}

ATDateTimeOrDerived::Ptr DatatypeFactory::STR2AT::createDateTime(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createDateTimeOrDerived(
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DATETIME,
	  value, context);  
}

ATTimeOrDerived::Ptr DatatypeFactory::STR2AT::createTime(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createTimeOrDerived(
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
	  XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_TIME,
	  value, context);  
}

ATAnyURIOrDerived::Ptr DatatypeFactory::STR2AT::createAnyURI(const XMLCh* value, const StaticContext* context) {
    return STR2AT::createAnyURIOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYURI,
      value, context);
}

ATStringOrDerived::Ptr DatatypeFactory::STR2AT::createString(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createStringOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_STRING,
      value, context);
}

ATBase64BinaryOrDerived::Ptr DatatypeFactory::STR2AT::createBase64Binary(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createBase64BinaryOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_BASE64BINARY,
      value, context);
}

ATHexBinaryOrDerived::Ptr DatatypeFactory::STR2AT::createHexBinary(const XMLCh* value, const StaticContext* context) {
  return STR2AT::createHexBinaryOrDerived(
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA,
      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_HEXBINARY,
      value, context);
}

//////////////////////////
// All creation methods //
//////////////////////////


AnyAtomicType::Ptr DatatypeFactory::STR2AT::createDerivedFromAtomicType(const XMLCh* typeURI,
                                        const XMLCh* typeName, 
                                        const XMLCh* value, const StaticContext* context) {

  const DatatypeFactory* dtf = context->getDatatypeFactory(typeURI, typeName);
  return dtf->createInstance(typeURI, typeName, value, context);
}

ATAnySimpleType::Ptr DatatypeFactory::STR2AT::createAnySimpleType(const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYSIMPLETYPE);
  return (const ATAnySimpleType::Ptr ) dtf->createInstance(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, 
                                                      XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYSIMPLETYPE, 
                                                      value, context);
}

/** create a xs:anyURI */
ATAnyURIOrDerived::Ptr DatatypeFactory::STR2AT::createAnyURIOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName, 
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {

  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYURI);
  return (const ATAnyURIOrDerived::Ptr) dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:base64Binary */
ATBase64BinaryOrDerived::Ptr DatatypeFactory::STR2AT::createBase64BinaryOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context){
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_BASE64BINARY);
  return (const ATBase64BinaryOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:boolean */
ATBooleanOrDerived::Ptr DatatypeFactory::STR2AT::createBooleanOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_BOOLEAN);
  return (const ATBooleanOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);

}

/** create a xs:boolean with a bool value */
ATBooleanOrDerived::Ptr DatatypeFactory::POD2AT::createBooleanOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  bool value, 
                                                  const StaticContext* context) {
  // No need to validate
  return new ATBooleanOrDerivedImpl(typeURI, typeName, value, context);
}

/** create a xs:date */
ATDateOrDerived::Ptr DatatypeFactory::STR2AT::createDateOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context){
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DATE);
  return (const ATDateOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}


/** create a xs:dateTime */
ATDateTimeOrDerived::Ptr DatatypeFactory::STR2AT::createDateTimeOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value,
                                                  const StaticContext* context){
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DATETIME);
  return (const ATDateTimeOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:decimal */
ATDecimalOrDerived::Ptr DatatypeFactory::STR2AT::createDecimalOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value,
                                                  const StaticContext* context){
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DECIMAL);
  return (const ATDecimalOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:decimal with a MAPM */
ATDecimalOrDerived::Ptr DatatypeFactory::POD2AT::createDecimalOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const MAPM value,
                                                  const StaticContext* context){
  // No need to validate
  return new ATDecimalOrDerivedImpl(typeURI, typeName, value, context);
}


/** create a xs:double */  
ATDoubleOrDerived::Ptr DatatypeFactory::STR2AT::createDoubleOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context){
  if(XPath2Utils::equals(value, Numeric::NAN_string)) {
	value= Numeric::NaN_string;
  }
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DOUBLE);
  return (const ATDoubleOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:double with a MAPM */
ATDoubleOrDerived::Ptr DatatypeFactory::POD2AT::createDoubleOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const MAPM value, 
                                                  const StaticContext* context){
  // No need to validate
  return  new ATDoubleOrDerivedImpl(typeURI, typeName, value, context);
}


/** create a xs:duration */
ATDurationOrDerived::Ptr DatatypeFactory::STR2AT::createDurationOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context){
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DURATION);
  return (const ATDurationOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:float */
ATFloatOrDerived::Ptr DatatypeFactory::STR2AT::createFloatOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context){
  if(XERCES_CPP_NAMESPACE_QUALIFIER XMLString::equals(value, Numeric::NAN_string)) {
	value= Numeric::NaN_string;
  }
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_FLOAT);
  return (const ATFloatOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);

}

/** create a xs:float with a MAPM */
ATFloatOrDerived::Ptr DatatypeFactory::POD2AT::createFloatOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const MAPM value, 
                                                  const StaticContext* context) {
  // No need to validate
  return  new ATFloatOrDerivedImpl(typeURI, typeName, value, context);
}

/** create a xs:gDay */
ATGDayOrDerived::Ptr DatatypeFactory::STR2AT::createGDayOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_DAY);
  return (const ATGDayOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:gMonth */
ATGMonthOrDerived::Ptr DatatypeFactory::STR2AT::createGMonthOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_MONTH);
  return (const ATGMonthOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);

}


/** create a xs:gMonthDay */
ATGMonthDayOrDerived::Ptr DatatypeFactory::STR2AT::createGMonthDayOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_MONTHDAY);
  return (const ATGMonthDayOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:gYear */
ATGYearOrDerived::Ptr DatatypeFactory::STR2AT::createGYearOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_YEAR);
  return (const ATGYearOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}


/** create a xs:gYearMonth */
ATGYearMonthOrDerived::Ptr DatatypeFactory::STR2AT::createGYearMonthOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_YEARMONTH);
  return (const ATGYearMonthOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:hexBinary */
ATHexBinaryOrDerived::Ptr DatatypeFactory::STR2AT::createHexBinaryOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_HEXBINARY);
  return (const ATHexBinaryOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:NOTATION*/
ATNotationOrDerived::Ptr DatatypeFactory::STR2AT::createDerivedFromNotation(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgNotationString);
  return (const ATNotationOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:QName */
ATQNameOrDerived::Ptr DatatypeFactory::STR2AT::createQNameOrDerived(const XMLCh* typeURI, 
                                                 const XMLCh* typeName,
                                                 const XMLCh* value, 
                                                 const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_QNAME);
  return (const ATQNameOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:QName with two parameters */
ATQNameOrDerived::Ptr DatatypeFactory::STR2AT::createQNameOrDerived(const XMLCh* typeURI, 
                                                 const XMLCh* typeName,
                                                 const XMLCh* uri,
                                                 const XMLCh* name, 
                                                 const StaticContext* context) {

  ATQNameOrDerivedImpl* tmp =  new ATQNameOrDerivedImpl(typeURI, typeName, uri, name, context);
  
  const DatatypeFactory* dtf_anyURI = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYURI);
  if(dtf_anyURI->checkInstance(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYURI, uri, context)) {
    const DatatypeFactory* dtf_NCName = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_STRING);
    if (dtf_NCName->checkInstance(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_NCNAME, name, context)) {
      return tmp;
    } else {
      // this call will obviously fail, but it is better for error reporting, 
      // since we actually get the XMLException's error message 
      return (const ATQNameOrDerived::Ptr )dtf_NCName->createInstance(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_NCNAME, name, context);
    }
  } else {
    // this call will obviously fail, but it is better for error reporting, 
    // since we actually get the XMLException's error message 
    return (const ATQNameOrDerived::Ptr )dtf_anyURI->createInstance(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_ANYURI, uri, context);
  }
}


/** create a xs:string */
ATStringOrDerived::Ptr DatatypeFactory::STR2AT::createStringOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_STRING);
  return (const ATStringOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create a xs:time */
ATTimeOrDerived::Ptr DatatypeFactory::STR2AT::createTimeOrDerived(const XMLCh* typeURI, 
                                                  const XMLCh* typeName,
                                                  const XMLCh* value, 
                                                  const StaticContext* context){
  const DatatypeFactory* dtf = context->getDatatypeFactory(XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgURI_SCHEMAFORSCHEMA, XERCES_CPP_NAMESPACE_QUALIFIER SchemaSymbols::fgDT_TIME);
  return (const ATTimeOrDerived::Ptr)dtf->createInstance(typeURI, typeName, value, context);
}

/** create an xdt:untypedAtomic */
ATUntypedAtomic::Ptr DatatypeFactory::STR2AT::createUntypedAtomic(const XMLCh* value, const StaticContext* context) {
  const DatatypeFactory* dtf = context->getDatatypeFactory(FunctionConstructor::XMLChXPath2DatatypesURI, ATUntypedAtomic::fgDT_UNTYPEDATOMIC);
  return (const ATUntypedAtomic::Ptr)dtf->createInstance(FunctionConstructor::XMLChXPath2DatatypesURI, ATUntypedAtomic::fgDT_UNTYPEDATOMIC, value, context);
}
