/*
 * Copyright (c) 2004, Parthenon Computing Limited, All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../config/pathan_config.h"
#include <sstream>

#include <pathan/dataItem/ResultImpl.hpp>
#include <pathan/dataItem/SequenceResult.hpp>
#include <pathan/DynamicContext.hpp>
#include <pathan/ATFloatOrDerived.hpp>
#include <pathan/ATDoubleOrDerived.hpp>
#include <pathan/ATBooleanOrDerived.hpp>
#include <pathan/XPath2Utils.hpp>

ResultImpl::ResultImpl(DynamicContext *context)
  : ReferenceCounted(),
    context_(context)
{
  // Do nothing
}

ResultImpl::~ResultImpl()
{
  // Do nothing
}

bool ResultImpl::getEffectiveBooleanValue(DynamicContext* context)
{
  const Item::Ptr first = next(context);
  if(first == NULLRCP) {
    return false;
  }
  return getEffectiveBooleanValue(first, next(context), context);
}

bool ResultImpl::getEffectiveBooleanValue(const Item::Ptr &first, const Item::Ptr &second, DynamicContext* context)
{
  // From $ 15.1.4 of the F&O specs:
  // The effective boolean value of an operand is defined as follows:
  //
  // If $arg is the empty sequence, returns false.

  if(first == NULLRCP) {
    return false;
  }

  // If $arg contains a single atomic value, then the function returns false if $arg is:
  if(second == NULLRCP && first->isAtomicValue()) {
      const AnyAtomicType* atom=(const AnyAtomicType*)(const Item*)first;
    // The singleton xs:boolean value false.
    if(atom->getPrimitiveTypeIndex() == AnyAtomicType::BOOLEAN && ((const ATBooleanOrDerived*)atom)->isFalse())
      return false;

    // The singleton value "" (zero-length string) of type xs:string or xdt:untypedAtomic.
    if((atom->getPrimitiveTypeIndex() == AnyAtomicType::STRING || atom->getPrimitiveTypeIndex() == AnyAtomicType::UNTYPED_ATOMIC) &&
       XPath2Utils::equals(atom->asString(context), XERCES_CPP_NAMESPACE_QUALIFIER XMLUni::fgZeroLenString))
      return false;

    // A singleton numeric value that is numerically equal to zero.
    if(atom->isNumericValue()) {
      if(((const Numeric*)atom)->isZero())
        return false;

      // The singleton xs:float or xs:double value NaN.
      if(atom->getPrimitiveTypeIndex() == AnyAtomicType::FLOAT && ((const ATFloatOrDerived*)atom)->isNaN())
        return false;

      if(atom->getPrimitiveTypeIndex() == AnyAtomicType::DOUBLE && ((const ATDoubleOrDerived*)atom)->isNaN())
        return false;
    }
  }

  // In all other cases, returns true.
  return true;
}

Sequence ResultImpl::toSequence(DynamicContext *context)
{
  if(context == 0) context = context_;

  Sequence result(context->getMemoryManager());

  Item::Ptr item = 0;
  while((item = next(context)) != NULLRCP) {
    result.addItem(item);
  }

  return result;
}

std::string ResultImpl::getIndent(int indent)
{
  std::ostringstream oss;

  for(int i = 0; i < indent; ++i) {
    oss << "  ";
  }

  return oss.str();
}
