/*
 * Copyright (c) 2001, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../config/pathan_config.h"
#include <sstream>

#include <pathan/internal/utils/PrintDataItemTree.hpp>
#include <pathan/dataItem/DataItemNav.hpp>
#include <pathan/dataItem/DataItemStep.hpp>
#include <pathan/Sequence.hpp>
#include <xercesc/dom/DOMDocument.hpp>
#include <pathan/internal/navigation/SelfAxis.hpp>
#include <pathan/exceptions/NavigationException.hpp>
#include <pathan/Node.hpp>
#include <pathan/functions/FunctionRoot.hpp>
#include <pathan/dataItem/StaticResolutionContext.hpp>
#include <pathan/dataItem/SequenceResult.hpp>
#include <pathan/internal/factory/DatatypeFactory.hpp>
#include <set>

DataItemNav::DataItemNav(XPath2MemoryManager* memMgr)
	: DataItemImpl(memMgr), _isSorted(-1),
	  _steps(PathanAllocator<StepInfo>(memMgr))
{
  setType(DataItem::NAVIGATION);
  _gotoRoot = false;
}


DataItemNav::~DataItemNav()
{
  //no-op
}

Result DataItemNav::createResult(DynamicContext* context, int flags) const
{
  Result result(0);
  flags &= ~(RETURN_ONE | RETURN_TWO);

  if(_gotoRoot) {
    result = new GotoRootResult(context);
  }

  Steps::const_iterator end = _steps.end();
  for(Steps::const_iterator it = _steps.begin(); it != end; ++it) {
    if(it->usesContextSize) {
      // We need the context size, so convert to a Sequence to work it out
      Sequence seq(result.toSequence(context));
      result = new StepResult(new SequenceResult(seq), it->step, seq.getLength(), flags, context);
    }
    else {
      result = new StepResult(result, it->step, 0, flags, context);
    }
  }

  if(context->getNodeSetOrdering()==StaticContext::ORDERING_UNORDERED ||
     flags & DataItem::UNORDERED || getIsSorted()) {
    return result;
  }
  else {
    return result.sortIntoDocumentOrder(context);
  }
}

void DataItemNav::addStep(NavStepImpl* step)
{
  _steps.push_back(new (getMemoryManager()) DataItemStep(step, getMemoryManager()));
}

void DataItemNav::addStep(const StepInfo &step)
{
  _steps.push_back(step);
}

void DataItemNav::addStepFront(DataItem* step)
{
  _steps.insert(_steps.begin(), step);
}

void DataItemNav::setGotoRootFirst(bool gotoRoot)
{
  _gotoRoot = gotoRoot;
}

DataItem* DataItemNav::staticResolution(StaticContext *context, StaticResolutionContext *src)
{
  StaticResolutionContext newSrc(context->getMemoryManager());
  Steps newSteps(PathanAllocator<StepInfo>(context->getMemoryManager()));

  if(_gotoRoot) {
    newSrc.contextItemUsed(true);
  }

  Steps::iterator begin = _steps.begin();
  Steps::iterator end = _steps.end();
  for(Steps::iterator it = begin; it != end; ++it) {

    // Statically resolve our step
    StaticResolutionContext stepSrc(context->getMemoryManager());
    DataItem *step = it->step->staticResolution(context, &stepSrc);

    if(stepSrc.areContextFlagsUsed() || newSrc.isNoFoldingForced()) {
      newSteps.push_back(StepInfo(step, stepSrc.isContextSizeUsed()));

      if(it != begin || _gotoRoot) {
        // Remove context item usage
        stepSrc.contextItemUsed(false);
        stepSrc.contextPositionUsed(false);
        stepSrc.contextSizeUsed(false);
      }
      newSrc.add(&stepSrc);
    }
    else {
      // Constant fold, by clearing all our previous steps (and the root pseudo-step)
      // and just having our most recent step.
      // This is only possible because the result of steps has to be nodes, and
      // duplicates are removed, which means we aren't forced to execute a constant
      // step a set number of times.
      _gotoRoot = false;
      newSteps.clear();
      newSteps.push_back(StepInfo(step, stepSrc.isContextSizeUsed()));
      newSrc.add(&stepSrc);
    }
  }

  _steps = newSteps;

  if(newSrc.isUsed()) {
    src->add(&newSrc);
    return resolvePredicates(context, src);
  }
  else {
    return constantFold(context, src);
  }
}

bool DataItemNav::getGotoRootFirst() const {
  return _gotoRoot;
}

const DataItemNav::Steps &DataItemNav::getSteps() const {
  return _steps;
}

// If the results of this navigation are already sorted,
// return true.  _isSorted is a cache of the answer,
// which is obtained by walking the steps.
//  -1 - unknown
//  0 - not sorted
//  1 - sorted
bool DataItemNav::getIsSorted() const {
      if (_isSorted == -1) {
              // Simplistic algorithm for now
              // If first step is doc order and
              // has PEER property, and the rest
              // have SUBTREE, avoid sorting.
              // TBD: do something more intelligent
              Steps::const_iterator end = _steps.end();
              Steps::const_iterator it = _steps.begin();

              unsigned int props;
              if(_gotoRoot) {
                props = DataItem::DOCORDER | DataItem::PEER;
              }
              else {
                props = it->step->getStaticProperties();
                ++it;
              }

              _isSorted = 0;
              if ((props & DataItem::DOCORDER) &&
                  (props & DataItem::PEER)) {
                      for(; it != end; ++it) {
                              props = it->step->getStaticProperties();
                              if (!(props & DataItem::SUBTREE)) {
                                      return false;
                              }
                      }
                      _isSorted = 1;
              }
      }
      return (_isSorted > 0);
}

/////////////////////////////////////
// GotoRootResult
/////////////////////////////////////

DataItemNav::GotoRootResult::GotoRootResult(DynamicContext *context)
  : SingleResult(context)
{
}

Item::Ptr DataItemNav::GotoRootResult::getSingleResult(DynamicContext *context) const
{
  // Do equivalent of root()
  const Item::Ptr contextItem = context->getContextItem();
  if(contextItem != NULLRCP && contextItem->isNode()) {
    const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *root =
      FunctionRoot::root(((const Node::Ptr)contextItem)->getDOMNode());
    
    if(root->getNodeType()!=XERCES_CPP_NAMESPACE_QUALIFIER DOMNode::DOCUMENT_NODE) {
      DSLthrow(NavigationException,X("DataItemNav::collapseTreeInternal"), X("The root of the context node is not a document node [err:XP0050]"));
    }

    return (const Item::Ptr)DatatypeFactory::POD2AT::createNode(root, context);
  } else {
    DSLthrow(NavigationException,X("DataItemNav::collapseTreeInternal"), X("An attempt to navigate when the Context Item was not a node was made [err:XP0020]"));
  }
}

std::string DataItemNav::GotoRootResult::asString(DynamicContext *context, int indent) const
{
  std::ostringstream oss;
  std::string in(getIndent(indent));

  oss << in << "<nav_goto_root/>" << std::endl;

  return oss.str();
}

/////////////////////////////////////
// StepResult
/////////////////////////////////////

DataItemNav::StepResult::StepResult(const Result &parent, DataItem *step, unsigned int contextSize, int flags, DynamicContext *context)
  : ResultImpl(context),
    initialised_(false),
    flags_(flags),
    parent_(parent),
    step_(step),
    stepResult_(0),
    contextPos_(0),
    contextSize_(contextSize),
    contextItem_(0)
{
}

Item::Ptr DataItemNav::StepResult::next(DynamicContext *context)
{
  unsigned int oldContextSize = context->getContextSize();
  unsigned int oldContextPosition = context->getContextPosition();
  const Item::Ptr oldContextItem = context->getContextItem();

  context->setContextSize(contextSize_);
  context->setContextPosition(contextPos_);
  context->setContextItem(contextItem_);

  Item::Ptr result = 0;
  while(true) {
    result = stepResult_.next(context);
    if(result == NULLRCP) {
      if(!initialised_ && parent_.isNull()) {
        initialised_ = true;
        // We have no parent step, so navigate from the context item
        contextItem_ = oldContextItem;
        contextPos_ = oldContextPosition;
        contextSize_ = oldContextSize;
      }
      else {
        context->setContextSize(oldContextSize);
        context->setContextPosition(oldContextPosition);
        context->setContextItem(oldContextItem);

        contextItem_ = parent_.next(context);
        if(contextItem_ == NULLRCP) {
          parent_ = 0;
          stepResult_ = 0;
          return 0;
        }
        ++contextPos_;
      }

      context->setContextSize(contextSize_);
      context->setContextPosition(contextPos_);
      context->setContextItem(contextItem_);

      stepResult_ = step_->collapseTree(context, flags_);
    }
    else {
      // Check it's a node
      if(!result->isNode()) {
        DSLthrow(NavigationException,X("DataItemNav::StepResult::next"), X("The result of a step expression (StepExpr) is not a sequence of nodes [err:XP0019]"));
      }

      context->setContextSize(oldContextSize);
      context->setContextPosition(oldContextPosition);
      context->setContextItem(oldContextItem);
      return result;
    }
  }
}

std::string DataItemNav::StepResult::asString(DynamicContext *context, int indent) const
{
  std::ostringstream oss;
  std::string in(getIndent(indent));

  oss << in << "<step contextSize=\"" << contextSize_ << "\">" << std::endl;
  if(!parent_.isNull()) {
    oss << in << "  <parent>" << std::endl;
    oss << parent_.asString(context, indent + 2);
    oss << in << "  </parent>" << std::endl;
  }
  oss << in << "  <dataitem>" << std::endl;
  oss << PrintDataItemTree::print(step_, context, indent + 2);
  oss << in << "  </dataitem>" << std::endl;
  oss << in << "</step>" << std::endl;

  return oss.str();
}
