/*
 * Copyright (c) 2001, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#include "../config/pathan_config.h"
#include <sstream>

#include <pathan/internal/utils/PrintDataItemTree.hpp>
#include <pathan/dataItem/DataItemImpl.hpp>
#include <pathan/XPath2Utils.hpp>
#include "../utils/NumUtils.hpp"
#include <pathan/dataItem/DataItem.hpp>
#include <pathan/Sequence.hpp>
#include <pathan/exceptions/DataItemException.hpp>
#include "../exceptions/XPath2TypeCastException.hpp"
#include <pathan/Numeric.hpp>
#include <pathan/Node.hpp>
#include <pathan/ATBooleanOrDerived.hpp>
#include <pathan/ATDecimalOrDerived.hpp>
#include <pathan/DateOrTimeType.hpp>
#include <pathan/DynamicContext.hpp>
#include <pathan/internal/context/DynamicContextImpl.hpp>
#include <pathan/internal/factory/DatatypeFactory.hpp>
#include <pathan/dataItem/DataItemSequence.hpp>
#include <pathan/dataItem/StaticResolutionContext.hpp>
#include <xercesc/validators/schema/SchemaSymbols.hpp>
#include <pathan/dataItem/SequenceResult.hpp>

DataItemImpl::DataItemImpl(XPath2MemoryManager* memMgr)
  : _predList(PathanAllocator<PredInfo>(memMgr)),
    _memMgr(memMgr)
{
  // Nothing to do
}

DataItemImpl::~DataItemImpl(void)
{
  //no-op
}

void DataItemImpl::addPredicates(const VectorOfDataItems &steps)
{
    for(VectorOfDataItems::const_iterator i=steps.begin();i!=steps.end();i++)
        _predList.push_back(PredInfo(*i));
}

void DataItemImpl::addPredicates(const Predicates &steps)
{
    _predList.insert(_predList.end(),steps.begin(),steps.end());
}

const DataItemImpl::Predicates& DataItemImpl::getPredicates(void) const
{
  return _predList;
}


/** Returns true if this DataItem has no predicates, and is an instance of
    DataItemSequence or DataItemLiteral */
bool DataItemImpl::isConstant() const
{
  return (_type == DataItem::SEQUENCE || _type == DataItem::LITERAL) && _predList.empty();
}

/** Overridden in DataItemSequence and DataItemLiteral */
bool DataItemImpl::isConstantAndHasTimezone(StaticContext *context) const
{
  return false;
}

DataItem::whichType DataItemImpl::getType(void) const
{
  return _type;
}

void DataItemImpl::setType(DataItem::whichType t)
{
  _type = t;
}

bool DataItemImpl::isSingleNumericConstant(StaticContext *context) const
{
  return false;
}

Result DataItemImpl::createPredicateResult(Result &parent, const PredInfo &p, DynamicContext *context) const
{
  if(p.usesContextSize) {
    // We need the context size, so convert to a Sequence to work it out
    Sequence seq(parent.toSequence(context));
    return new PredicateFilterResult(new SequenceResult(seq), p, seq.getLength(), context);
  }
  else {
    return new PredicateFilterResult(parent, p, 0, context);
  }
}

Result DataItemImpl::collapseTree(DynamicContext* context, int flags) const
{
  Result result = createResult(context, flags);

  for(Predicates::const_iterator it = _predList.begin(); it != _predList.end(); ++it) {
    result = createPredicateResult(result, *it, context);
  }

  return postPredicateResultHook(result, context, flags);
}

Result DataItemImpl::postPredicateResultHook(Result &result, DynamicContext* context, int flags) const
{
  return result;
}

Result DataItemImpl::createResult(DynamicContext* context, int flags) const
{
  return new CollapseTreeInternalResult(this, flags, context);
}

Sequence DataItemImpl::collapseTreeInternal(DynamicContext* context, int flags) const
{
  return Sequence(context->getMemoryManager());
}

/** Calls staticResolution on the DataItem, then if isConstant() is true for
    it, and constantFold is true, returns the result of the
    constantFold() method, otherwise returns the result of the resolvePredicates()
    method. */
DataItem *DataItemImpl::resolveDataItem(DataItem *&di, StaticContext *context, StaticResolutionContext *src, bool cf)
{
  di = di->staticResolution(context, src);
  if(di->isConstant() && cf) {
    return constantFold(context, src);
  }
  else {
    return resolvePredicates(context, src);
  }
}

/** Calls staticResolution on the DataItems, then if isConstant() is true for
    all of them, and constantFold is true, returns the result of the
    constantFold() method, otherwise returns the result of the resolvePredicates()
    method. */
DataItem *DataItemImpl::resolveDataItems(VectorOfDataItems &dis, StaticContext *context, StaticResolutionContext *src, bool cf)
{
  bool allConstant = true;
  for(VectorOfDataItems::iterator i = dis.begin(); i != dis.end(); ++i) {
    *i = (*i)->staticResolution(context, src);
    if(!(*i)->isConstant()) {
      allConstant = false;
    }
  }

  if(allConstant && cf) {
    return constantFold(context, src);
  }
  else {
    return resolvePredicates(context, src);
  }
}

/** Calls staticResolution on the DataItems, then if isConstantAndHasTimezone() is true for
    all of them, and constantFold is true, returns the result of the
    constantFold() method, otherwise returns the result of the resolvePredicates()
    method. */
DataItem *DataItemImpl::resolveDataItemsForDateOrTime(VectorOfDataItems &dis, StaticContext *context, StaticResolutionContext *src, bool cf)
{
  bool allConstant = true;
  for(VectorOfDataItems::iterator i = dis.begin(); i != dis.end(); ++i) {
    *i = (*i)->staticResolution(context, src);
    if(!(*i)->isConstantAndHasTimezone(context)) {
      allConstant = false;
      if((*i)->isConstant()) {
        src->implicitTimezoneUsed(true);
      }
    }
  }

  if(allConstant && cf) {
    return constantFold(context, src);
  }
  else {
    return resolvePredicates(context, src);
  }
}

/** Calls staticResolution on the predicates of this DataItem, constant folding
    if possible */
DataItem *DataItemImpl::resolvePredicates(StaticContext *context, StaticResolutionContext *src)
{
  Predicates newPreds(PathanAllocator<PredInfo>(context->getMemoryManager()));
  DataItemImpl *result = resolvePredicate(_predList.rbegin(), newPreds, context, src);
  result->_predList = newPreds;

  return result;
}

/** Calls staticResolution on the predicates of this DataItem, constant folding
    if possible */
DataItemImpl *DataItemImpl::resolvePredicate(Predicates::reverse_iterator it, Predicates &newPreds, StaticContext *context, StaticResolutionContext *src)
{
  if(it == _predList.rend()) {
    return this;
  }
  else {
    DataItemImpl *newDI = resolvePredicate(it + 1, newPreds, context, src);

    // A predicate before us has short circuited the predicate resolving,
    // so we'll just pass back it's result.
    if(newDI != this) {
      return newDI;
    }

    StaticResolutionContext newSrc(context->getMemoryManager());
    DataItem *pred = it->pred->staticResolution(context, &newSrc);

    if(!newSrc.isUsed() && !pred->isSingleNumericConstant(context)) {
      // It's not a numeric constant
      AutoRelease<DynamicContext> dContext(context->createDynamicContext());
      dContext->setMemoryManager(context->getMemoryManager());
      if(pred->collapseTree(dContext).getEffectiveBooleanValue(dContext)) {
        // We have a true predicate, so don't add it to the new predicate list
        return this;
      }
      else {
        // We have a false predicate, which is constant folded to an empty sequence,
        // short circuiting the subsequent predicate resolutions
        newPreds.clear();
        return new (getMemoryManager()) DataItemSequence(getMemoryManager());
      }
    }
    else {
      // It's not constant, or it's numeric, so add it to the new predicate list, and return
      newPreds.push_back(PredInfo(pred, newSrc.areContextFlagsUsed(), newSrc.isContextSizeUsed()));

      // Remove context item usage
      newSrc.contextItemUsed(false);
      newSrc.contextPositionUsed(false);
      newSrc.contextSizeUsed(false);

      src->add(&newSrc);

      return this;
    }
  }
}

/** Performs constant folding on this DataItem, transfering any predicates to
    the returned DataItem */
DataItem *DataItemImpl::constantFold(StaticContext *context, StaticResolutionContext *src) const
{
  AutoRelease<DynamicContext> dContext(context->createDynamicContext());
  dContext->setMemoryManager(context->getMemoryManager());
  Result result = createResult(dContext);
  DataItem* newBlock = new (getMemoryManager()) DataItemSequence(result, dContext, getMemoryManager());
  newBlock->addPredicates(_predList);
  return newBlock->staticResolution(context, src);
}

const SequenceType *DataItemImpl::getStaticType() const
{
  return 0;
}

XPath2MemoryManager* DataItemImpl::getMemoryManager(void) const {

  return _memMgr;
}

/////////////////////////////////////
// CollapseTreeInternalResult
/////////////////////////////////////

DataItemImpl::CollapseTreeInternalResult::CollapseTreeInternalResult(const DataItemImpl *di, int flags, DynamicContext *context)
  : LazySequenceResult(context),
    _flags(flags),
    _di(di)
{
}

void DataItemImpl::CollapseTreeInternalResult::getResult(Sequence &toFill, DynamicContext *context) const
{
  const Sequence &tarts = _di->collapseTreeInternal(context, _flags);
  toFill = tarts;
}

std::string DataItemImpl::CollapseTreeInternalResult::asString(DynamicContext *context, int indent) const
{
  std::ostringstream oss;
  std::string in(getIndent(indent));

  oss << in << "<collapsetreeinternal/>" << std::endl;

  return oss.str();
}

/////////////////////////////////////
// PredicateFilterResult
/////////////////////////////////////

DataItemImpl::PredicateFilterResult::PredicateFilterResult(const Result &parent, const PredInfo &pred, unsigned int contextSize, DynamicContext *context)
  : ResultImpl(context),
    todo_(true),
    parent_(parent),
    pred_(pred),
    contextPos_(0),
    contextSize_(contextSize)
{
}

Item::Ptr DataItemImpl::PredicateFilterResult::next(DynamicContext *context)
{
  unsigned int oldContextSize = context->getContextSize();
  unsigned int oldContextPosition = context->getContextPosition();
  const Item::Ptr oldContextItem = context->getContextItem();

  Item::Ptr result = 0;
  while(result == NULLRCP) {
    result = parent_.next(context);
    if(result == NULLRCP) {
      parent_ = 0;
      return 0;
    }

    ++contextPos_;

    if(todo_ || pred_.usesContext) {
      context->setContextSize(contextSize_);
      context->setContextPosition(contextPos_);
      context->setContextItem(result);

      Result pred_result = pred_.pred->collapseTree(context, DataItem::UNORDERED);
      first_ = pred_result.next(context);
      second_ = pred_result.next(context);
    }

    // 3.2.2 ...
    // The predicate truth value is derived by applying the following rules, in order:
    // 1) If the value of the predicate expression is an atomic value of a numeric type, the predicate truth
    // value is true if and only if the value of the predicate expression is equal to the context position.
    if(first_ != NULLRCP && second_ == NULLRCP && first_->isAtomicValue() &&
       ((const AnyAtomicType::Ptr)first_)->isNumericValue()) {
      const Numeric::Ptr num = (const Numeric::Ptr)first_;
      if(!num->equals((const AnyAtomicType::Ptr)DatatypeFactory::POD2AT::createInteger((long)contextPos_, context), context)) {
        result = 0;
      }
      else if(!pred_.usesContext) {
        parent_ = 0;
      }
    }
    else {
      // 2) Otherwise, the predicate truth value is the effective boolean value of the predicate expression
      if(!ResultImpl::getEffectiveBooleanValue(first_, second_, context)) {
        result = 0;
      }
    }

    if(todo_ || pred_.usesContext) {
      context->setContextSize(oldContextSize);
      context->setContextPosition(oldContextPosition);
      context->setContextItem(oldContextItem);
      todo_ = false;
    }
  }

  return result;
}

std::string DataItemImpl::PredicateFilterResult::asString(DynamicContext *context, int indent) const
{
  std::ostringstream oss;
  std::string in(getIndent(indent));

  oss << in << "<predicate contextSize=\"" << contextSize_ << "\">" << std::endl;
  if(!parent_.isNull()) {
    oss << in << "  <parent>" << std::endl;
    oss << parent_.asString(context, indent + 2);
    oss << in << "  </parent>" << std::endl;
  }
  oss << in << "  <dataitem>" << std::endl;
  oss << PrintDataItemTree::print(pred_.pred, context, indent + 2);
  oss << in << "  </dataitem>" << std::endl;
  oss << in << "</predicate>" << std::endl;

  return oss.str();
}
