/*
 * Copyright (c) 2001, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#ifndef _NAVIGATIONSTEPIMPL_HPP
#define _NAVIGATIONSTEPIMPL_HPP

#include <pathan/Pathan.hpp>

#include <xercesc/util/XercesDefs.hpp>
#include <pathan/Sequence.hpp>
#include <pathan/dataItem/Result.hpp>
#include <pathan/dataItem/ResultImpl.hpp>
#include <pathan/DynamicContext.hpp>
#include <pathan/SequenceType.hpp>

class XPath2MemoryManager;

XERCES_CPP_NAMESPACE_BEGIN
class DOMNode;
XERCES_CPP_NAMESPACE_END

typedef enum {
  FORWARD, REVERSE
} AxisOrientation;

/**
This class stores information about a Location path step. When createResult is called the candidate
list passed in is filtered based on predictes. 
 */
class PATHAN_EXPORT NavStepImpl
{
public:
  NavStepImpl(void);
  virtual ~NavStepImpl(void);

  typedef enum {
    ANCESTOR,
    ANCESTOR_OR_SELF,
    ATTRIBUTE,
    CHILD,
    DESCENDANT,
    DESCENDANT_OR_SELF,
    FOLLOWING,
    FOLLOWING_SIBLING,
    NAMESPACE,
    PARENT,
    PRECEDING,
    PRECEDING_SIBLING,
    SELF,
    UNKNOWN
  } Axis;

  ///this acts on the candidate list and returns the result
  virtual Result createResult(DynamicContext* context) const;

  void staticResolution(StaticContext *context);

  /// Overridden in derived classes, to give correct answer
  virtual Axis getAxis() const;

  ///set function for private member
  void setType(SequenceType::ItemType* type);
  
  ///set function for private member
  void setNodeName(const XMLCh* name);

  ///set function for private member
  void setNodeUri(const XMLCh* uri);

  ///set function for private member
  void setNodePrefix(const XMLCh* prefix);

  ///set node type looked for
  void setNodeType(int type);

  ///sets the wildcard nodeName bool
  void setNameWildcard(void);

  ///sets the prefix wildcard bool
  void setNamespaceWildcard(void);

  ///sets the type wildcard bool
  void setTypeWildcard(void);

  ///sets the axis orientation used in predicates
  void setAxisOrientation(const AxisOrientation ori);

  ///gets the node type we are selecting
  int getNodeType(void) const;

  ///gets the uri of the node we are selecting
  const XMLCh* getNodeUri(void) const;

  ///gets the prefix of the node we are selecting
  const XMLCh* getNodePrefix(void) const;

  ///gets the sequence type used to match the nodes
  SequenceType::ItemType* getType(void) const;
  
  ///gets the name of the node we are selecting
  const XMLCh* getNodeName(void) const;

  ///returns true if the Node Name ia a wildcard
  bool getNameWildcard(void) const;

  ///returns true if the Node prefix ia a wildcard
  bool getNamespaceWildcard(void) const;

  ///returns true if the type is a wildcard
  bool getTypeWildcard(void) const;


  ///retuns whether the type has been set in this NavStep
  bool getTypeSet(void) const;

  ///returns whether this is a forward or reverse axis
  int getAxisOrientation() const;

  ///returns whether the prefix has been set (instead of the uri)
  bool getPrefixSet(void) const;

  ///check if a node has to be placed in the result set
  bool filterNode(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node, DynamicContext* context) const;

  /**
   * Performs filtering based on type or name.
   * Sub-classes should implement stepNext().
   */
  class PATHAN_EXPORT AxisResult : public ResultImpl
  {
  public:
    AxisResult(const NavStepImpl *step, DynamicContext *context);
    virtual ~AxisResult() {}

    Item::Ptr next(DynamicContext *context);
    virtual const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *stepNext(DynamicContext *context) = 0;

    std::string asString(DynamicContext *context, int indent) const;

  protected:
    const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *contextNode_;
    const NavStepImpl *step_;
  };

protected:
  ///check a node's type against _type, considers wildcard
  bool checkNodeType(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node) const;

  ///check a node's name against _name, considers wildcard
  bool checkNodeName(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node) const;

  /// returns whether node has the same uri as _uri, considers wildcard
  bool checkNodeNamespace(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node, DynamicContext* context) const;

private:
  const XMLCh* _name,*_uri,*_prefix;
  int _type;
  bool _wildcardName, _wildcardNamespace, _wildcardType, _usePrefix;
  AxisOrientation _axisOrientation;
  SequenceType::ItemType* _itemType;
};

#endif
