/*
 * Copyright (c) 2002, DecisionSoft Limited All rights reserved.
 * Please see LICENSE.TXT for more information.
 */

#ifndef _XPATH2CONTEXTIMPL_HPP
#define _XPATH2CONTEXTIMPL_HPP

#include <pathan/Pathan.hpp>

#include <pathan/StaticContext.hpp>
#include <pathan/DynamicContext.hpp>
#include "../ProxyMemoryManager.hpp"
#include <time.h>
#include <vector>

class DatatypeLookup;
class DatatypeFactory;
class XPath2MemoryManager;
class DocumentCache;
class FunctionLookup;
class Sequence;
class VariableStore;
class Collation;
class ATDurationOrDerived;
class Item;
class URIResolver;

XERCES_CPP_NAMESPACE_BEGIN
class DOMNode;
class DOMDocument;
class DOMXPathNSResolver;
class XMLGrammarPool;
XERCES_CPP_NAMESPACE_END

///Expression Context is a storage for contexts.
class PATHAN_EXPORT XPath2ContextImpl : public DynamicContext
{
public:
  XPath2ContextImpl(XERCES_CPP_NAMESPACE_QUALIFIER MemoryManager* memMgr,
                    XERCES_CPP_NAMESPACE_QUALIFIER XMLGrammarPool* xmlgr,
                    XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* contextNode);
  virtual ~XPath2ContextImpl();

  virtual void release();

  virtual DynamicContext *createDynamicContext(XERCES_CPP_NAMESPACE_QUALIFIER MemoryManager *memMgr
                                               = XERCES_CPP_NAMESPACE_QUALIFIER XMLPlatformUtils::fgMemoryManager) const;
  
  //////////////////////////////
  // Static Context Accessors //
  //////////////////////////////
    
  /** Get the current XPath 1.0 compatibility mode */
  virtual bool getXPath1CompatibilityMode() const;
  /** Set the current XPath 1.0 compatibility mode */
  virtual void setXPath1CompatibilityMode(bool newMode);

  /** Get the NS resolver */
  virtual const XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNSResolver* getNSResolver() const;
  /** Set the NS resolver */
  virtual void setNSResolver(const XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNSResolver* resolver);

  /** get the value of the default namespace for elements and types */
  virtual const XMLCh* getDefaultElementAndTypeNS() const;
  /** set the value of the default namespace for elements and types */
  virtual void setDefaultElementAndTypeNS(const XMLCh* newNS);

  /** Get the default namespace for functions */
  virtual const XMLCh* getDefaultFuncNS() const;
  /** Set the default namespace for functions */
  virtual void setDefaultFuncNS(const XMLCh* newNS);

  /** retrieve the repository for the grammars **/
  virtual const DocumentCache* getDocumentCache() const;
  /** returns the validated node */
  virtual XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *validate(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode* node, DocumentCache::ValidationMode valMode);
  /** add the location for the grammar of a specific namespace **/
  virtual void addSchemaLocation(const XMLCh* uri, VectorOfStrings* locations);  
  
  /** get the variable type store */  
  virtual VariableTypeStore* getVariableTypeStore();
  
  /** adds a custom function to the function table */
  virtual void addCustomFunction(FuncFactory *func);
	/** returns a function with name name in the namespace represented by prefix */
  virtual DataItem* lookUpFunction(const XMLCh* prefix, const XMLCh* name, VectorOfDataItems& v) const;
  /* looks up the correct DatatypeFactory for the typeURI:typeName pair */
  virtual const DatatypeFactory* getDatatypeFactory(const XMLCh* typeURI, const XMLCh* typeName) const;

  /** Get the implementation for the specified collation */
  virtual Collation* getCollation(const XMLCh* const URI) const;
  /** Add a collation	*/
  virtual void addCollation(Collation* collation);
  /** Get the default collation */
  virtual Collation* getDefaultCollation() const;
  /** Specify which collation is the default one */
  virtual void setDefaultCollation(const XMLCh* const URI);

  
  /** Return the base URI */
  virtual const XMLCh* getBaseURI() const;
  /** Set the base URI */
  virtual void setBaseURI(const XMLCh* newURI);

  /** Return the ordering method for node sets */
  virtual NodeSetOrdering getNodeSetOrdering() const;
  /** Set the ordering method for node sets */
  virtual void setNodeSetOrdering(NodeSetOrdering newOrder);

  //////////////////////////////////
  // Evaluation Context Accessors //
  //////////////////////////////////

  /** Resets the dynamic context, as if it had never been used */
  virtual void clearDynamicContext();

  /** Get the context Item */
  virtual const Item::Ptr &getContextItem() const;
  /** Set the context item to item */
  virtual void setContextItem(const Item::Ptr &item);
  /** Sets the context item to an external document.
      This is needed so that the DOMDocument reference counting
      does not release the document that the node comes from. */
  virtual void setExternalContextNode(const XERCES_CPP_NAMESPACE_QUALIFIER DOMNode *node);

  /** Get the context position */
  virtual unsigned int getContextPosition() const;
  /** Set the context position */ 
  virtual void setContextPosition(unsigned int pos);
  
  /** Get the context size */
  virtual unsigned int getContextSize() const;
  /** Set the context size */
  virtual void setContextSize(unsigned int size);
  
  /** get the variable store */  
  virtual VariableStore* getVariableStore();
  
  /** Return the current time */
  virtual time_t getCurrentTime() const;
  /** Set the current time */
  virtual void setCurrentTime(time_t newTime);

  /** Return the implicit timezone for this system.*/
  virtual const ATDurationOrDerived::Ptr &getImplicitTimezone() const;
  /** Set the implicit timezone */
  virtual void setImplicitTimezone(const ATDurationOrDerived::Ptr &timezoneAsDuration);

  /////////////////////////////////////////
  //  Pathan context specific accessors  //
  /////////////////////////////////////////

  /** Returns the XPath2MemoryManager* for this context */
  virtual XPath2MemoryManager* getMemoryManager() const;
  /** Set the memory manager to the one given */
  virtual void setMemoryManager(XPath2MemoryManager* memMgr);

  ////////////////////////////////////////////
  // Pathan context specific helper methods //
  ////////////////////////////////////////////

	/** Get the uri bound to a prefix in the XPathNSResolver */
  virtual const XMLCh* getUriBoundToPrefix(const XMLCh* prefix) const;

  virtual const XMLCh* getPrefixBoundToUri(const XMLCh* uri) const;

  /** returns true if the type represented by uri:typename is an instance of uriToCheck:typeNameToCheck 
   *
   * ie: to check 
   * xs:integer instance of xs:decimal, 
   * call
   * isTypeOrDerivedFromType("xs", "integer", "xs", "decimal") 
   * (except of course, call with URIs, not prefixes!)
   */ 
  virtual bool isTypeOrDerivedFromType(const XMLCh* const uri, const XMLCh* const typeName, const XMLCh* const uriToCheck, const XMLCh* const typeNameToCheck) const; 

  /** Used whenever we need to create a new document (including parsing in documents) */
  virtual XERCES_CPP_NAMESPACE::DOMDocument *createNewDocument() const;
  virtual void releaseDocument(XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument *doc) const;

  /**
   * Sets the XMLEntityResolver that is used by Xerces when it is used
   * to parse documents. This affects the behaviour of Pathan whenever
   * it retrieves a DTD or XML Schema grammar.
   */
  void setXMLEntityResolver(XERCES_CPP_NAMESPACE_QUALIFIER XMLEntityResolver* const handler);

  /**
     Register a callback object for resolving URIs. Multiple URIResolver
     objects can be registered with the DynamicContext at the same time.
     The most recently added URIResolver will be called first, and if it
     returns NULL, subsequent ones will be called - ending in a call to
     the fallback routines.

     The DynamicContext will not
     adopt this object, making the user responsible for making sure that the
     URIResolver object's lifespan matches or exceeds the life of the
     DynamicContext.
  */
  virtual void registerURIResolver(URIResolver *resolver);
  /* Resolve the given uri (and baseUri) to a DOMDocument. If the uri
     is relative, the base uri is obtained from the context. */
  virtual Sequence resolveDocument(const XMLCh* uri);
  /* Resolve the given uri (and baseUri) to a list of DOMNode objects. If
     the uri is relative, the base uri is obtained from the context. */
  virtual Sequence resolveCollection(const XMLCh* uri);

  /** send a pair of strings to the "trace" data set */
  virtual void trace(const XMLCh* message1, const XMLCh* message2);

protected:

  // The memory manager used to create this context
  XERCES_CPP_NAMESPACE_QUALIFIER MemoryManager* _createdWith;

  // used for managing the memory of objects inside this context
  ProxyMemoryManager _internalMM;

  ////////////////////////////////
  //  Static Context variables  //
  ////////////////////////////////

  /** XPath 1.0 compatibility mode. This value is true if rules for 
   * backward compatibility with XPath Version 1.0 are in effect; 
   * otherwise it is false */
  bool _xpath1Compatibility;

  /** In-scope namespaces. This is a set of  (prefix, URI) pairs. 
   * The in-scope namespaces are used for resolving prefixes used 
   * in QNames within the expression */
  const XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNSResolver* _nsResolver;

  /** The original (global scope) DOMXPathNSResolver */
  const XERCES_CPP_NAMESPACE_QUALIFIER DOMXPathNSResolver* _globalNSResolver;

  /** Default element/type namespace. This is a namespace URI or "none". This 
   * namespace is used for any unprefixed QName appearing in a 
   * position where an element or type name is expected. */
  const XMLCh* _defaultElementNS;

  /** Default function namespace. This is a namespace URI. 
   * This namespace URI is used for any unprefixed QName appearing 
   * as the function name in a function call. */
  const XMLCh* _functionNS;

  /** In-scope schema definitions. This is a generic term for all 
   * the element, attribute, and type definitions that are in scope 
   * during processing of an expression. */
  DocumentCache* _docCache;

  /** In-scope variable types. This is a set of  (QName, type) pairs.
   * It defines the set of variables that have been declared and are 
   * available for reference within the expression. The QName 
   * represents the name of the variable, and the type represents its 
   * static data type */
  VariableTypeStore* _varTypeStore;

  /** In-scope functions. This part of the  static context defines the 
   * set of functions that are available to be called from within an 
   * expression. Each function is uniquely identified by its QName and 
   * its arity (number of parameters). The  static context maps the QName 
   * and arity into a function signature and a function definition. The 
   * function signature specifies the static types of the function 
   * parameters and the function result. */
  FunctionLookup* _functionTable; 

  /** For each atomic type in the in-scope type  definitions, there is 
   * a constructor function in the in-scope functions. Constructor 
   * functions are discussed in 3.10.4 Constructor Functions */
  DatatypeLookup* _datatypeLookup;

  /** In-scope collations. This is a set of  (URI, collation) pairs. It 
   * defines the names of the collations that are available for use in 
   * function calls that take a collation name as an argument. A collation 
   * may be regarded as an object that supports two functions: a function 
   * that given a set of strings, returns a sequence containing those strings 
   * in sorted order; and a function that given two strings, returns true if 
   * they are considered equal, and false if not. */
  std::vector<Collation*, PathanAllocator<Collation*> > _collations;

  /** Default collation. This is a collation. This collation is used by 
   * string comparison functions when no explicit collation is specified. */
  const XMLCh* _defaultCollation;

  /** Ordering mode, which has the value ordered or unordered, affects 
   *  the ordering of the result sequence returned by path expressions, 
   *  union, intersect, and except expressions, and FLWOR expressions 
   *  that have no order by clause. */
  NodeSetOrdering _ordering;

  /** Base URI. This is an absolute URI, used when necessary in the 
   * resolution of relative URIs (for example, by the fn:resolve- uri 
   * function.) The base URI is always provided by the external environment. */
  const XMLCh* _baseURI;

  ////////////////////////////////////
  //  Evaluation Context Variables  //
  ////////////////////////////////////

  /** The context item is the item currently being processed. */
  Item::Ptr _contextItem;

  /** The context position is the position of  the context item within 
   * the sequence of items currently being processed. */
  unsigned int _contextPosition;

  /** The context size is the number of items in the sequence of 
   * items currently being processed. */
  unsigned int _contextSize;
  
  /** In-scope variable values. This is a set of  (QName, Sequence) pairs.
   * It defines the set of variables that have been declared and are 
   * available for reference within the expression. The QName 
   * represents the name of the variable, and the Sequence represents its 
   * value */
  VariableStore* _varStore;

  /** Current date and time. This information  represents an 
   * implementation-dependent point in time during processing of a query 
   * or transformation */
  time_t _currentTime;

  /** Implicit timezone. This is the timezone to be used when a date, 
   * time, or dateTime value that does not have a timezone is used in a 
   * comparison or in any other operation. This value is an instance of 
   * xdt:dayTimeDuration */
  ATDurationOrDerived::Ptr _implicitTimezone;


  ////////////////////////////////////////
  //  Pathan Context specific variables //
  ////////////////////////////////////////

  /// A stack of URIResolver pointers
  std::vector<URIResolver *, PathanAllocator<URIResolver*> > _resolvers;

  // used for memory management in expression evaluation
  XPath2MemoryManager* _memMgr;
};


#endif


