# See the file LICENSE for redistribution information.
#
# Copyright (c) 2000-2005
#	Sleepycat Software.  All rights reserved.
#
# $Id: xml010.tcl,v 1.104 2005/04/20 18:31:41 bostic Exp $
#
# TEST	xml010
# TEST	XML Indexer

proc xml010 { args } {
    puts "\nxml010: Container Indexing ($args)"
    source ./include.tcl

    # Determine whether procedure has been called within an environment,
    # and whether it is a transactional environment.
    # If we are using an env, then the filename should just be the test
    # number.  Otherwise it is the test directory and the test number.
    set eindex [lsearch -exact $args "-env"]
    set txnenv 0
    set oargs [eval {set_openargs} $args]

    if { $eindex == -1 } {
	set env NULL
    } else {
	incr eindex
	set env [lindex $args $eindex]
	set txnenv [is_txnenv $env]
	set testdir [get_home $env]
    }

    # assign the container type (default to container storage)
    set tindex [lsearch -exact $args "-storage"]
    if { $tindex == -1 } {
	set global_container_type $XmlContainer_WholedocContainer
    } else {
	incr tindex
	set global_container_type [lindex $args $tindex]
    }

    set tnum 10
    set basename $tnum

    # API tests
    xml010_1 $env $txnenv $basename.1 $oargs

    # old data driven tests (restricted to comparing results for (un)indexed c'r)
    xml010_2_and_3 $env $txnenv $basename.2 $oargs

    # new regression tests, based on bugs found and extensions
    xml010_4 $env $txnenv $basename.4 $oargs
}

# API tests - base procedure
proc xml010_1 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\txml010.1: API testing"
    set id 0

    # 1.x regression tests
    xml010_1_1 $env $txnenv $basename.[incr id] $oargs

    # default indexes
    xml010_1_2 $env $txnenv $basename.[incr id] $oargs

    # unique indexes
    xml010_1_3 $env $txnenv $basename.[incr id] $oargs

    # indexing a non-empty container
    xml010_1_4 $env $txnenv $basename.[incr id] $oargs

    # decimal index on a zero value etc.
    xml010_1_5 $env $txnenv $basename.[incr id] $oargs

    # lookup methods on indexes and statistics
    xml010_1_6 $env $txnenv $basename.[incr id] $oargs
}

# API tests - regression test suite
proc xml010_1_1 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t$basename: Regression Testing"
    xml010_1_1_2 $env $txnenv $basename.2 $oargs
    xml010_1_1_4 $env $txnenv $basename.4 $oargs
    xml010_1_1_5 $env $txnenv $basename.5 $oargs
    xml010_1_1_6 $env $txnenv $basename.6 $oargs
    xml010_1_1_8 $env $txnenv $basename.8 $oargs
    xml010_1_1_9 $env $txnenv $basename.9 $oargs
    xml010_1_1_10 $env $txnenv $basename.10 $oargs
    xml010_1_1_11 $env $txnenv $basename.11 $oargs
    xml010_1_1_12 $env $txnenv $basename.12 $oargs
    xml010_1_1_13 $env $txnenv $basename.13 $oargs
    xml010_1_1_14 $env $txnenv $basename.14 $oargs
    xml010_1_1_15 $env $txnenv $basename.15 $oargs
    xml010_1_1_16 $env $txnenv $basename.16 $oargs
}

proc xml010_1_1_2 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: getIndexSpecification on an open container."

    source ./include.tcl
    set txn NULL
    xml_cleanup $testdir $env

    xml_database db $testdir $env

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs]
    }

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification is [$container getIndexSpecification]
	}
	delete is
    } ret

    dbxml_error_check_good $basename $ret ""
    delete container
    delete db
}

proc xml010_1_1_4 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: setIndexSpecification on an open container."

    source ./include.tcl
    set txn NULL
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs]
    }

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]

	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis addIndex "" "foo" "node-element-presence-none"
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis addIndex "" "foo" "node-element-presence-none"
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    dbxml_error_check_good $basename $ret ""
    delete container
    delete uc
    delete db
}

proc xml010_1_1_5 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: setIndexSpecification on non-empty container."
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs]
    }

    set txn "NULL"
    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
    }
    set ok [putFileInContainer $db $container "$test_path/document_set_10_2/document_10_2_match.xml" $txn]
    dbxml_error_check_bad "$basename: Error putting file in container" $ok 0
    if {$txnenv == 1} {
	$txn commit
	delete txn
    }

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    $container addIndex $txn "" "foo" "node-element-presence-none" $uc
	    $txn commit
	    delete txn
	} else {
	    $container addIndex "" "foo" "node-element-presence-none" $uc
	}
    } ret

    dbxml_error_check_good $basename $ret ""
    delete container
    delete uc
    delete db
}

proc xml010_1_1_6 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: Index specification persists after close."
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    set container_name $basename.dbxml
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $txn $container_name $oargs $global_container_type]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$db createContainer $container_name $oargs $global_container_type]
    }

    set uri www.uri.com
    set name foo
    set index node-element-presence-none
    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    $container addIndex $txn $uri $name $index $uc
	    $txn commit
	    delete txn
	} else {
	    $container addIndex $uri $name $index $uc
	}
    } ret
    delete container
    dbxml_error_check_good $basename $ret ""

    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlContainer container [$db openContainer $txn $container_name $oargs]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$db openContainer $container_name $oargs]
    }

    # Test that index spec was persisted.
    wrap XmlIndexSpecification xis [$container getIndexSpecification]
    set idecl [$xis find $uri $name]
    delete xis
    dbxml_error_check_bad foundindex $idecl NULL

    wrap XmlIndexDeclaration xid $idecl
    set retindex [$xid get_index]
    delete xid
    dbxml_error_check_good index $index $retindex

    delete container
    delete uc
    delete db
}

proc xml010_1_1_8 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: deleteIndex on non-empty container."
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs]
    }

    set txn "NULL"
    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
    }
    set ok [putFileInContainer $db $container "$test_path/document_set_10_2/document_10_2_match.xml" $txn]
    dbxml_error_check_bad "$basename: Error putting file in container" $ok 0
    if {$txnenv == 1} {
	$txn commit
	delete txn
    }

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]

	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis addIndex "" "person" "node-element-presence-none"
	    $xis addIndex "" "person" "node-element-equality-string"
	    $xis addIndex "" "name" "node-element-presence-none"
	    $container setIndexSpecification $txn $xis $uc
	    delete xis
	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis deleteIndex "" "person" "node-element-equality-string"
	    $xis deleteIndex "" "name" "node-element-presence-none"
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis addIndex "" "person" "node-element-presence-none"
	    $xis addIndex "" "person" "node-element-equality-string"
	    $xis addIndex "" "name" "node-element-presence-none"
	    $container setIndexSpecification $xis $uc
	    delete xis
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis deleteIndex "" "person" "node-element-equality-string"
	    $xis deleteIndex "" "name" "node-element-presence-none"
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    # Test that index spec was persisted.
    wrap XmlIndexSpecification xis [$container getIndexSpecification]
    dbxml_error_check_good findindex [$xis find "" "name"] NULL
    wrap XmlIndexDeclaration xid [$xis find "" "person"]
    dbxml_error_check_good index [$xid get_index] "node-element-presence-none"
    delete xid
    delete xis

    dbxml_error_check_good $basename $ret ""
    delete container
    delete uc
    delete db
}

proc xml010_1_1_9 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: replaceIndex on non-empty container."
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs]
    }

    set txn "NULL"
    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
    }
    set ok [putFileInContainer $db $container "$test_path/document_set_10_2/document_10_2_match.xml" $txn]
    dbxml_error_check_bad "$basename: Error putting file in container" $ok 0
    if {$txnenv == 1} {
	$txn commit
	delete txn
    }

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]

	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis addIndex "" "person" "node-element-presence-none"
	    $xis addIndex "" "person" "node-element-equality-string"
	    $container setIndexSpecification $txn $xis $uc
	    delete xis
	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis replaceIndex "" "person" "edge-element-presence-none edge-element-equality-string"
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis addIndex "" "person" "node-element-presence-none"
	    $xis addIndex "" "person" "node-element-equality-string"
	    $container setIndexSpecification $xis $uc
	    delete xis
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis replaceIndex "" "person" "edge-element-presence-none edge-element-equality-string"
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    # Test that index spec was persisted.
    wrap XmlIndexSpecification xis [$container getIndexSpecification]
    wrap XmlIndexDeclaration xid [$xis find "" "person"]
    dbxml_error_check_good index [$xid get_index] "edge-element-presence-none edge-element-equality-string"
    delete xid
    delete xis

    dbxml_error_check_good $basename $ret ""
    delete container
    delete uc
    delete db
}

proc xml010_1_1_10 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: addIndex on an open container."
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs]
    }

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    $container addIndex $txn "" "foo" "node-element-presence-none" $uc
	    $txn commit
	    delete txn
	} else {
	    $container addIndex "" "foo" "node-element-presence-none" $uc
	}
    } ret

    dbxml_error_check_good $basename $ret ""
    delete container
    delete uc
    delete db
}

proc xml010_1_1_11 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: deleteIndex on an open container."
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs]
    }

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    $container deleteIndex $txn "" "foo" "node-element-presence-none" $uc
	    $txn commit
	    delete txn
	} else {
	    $container deleteIndex "" "foo" "node-element-presence-none" $uc
	}
    } ret

    dbxml_error_check_good $basename $ret ""
    delete container
    delete uc
    delete db
}

proc xml010_1_1_12 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: replaceIndex on an open container."
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs]
    }

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    $container replaceIndex $txn "" "foo" "node-element-presence-none" $uc
	    $txn commit
	    delete txn
	} else {
	    $container replaceIndex "" "foo" "node-element-presence-none" $uc
	}
    } ret

    dbxml_error_check_good $basename $ret ""
    delete container
    delete uc
    delete db
}

proc xml010_1_1_13 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: addIndex and reindex"

    # prepare - create database, contexts, container
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlQueryContext qc [$db createQueryContext]
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
        wrap XmlTransaction xtxn [$db createTransaction]
        wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs $global_container_type]
        $xtxn commit
        delete xtxn
    } else {
        wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # add a document to the unindexed container
    set xtxn "NULL"
    if {$txnenv == 1} {
        wrap XmlTransaction xtxn [$db createTransaction]
    }
    set ok [putFileInContainer $db $container "$test_path/document_set_10_2/document_10_2_match.xml" $xtxn]
    dbxml_error_check_bad "$basename: Error putting file in container" $ok 0
    if {$txnenv == 1} {
        $xtxn commit
        delete xtxn
    }

    # now index the container
    catch {
        if { $txnenv == 1 } {
            wrap XmlTransaction xtxn [$db createTransaction]

            wrap XmlIndexSpecification xis [$container getIndexSpecification $xtxn]
            $xis addIndex "" "person" "node-element-equality-string"
            $container setIndexSpecification $xtxn $xis $uc
            delete xis

            wrap XmlResults xr [$container lookupIndex $xtxn $qc "" "person" "node-element-equality-string"]
            set xr_size [$xr size]
            delete xr

            wrap XmlStatistics xs [$container lookupStatistics $xtxn "" "person" "node-element-equality-string"]
            set xs_keys [$xs getNumberOfIndexedKeys]
            delete xs

            $xtxn commit
            delete xtxn
        } else {
            wrap XmlIndexSpecification xis [$container getIndexSpecification]
            $xis addIndex "" "person" "node-element-equality-string"
            $container setIndexSpecification $xis $uc
            delete xis

            wrap XmlResults xr [$container lookupIndex $qc "" "person" "node-element-equality-string"]
            set xr_size [$xr size]
            delete xr

            wrap XmlStatistics xs [$container lookupStatistics "" "person" "node-element-equality-string"]
            set xs_keys [$xs getNumberOfIndexedKeys]
            delete xs
        }

        # verify that the index was added and will be used
        dbxml_error_check_good $basename $xr_size 1
        dbxml_error_check_good $basename $xs_keys 1.0

        set foo ""
    } ret

    # verify that everything worked
    dbxml_error_check_good $basename $ret ""

    # clean up
    delete container
    delete qc
    delete uc
    delete db
}

proc xml010_1_1_14 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: deleteIndex and reindex"

    # prepare - create database, contexts, container
    source ./include.tcl
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]
    wrap XmlQueryContext qc [$db createQueryContext]

    if {$txnenv == 1} {
        wrap XmlTransaction xtxn [$db createTransaction]
        wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs $global_container_type]
        $xtxn commit
        delete xtxn
    } else {
        wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # add a document to the unindexed container
    set xtxn "NULL"
    if {$txnenv == 1} {
        wrap XmlTransaction xtxn [$db createTransaction]
    }
    set ok [putFileInContainer $db $container "$test_path/document_set_10_2/document_10_2_match.xml" $xtxn]
    dbxml_error_check_bad "$basename: Error putting file in container" $ok 0
    if {$txnenv == 1} {
        $xtxn commit
        delete xtxn
    }

    # add an index to the container; verify index; delete this index; verify
    catch {
        if { $txnenv == 1 } {
            wrap XmlTransaction xtxn [$db createTransaction]

            wrap XmlIndexSpecification xis [$container getIndexSpecification $xtxn]
            $xis addIndex "" "person" "node-element-equality-string"
            $container setIndexSpecification $xtxn $xis $uc

            # verify index
            wrap XmlResults xr [$container lookupIndex $xtxn $qc "" "person" "node-element-equality-string"]
            wrap XmlStatistics xs [$container lookupStatistics $xtxn "" "person" "node-element-equality-string"]
            dbxml_error_check_good $basename [$xr size] 1
            dbxml_error_check_good $basename [$xs getNumberOfIndexedKeys] 1.0
            delete xr
            delete xs

            # delete index
            $xis deleteIndex "" "person" "node-element-equality-string"
            $container setIndexSpecification $xtxn $xis $uc
            delete xis

            # verify that index has gone
            wrap XmlResults xr [$container lookupIndex $xtxn $qc "" "person" "node-element-equality-string"]
            wrap XmlStatistics xs [$container lookupStatistics $xtxn "" "person" "node-element-equality-string"]
            dbxml_error_check_good $basename [$xr size] 0
            dbxml_error_check_good $basename [$xs getNumberOfIndexedKeys] 0.0
            delete xr
            delete xs

            $xtxn commit
            delete xtxn
        } else {
            wrap XmlIndexSpecification xis [$container getIndexSpecification]
            $xis addIndex "" "person" "node-element-equality-string"
            $container setIndexSpecification $xis $uc

            # verify index
            wrap XmlResults xr [$container lookupIndex $qc "" "person" "node-element-equality-string"]
            wrap XmlStatistics xs [$container lookupStatistics "" "person" "node-element-equality-string"]
            dbxml_error_check_good $basename [$xr size] 1
            dbxml_error_check_good $basename [$xs getNumberOfIndexedKeys] 1.0
            delete xr
            delete xs

            # delete index
            $xis deleteIndex "" "person" "node-element-equality-string"
            $container setIndexSpecification $xis $uc
            delete xis

            # verify that index has gone
            wrap XmlResults xr [$container lookupIndex $qc "" "person" "node-element-equality-string"]
            wrap XmlStatistics xs [$container lookupStatistics "" "person" "node-element-equality-string"]
            dbxml_error_check_good $basename [$xr size] 0
            dbxml_error_check_good $basename [$xs getNumberOfIndexedKeys] 0.0
            delete xr
            delete xs
        }
    } ret

    # verify that everything worked
    dbxml_error_check_good $basename $ret ""

    # clean up
    delete container
    delete qc
    delete uc
    delete db

}

proc xml010_1_1_15 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: reindexing"

    # prepare - create database, contexts, container
    source ./include.tcl
    xml_cleanup $testdir $env
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]
    wrap XmlQueryContext qc [$db createQueryContext]

    if {$txnenv == 1} {
        wrap XmlTransaction xtxn [$db createTransaction]
        wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs $global_container_type]
        $xtxn commit
        delete xtxn
    } else {
        wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # add a document to the unindexed container
    set xtxn "NULL"
    if {$txnenv == 1} {
        wrap XmlTransaction xtxn [$db createTransaction]
    }
    set ok [putFileInContainer $db $container "$test_path/document_set_10_2/document_10_2_match.xml" $xtxn]
    dbxml_error_check_bad "$basename: Error putting file in container" $ok 0
    if {$txnenv == 1} {
        $xtxn commit
        delete xtxn
    }

    # add two indexes; verify; remove one and add another; verify
    catch {
        if { $txnenv == 1 } {
            wrap XmlTransaction xtxn [$db createTransaction]

            # add indexes on the "name" and "person" nodes
            wrap XmlIndexSpecification xis [$container getIndexSpecification $xtxn]
            $xis addIndex "" "name" "node-element-presence-none"
            $xis addIndex "" "person" "node-element-equality-string"
            $container setIndexSpecification $xtxn $xis $uc
            delete xis

            # verify these indexes
            wrap XmlResults xr_1 [$container lookupIndex $xtxn $qc "" "name" "node-element-presence-none"]
            wrap XmlResults xr_2 [$container lookupIndex $xtxn $qc "" "person" "node-element-equality-string"]
            wrap XmlStatistics xs_1 [$container lookupStatistics $xtxn "" "name" "node-element-presence-none"]
            wrap XmlStatistics xs_2 [$container lookupStatistics $xtxn "" "person" "node-element-equality-string"]
            dbxml_error_check_good $basename [$xr_1 size] 1
            dbxml_error_check_good $basename [$xr_2 size] 1
            dbxml_error_check_good $basename [$xs_1 getNumberOfIndexedKeys] 1.0
            dbxml_error_check_good $basename [$xs_2 getNumberOfIndexedKeys] 1.0
            delete xr_1
            delete xr_2
            delete xs_1
            delete xs_2

            # delete the "person" index, add an "age" index
            wrap XmlIndexSpecification xis [$container getIndexSpecification $xtxn]
            $xis deleteIndex "" "person" "node-element-equality-string"
            $xis addIndex "" "age" "node-attribute-equality-string"
            $container setIndexSpecification $xtxn $xis $uc
            delete xis

            # verify these indexes
            wrap XmlResults xr_1 [$container lookupIndex $xtxn $qc "" "name" "node-element-presence-none"]
            wrap XmlResults xr_2 [$container lookupIndex $xtxn $qc "" "person" "node-element-equality-string"]
            wrap XmlResults xr_3 [$container lookupIndex $xtxn $qc "" "age" "node-attribute-equality-string"]
            wrap XmlStatistics xs_1 [$container lookupStatistics $xtxn "" "name" "node-element-presence-none"]
            wrap XmlStatistics xs_2 [$container lookupStatistics $xtxn "" "person" "node-element-equality-string"]
            wrap XmlStatistics xs_3 [$container lookupStatistics $xtxn "" "age" "node-attribute-equality-string"]
            dbxml_error_check_good $basename [$xr_1 size] 1
            dbxml_error_check_good $basename [$xr_2 size] 0
            dbxml_error_check_good $basename [$xr_3 size] 1
            dbxml_error_check_good $basename [$xs_1 getNumberOfIndexedKeys] 1.0
            dbxml_error_check_good $basename [$xs_2 getNumberOfIndexedKeys] 0.0
            dbxml_error_check_good $basename [$xs_3 getNumberOfIndexedKeys] 1.0
            delete xr_1
            delete xr_2
            delete xr_3
            delete xs_1
            delete xs_2
            delete xs_3

            $xtxn commit
            delete xtxn
        } else {
            wrap XmlIndexSpecification xis [$container getIndexSpecification]

            # add indexes on the "name" and "person" nodes
            $xis addIndex "" "name" "node-element-presence-none"
            $xis addIndex "" "person" "node-element-equality-string"
            $container setIndexSpecification $xis $uc
            delete xis

            # verify these indexes
            wrap XmlResults xr_1 [$container lookupIndex $qc "" "name" "node-element-presence-none"]
            wrap XmlResults xr_2 [$container lookupIndex $qc "" "person" "node-element-equality-string"]
            wrap XmlStatistics xs_1 [$container lookupStatistics "" "name" "node-element-presence-none"]
            wrap XmlStatistics xs_2 [$container lookupStatistics "" "person" "node-element-equality-string"]
            dbxml_error_check_good $basename [$xr_1 size] 1
            dbxml_error_check_good $basename [$xr_2 size] 1
            dbxml_error_check_good $basename [$xs_1 getNumberOfIndexedKeys] 1.0
            dbxml_error_check_good $basename [$xs_2 getNumberOfIndexedKeys] 1.0
            delete xr_1
            delete xr_2
            delete xs_1
            delete xs_2

            # delete the "person" index, add an "age" index
            wrap XmlIndexSpecification xis [$container getIndexSpecification]
            $xis deleteIndex "" "person" "node-element-equality-string"
            $xis addIndex "" "age" "node-attribute-equality-string"
            $container setIndexSpecification $xis $uc
            delete xis

            # verify these indexes
            wrap XmlResults xr_1 [$container lookupIndex $qc "" "name" "node-element-presence-none"]
            wrap XmlResults xr_2 [$container lookupIndex $qc "" "person" "node-element-equality-string"]
            wrap XmlResults xr_3 [$container lookupIndex $qc "" "age" "node-attribute-equality-string"]
            wrap XmlStatistics xs_1 [$container lookupStatistics "" "name" "node-element-presence-none"]
            wrap XmlStatistics xs_2 [$container lookupStatistics "" "person" "node-element-equality-string"]
            wrap XmlStatistics xs_3 [$container lookupStatistics "" "age" "node-attribute-equality-string"]
            dbxml_error_check_good $basename [$xr_1 size] 1
            dbxml_error_check_good $basename [$xr_2 size] 0
            dbxml_error_check_good $basename [$xr_3 size] 1
            dbxml_error_check_good $basename [$xs_1 getNumberOfIndexedKeys] 1.0
            dbxml_error_check_good $basename [$xs_2 getNumberOfIndexedKeys] 0.0
            dbxml_error_check_good $basename [$xs_3 getNumberOfIndexedKeys] 1.0
            delete xr_1
            delete xr_2
            delete xr_3
            delete xs_1
            delete xs_2
            delete xs_3
        }
    } ret

    # verify that everything worked
    dbxml_error_check_good $basename $ret ""

    # clean up
    delete container
    delete qc
    delete uc
    delete db
}

# indexes on multiple attributes of a node - motivated by a bug
proc xml010_1_1_16 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: index multiple attributes of element"

    # prepare - create database, contexts, container
    source ./include.tcl
    xml_cleanup $testdir $env
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]
    wrap XmlQueryContext qc [$db createQueryContext]

    if {$txnenv == 1} {
        wrap XmlTransaction xtxn [$db createTransaction]
        wrap XmlContainer container [$db createContainer $xtxn $basename.dbxml $oargs $global_container_type]
        $xtxn commit
        delete xtxn
    } else {
        wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # add a document to the unindexed container
    set xtxn "NULL"
    if {$txnenv == 1} {
        wrap XmlTransaction xtxn [$db createTransaction]
    }
    set ok [putFileInContainer $db $container "$test_path/document_set_10_2/document_10_2_match.xml" $xtxn]
    dbxml_error_check_bad "$basename: Error putting file in container" $ok 0
    if {$txnenv == 1} {
        $xtxn commit
        delete xtxn
    }

    # add indexes on two attributes
    catch {
	set fullQuery "collection('[$container getName]')/person"
        if { $txnenv == 1 } {
            wrap XmlTransaction xtxn [$db createTransaction]

            wrap XmlIndexSpecification xis [$container getIndexSpecification $xtxn]
            $xis addIndex "" "age" "node-attribute-equality-string"
            $xis addIndex "" "drink" "node-attribute-equality-string"
            $container setIndexSpecification $xtxn $xis $uc
            delete xis

            # verify these indexes
            wrap XmlResults xr_1 [$container lookupIndex $xtxn $qc "" "age" "node-attribute-equality-string"]
            wrap XmlResults xr_2 [$container lookupIndex $xtxn $qc "" "drink" "node-attribute-equality-string"]
            wrap XmlStatistics xs_1 [$container lookupStatistics $xtxn "" "age" "node-attribute-equality-string"]
            wrap XmlStatistics xs_2 [$container lookupStatistics $xtxn "" "drink" "node-attribute-equality-string"]
            dbxml_error_check_good $basename [$xr_1 size] 1
            dbxml_error_check_good $basename [$xr_2 size] 1
            dbxml_error_check_good $basename [$xs_1 getNumberOfIndexedKeys] 1.0
            dbxml_error_check_good $basename [$xs_2 getNumberOfIndexedKeys] 1.0
            delete xr_1
            delete xr_2
            delete xs_1
            delete xs_2

            # query for the owner element (should be 1 hit)
            wrap XmlResults xr [$db query $xtxn $fullQuery $qc 0]
            dbxml_error_check_good "$basename: size_of_result_set" [$xr size] 1
            delete xr

            $xtxn commit
            delete xtxn
        } else {
            wrap XmlIndexSpecification xis [$container getIndexSpecification]
            $xis addIndex "" "age" "node-attribute-equality-string"
            $xis addIndex "" "drink" "node-attribute-equality-string"
            $container setIndexSpecification $xis $uc
            delete xis

            # verify these indexes
            wrap XmlResults xr_1 [$container lookupIndex $qc "" "age" "node-attribute-equality-string"]
            wrap XmlResults xr_2 [$container lookupIndex $qc "" "drink" "node-attribute-equality-string"]
            wrap XmlStatistics xs_1 [$container lookupStatistics "" "age" "node-attribute-equality-string"]
            wrap XmlStatistics xs_2 [$container lookupStatistics "" "drink" "node-attribute-equality-string"]
            dbxml_error_check_good $basename [$xr_1 size] 1
            dbxml_error_check_good $basename [$xr_2 size] 1
            dbxml_error_check_good $basename [$xs_1 getNumberOfIndexedKeys] 1.0
            dbxml_error_check_good $basename [$xs_2 getNumberOfIndexedKeys] 1.0
            delete xr_1
            delete xr_2
            delete xs_1
            delete xs_2

            # query for the owner element (should be 1 hit)
            wrap XmlResults xr [$db query $fullQuery $qc 0]
            dbxml_error_check_good "$basename: size_of_result_set" [$xr size] 1
            delete xr
        }
    } ret

    # verify that everything worked
    dbxml_error_check_good $basename $ret ""

    # clean up
    delete container
    delete uc
    delete qc
    delete db
}

proc xml010_1_2 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t$basename: Special Indexes"
    xml010_1_2_1 $env $txnenv $basename.1 $oargs
    xml010_1_2_2 $env $txnenv $basename.2 $oargs
    xml010_1_2_3 $env $txnenv $basename.3 $oargs
    xml010_1_2_4 $env $txnenv $basename.4 $oargs
}

proc is_default_index { xid } {
    if { [string match "*www.sleepycat.com*" [$xid get_uri] ] == 1 &&  \
	     [string compare "name" [$xid get_name] ] == 0  && \
	     [string compare "unique-node-metadata-equality-string" [$xid get_index] ] == 0 } {
	return 1
    } else {
	return 0
    }
}

# default index - verify that it has been added
proc xml010_1_2_1 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: Verify permanent index on a new container"

    source ./include.tcl
    xml_cleanup $testdir $env

    # create database, context, container
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # verify that default index has been added
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	$txn commit
	delete txn
    } else {
	wrap XmlIndexSpecification is [$container getIndexSpecification]
    }

    set found 0
    set err 0
    while { ! $err } {
	set x [$is next]
	if { $x == "NULL" } {
	    set err 1
	} else {
	    wrap XmlIndexDeclaration xid $x
	    if {[is_default_index $xid]} {
		set found 1
	    }
	    delete xid
	    if { $found } { break }
	}
    }
    delete is

    dbxml_error_check_bad "$basename: Cannot find default index" $found 0

    delete uc
    delete container
    delete db
}

# default index - add indexes that clashes with the default index
#   - expected behaviour is for the new index to be ignored
proc xml010_1_2_2 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: Clash with the permanent index"
    set unique_index { 1 0 }
    set c 0
    foreach i $unique_index {
	incr c
	xml010_1_2_2_run $i $env $txnenv $basename.$c $oargs
    }
}

proc xml010_1_2_2_run { unique env  txnenv basename oargs } {
    source ./include.tcl
    xml_cleanup $testdir $env

    # explain what is happening
    puts -nonewline "\t\t\t\t$basename: "
    if {$unique} {
	puts -nonewline "Unique Index"
    } else {
	puts -nonewline "Non-unique Index"
    }
    puts -nonewline "\n"

    # create database, context, container
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # add an index that clashes with the permanent index
    set uri "http://www.sleepycat.com/2002/dbxml"
    set node "name"
    set index ""
    if {$unique} {
	append index "unique-"
    }
    append index "metadata-equality-string"

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis addIndex $uri $node $index
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis addIndex $uri $node $index
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    # no exceptions should occur
    dbxml_error_check_good $basename $ret ""

    # only the permanent index should remain
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	$txn commit
	delete txn
    } else {
	wrap XmlIndexSpecification is [$container getIndexSpecification]
    }
    set count 0
    set found 0
    set err 0
    while { ! $err } {
	set x [$is next]
	if { $x == "NULL" } {
	    set err 1
	} else {
	    incr count
	    wrap XmlIndexDeclaration xid $x
	    if {[is_default_index $xid]} {
		set found 1
	    }
	    delete xid
	}
    }
    delete is

    dbxml_error_check_good "$basename: Wrong number of indexes found" $count 1
    dbxml_error_check_bad "$basename: Cannot find default index" $found 0

    delete uc
    delete container
    delete db
}

proc xml010_1_2_3 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: Default index"

    source ./include.tcl
    set id 0
    xml_cleanup $testdir $env

    # create database, context, container
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    set txn "NULL"
    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
    }
    set ok [putFileInContainer $db $container "$test_path/document_set_10_2/document_10_2_match.xml" $txn]
    dbxml_error_check_bad "$basename: Error putting file in container" $ok 0
    if {$txnenv == 1} {
	$txn commit
	delete txn
    }

    incr id
    puts "\t\t\t\t$basename.$id: Add default index"

    set index "edge-element-presence-none"

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis addDefaultIndex $index
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis addDefaultIndex $index
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    # no exceptions should occur
    dbxml_error_check_good "$basename.$id exception occured" $ret ""

    # verify that default index has been added
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	$txn commit
	delete txn
    } else {
	wrap XmlIndexSpecification is [$container getIndexSpecification]
    }

    dbxml_error_check_good "$basename.$id index does not exist" $index [$is getDefaultIndex]
    delete is

    incr id
    puts "\t\t\t\t$basename.$id: Replace default index"

    set index "node-element-presence-none"

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis replaceDefaultIndex $index
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis replaceDefaultIndex $index
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    # no exceptions should occur
    dbxml_error_check_good "$basename.$id exception occured" $ret ""

    # verify that default index has been added
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	$txn commit
	delete txn
    } else {
	wrap XmlIndexSpecification is [$container getIndexSpecification]
    }

    dbxml_error_check_good "$basename.$id index does not exist" $index [$is getDefaultIndex]
    delete is

    incr id
    puts "\t\t\t\t$basename.$id: Override default index"

    # add an index that clashes with the permanent index
    set uri ""
    set node "name"
    set new_index "edge-element-equality-string"

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis addIndex $uri $node $new_index
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis addIndex $uri $node $new_index
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    # no exceptions should occur
    dbxml_error_check_good "$basename.$id exception occured" $ret ""

    # verify that default index still exists
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	$txn commit
	delete txn
    } else {
	wrap XmlIndexSpecification is [$container getIndexSpecification]
    }

    wrap XmlIndexDeclaration decl [$is find $uri $node]
    dbxml_error_check_bad "$basename.$id index does not exist" $decl 0
    dbxml_error_check_good "$basename.$id index does not exist" [$decl get_index] $new_index

    dbxml_error_check_good "$basename.$id default index does not exist" [$is getDefaultIndex] $index
    delete is

    incr id
    puts "\t\t\t\t$basename.$id: Delete overriding index"

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis deleteIndex $uri $node $new_index
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis deleteIndex $uri $node $new_index
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    # no exceptions should occur
    dbxml_error_check_good "$basename.$id exception occured" $ret ""

    # verify that default index still exists
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	$txn commit
	delete txn
    } else {
	wrap XmlIndexSpecification is [$container getIndexSpecification]
    }

    dbxml_error_check_good "$basename.$id index exists" [$is find $uri $node] "NULL"
    dbxml_error_check_good "$basename.$id default index does not exist" $index [$is getDefaultIndex]
    delete is

    incr id
    puts "\t\t\t\t$basename.$id: Delete default index"

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	    $xis deleteDefaultIndex $index
	    $container setIndexSpecification $txn $xis $uc
	    delete xis

	    $txn commit
	    delete txn
	} else {
	    wrap XmlIndexSpecification xis [$container getIndexSpecification]
	    $xis deleteDefaultIndex $index
	    $container setIndexSpecification $xis $uc
	    delete xis
	}
    } ret

    # no exceptions should occur
    dbxml_error_check_good "$basename.$id exception occured" $ret ""

    # verify that default index has been added
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	$txn commit
	delete txn
    } else {
	wrap XmlIndexSpecification is [$container getIndexSpecification]
    }

    dbxml_error_check_good "$basename.$id index exists" "" [$is getDefaultIndex]
    delete is

    delete uc
    delete container
    delete db
}

proc xml010_1_2_4 { {env "NULL"} {txnenv 0} basename oargs } {
    source ./include.tcl
    xml_cleanup $testdir $env

    puts "\t\t\t$basename: test permanent (name) index removal"

    # create database, context, container
    xml_database mgr $testdir $env
    wrap XmlUpdateContext uc [$mgr createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction txn [$mgr createTransaction]
	wrap XmlContainer container [$mgr createContainer $txn $basename.dbxml $oargs $global_container_type]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$mgr createContainer $basename.dbxml $oargs $global_container_type]
    }

    # attempt to remove the permanent index via XmlContainer API
    # 1) via XmlContainer API directly
    # 2) via XmlIndexSpecification
    set uri "http://www.sleepycat.com/2002/dbxml"
    set node "name"
    # NOTE: the "unique" part doesn't matter, both fail to
    # be removed, which is the correct behavior.
    set index "unique-metadata-equality-string"
    #set index "metadata-equality-string"

    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$mgr createTransaction]
	wrap XmlIndexSpecification xis [$container getIndexSpecification $txn]
	catch {
	    $xis deleteIndex $uri $node $index
	    $container setIndexSpecification $txn $xis $uc
	} ret
	delete xis
	$txn commit
	delete txn
    } else {

	wrap XmlIndexSpecification xis [$container getIndexSpecification]
	catch {
	    $xis deleteIndex $uri $node $index
	    $container setIndexSpecification $xis $uc
	} ret
	delete xis
    }

    # exception should have been thrown
    dbxml_error_check_good $basename [string match "*cannot disable*" $ret] 1

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction xtxn [$mgr createTransaction]
	    $container deleteIndex $xtxn $uri $node $index $uc
	    $xtxn commit
	    delete xtxn
	} else {
	    $container deleteIndex $uri $node $index $uc
	}
    } ret

    # exception should have been thrown
    dbxml_error_check_good $basename [string match "*cannot disable*" $ret] 1

    # only the permanent index should remain
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$mgr createTransaction]
	wrap XmlIndexSpecification is [$container getIndexSpecification $txn]
	$txn commit
	delete txn
    } else {
	wrap XmlIndexSpecification is [$container getIndexSpecification]
    }
    set count 0
    set found 0
    set err 0
    while { ! $err } {
	set x [$is next]
	if { $x == "NULL" } {
	    set err 1
	} else {
	    incr count
	    wrap XmlIndexDeclaration xid $x
	    if {[is_default_index $xid]} {
		set found 1
	    }
	    delete xid
	}
    }
    delete is

    dbxml_error_check_good "$basename: Wrong number of indexes found" $count 1
    dbxml_error_check_bad "$basename: Cannot find default index" $found 0

    delete uc
    delete container
    delete mgr
}

# API tests - unique indexes
proc xml010_1_3 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t$basename: Unique indexes"
    xml010_1_3_1 $env $txnenv $basename.1 $oargs
    xml010_1_3_2 $env $txnenv $basename.2 $oargs
}

# adding a unique index but not using "equality" - expected behavour is to throw an exception
proc xml010_1_3_1 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: Adding an illegal unique index (failure case)"

    source ./include.tcl
    xml_cleanup $testdir $env

    # create database, context, container
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # add illegal unique index
    set uri www.uri.com
    set node foo
    set index unique-node-element-presence

    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    $container addIndex $txn $uri $node $index $uc
	    $txn commit
	    delete txn
	} else {
	    $container addIndex $uri $node $index $uc
	}
    } ret
    # exception (unknown index spec) expected
    dbxml_error_check_good $basename [string match "*index specification*${index}*" $ret] 1

    delete uc
    delete container
    delete db
}

# adding the same document twice - expected behaviour is to fail when a unique index is specified and pass otherwise
proc xml010_1_3_2 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: Adding the same document twice"
    set unique_index { 1 0 }
    set c 0
    foreach i $unique_index {
	incr c
	xml010_1_3_2_run $i $env $txnenv $basename.$c $oargs
    }
}

proc xml010_1_3_2_run { unique env txnenv basename oargs } {
    source ./include.tcl
    xml_cleanup $testdir $env

    # explain what is happening
    puts -nonewline "\t\t\t\t$basename: "
    if {$unique} {
	puts -nonewline "Unique Index (failure)"
    } else {
	puts -nonewline "Non-unique Index (success)"
    }
    puts -nonewline "\n"

    # create database, context, container
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # add index
    set uri {}
    set node foo
    set index ""
    if {$unique} {
	append index "unique-"
    }
    append index "node-element-equality-string"

    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	$container addIndex $txn $uri $node $index $uc
	$txn commit
	delete txn
    } else {
	$container addIndex $uri $node $index $uc
    }

    # add document twice
    set flags $DBXML_GEN_NAME

    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]

	wrap XmlDocument xd [$db createDocument]
	$xd setContent {<foo>bar</foo>}
	$xd setName hello

	$container putDocument $txn $xd $uc $flags
	delete xd

	$txn commit
	delete txn
    } else {
	wrap XmlDocument xd [$db createDocument]
	$xd setContent {<foo>bar</foo>}
	$xd setName hello
	$container putDocument $xd $uc $flags
	delete xd
    }

    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
    }
    wrap XmlDocument xd [$db createDocument]
    $xd setContent {<foo>bar</foo>}
    $xd setName hello

    catch {
	if { $txnenv == 1 } {
	    $container putDocument $txn $xd $uc $flags
	} else {
	    $container putDocument $xd $uc $flags
	}
    } ret

    delete xd
    if { $txnenv == 1 } {
	$txn commit
	delete txn
    }

    # success depends upon uniqueness of index specification
    if {$unique} {
	# 2004-08-26 ARW exception is currently "NsSAX2Reader parse File"
	dbxml_error_check_good $basename [string match "*Error*Uniqueness*" $ret] 1
    } else {
	dbxml_error_check_good $basename $ret ""
    }

    delete uc
    delete container
    delete db
}

# indexing a container that contains documents - operations that should fail gracefully if they are illegal
proc xml010_1_4 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t$basename: Indexing a non-empty container"
    xml010_1_4_1 $env $txnenv $basename.1 $oargs
}

proc xml010_1_4_1 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: Adding an index for a non-existent node"

    source ./include.tcl
    xml_cleanup $testdir $env

    # create database, context, container
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
	$txn commit
	delete txn
    } else {
	wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # add a document
    wrap XmlDocument xd [$db createDocument]
    $xd setContent {<foo>bar</foo>}
    $xd setName "foo"
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$db createTransaction]
	$container putDocument $txn $xd $uc
	$txn commit
	delete txn
    } else {
	$container putDocument $xd $uc
    }

    # add an index for a node that does not exist in the document
    set uri {}
    set name cost
    set index node-element-equality-string
    catch {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$db createTransaction]
	    $container addIndex $txn $uri $name $index $uc
	    $txn commit
	    delete txn
	} else {
	    $container addIndex $uri $name $index $uc
	}
    } ret

    # expected behaviour is to succeed
    dbxml_error_check_good $basename $ret ""

    delete uc
    delete container
    delete db
}

proc xml010_1_5 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t$basename: Finding zero targets"

    # prepare - create database, contexts, container
    source ./include.tcl
    xml_cleanup $testdir $env
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]
    wrap XmlQueryContext qc [$db createQueryContext]

    if {$txnenv == 1} {
        wrap XmlTransaction txn [$db createTransaction]
        wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
        $txn commit
        delete txn
    } else {
        wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # create and load document
    set content " \
<root> \
  <a>0</a> \
  <b></b> \
  <c attc='0'>ccc</c> \
  <d attd=''>ddd</d> \
</root> \
"
    wrap XmlDocument xd [$db createDocument]
    $xd setContent $content
    $xd setName "foo"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
        $container putDocument $txn $xd $uc
        $txn commit
        delete txn
    } else {
        $container putDocument $xd $uc
    }

    # query the unindexed container - all queries should hit
    set id 0
    set cname "collection('[$container getName]')"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]

        #  element with zero value
        wrap XmlResults xr [$db query $txn "$cname//a\[.=0\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        #  attribute with zero value
        wrap XmlResults xr [$db query $txn "$cname//@attc\[.=0\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        # empty element
        wrap XmlResults xr [$db query $txn "$cname//b\[.=''\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        #  empty attribute
        wrap XmlResults xr [$db query $txn "$cname//@attd\[.=''\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        $txn commit
        delete txn
    } else {
        #  element with zero value
        wrap XmlResults xr [$db query "$cname//a\[.=0\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        #  attribute with zero value
        wrap XmlResults xr [$db query "$cname//@attc\[.=0\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        # empty element
        wrap XmlResults xr [$db query "$cname//b\[.=''\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        #  empty attribute
        wrap XmlResults xr [$db query "$cname//@attd\[.=''\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr
    }

    # add indexes on the zero and empty element
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]

        $container addIndex $txn "" "a" "node-element-equality-decimal" $uc

        $txn commit
        delete txn
    } else {
        $container addIndex "" "a" "node-element-equality-decimal" $uc
    }

    # query the indexed container - all queries should hit
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]

        #  element with zero value
        wrap XmlResults xr [$db query $txn "$cname//a\[.=0\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        #  attribute with zero value
        wrap XmlResults xr [$db query $txn "$cname//@attc\[.=0\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        # empty element
        wrap XmlResults xr [$db query $txn "$cname//b\[.=''\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        #  empty attribute
        wrap XmlResults xr [$db query $txn "$cname//@attd\[.=''\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        $txn commit
        delete txn
    } else {
        #  element with zero value
        wrap XmlResults xr [$db query "$cname//a\[.=0\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        #  attribute with zero value
        wrap XmlResults xr [$db query "$cname//@attc\[.=0\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        # empty element
        wrap XmlResults xr [$db query "$cname//b\[.=''\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr

        #  empty attribute
        wrap XmlResults xr [$db query "$cname//@attd\[.=''\]" $qc 0]
        dbxml_error_check_good $basename.[incr id] [$xr size] 1
        delete xr
    }

    # clean up
    delete container
    delete uc
    delete qc
    delete db
}

# look up methods on indexes and statistics
proc xml010_1_6 { {env "NULL"} {txnenv 0} basename oargs } {
    source ./include.tcl
    puts "\t\t$basename: Lookup methods on indexes and statistics"

    # execute all tests using both eager and lazy evaluation
    list set eval_types {}
    lappend eval_types "$XmlQueryContext_Eager $oargs"
    lappend eval_types "$XmlQueryContext_Lazy $oargs"
    if { [expr $global_container_type == $XmlContainer_NodeContainer] } {
        lappend eval_types "$XmlQueryContext_Eager [expr $oargs|$DBXML_INDEX_NODES]"
        lappend eval_types "$XmlQueryContext_Lazy [expr $oargs|$DBXML_INDEX_NODES]"
    }

    set c 0
    foreach et $eval_types {
        # indexes and statistics - basic success cases
        xml010_1_6_1 $env $txnenv $basename.[incr c] [lindex $et 1] [lindex $et 0]

        # indexes - failure cases
        xml010_1_6_2 $env $txnenv $basename.[incr c] [lindex $et 1] [lindex $et 0]

        # statistics - failure cases
        xml010_1_6_3 $env $txnenv $basename.[incr c] [lindex $et 1] [lindex $et 0]
    }
}

# lookupIndex and lookupStatistics - basic success cases
proc xml010_1_6_1 { env txnenv basename oargs et } {
    source ./include.tcl

    # explain what is happening
    set msg "\t\t\t$basename: Basic success cases / "
    if { $et == $XmlQueryContext_Eager } {
        append msg "eager"
    } elseif { $et == $XmlQueryContext_Lazy } {
        append msg "lazy"
    } else {
        dbxml_error_check_good "$basename - unexpected eval type $et" 1 0
        return
    }
    if { [expr $oargs & $DBXML_INDEX_NODES] } {
        set index_nodes 1
        append msg " / node indexes"
    } else {
        set index_nodes 0
        append msg " / document indexes"
    }
    puts $msg

    # prepare - database, contexts, container
    xml_cleanup $testdir $env
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]
    wrap XmlQueryContext qc [$db createQueryContext]
    $qc setEvaluationType $et

    if {$txnenv == 1} {
        wrap XmlTransaction txn [$db createTransaction]
        wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
        $txn commit
        delete txn
    } else {
        wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }

    # prepare - add documents and index the container
    set contentA "\
<docA> \
  <foo>hello</foo>
  <foo>charlie</foo>
</docA> \
"

    set contentB "\
<docB xmlns:bar='http://www.example.com/schema'> \
  <bar:foo>hello</bar:foo>
  <bar:foo>charlie</bar:foo>
</docB> \
"

    set contentC "\
<docC> \
  <foobar>
    <baz>goodbye</baz>
    <baz>charlie</baz>
  </foobar>
</docC> \
"

    set contentD "\
<docD xmlns:bar='http://www.example.com/schema'> \
  <bar:foobar>
    <bar:baz>goodbye</bar:baz>
    <bar:baz>charlie</bar:baz>
  </bar:foobar>
</docD> \
"

    wrap XmlDocument xdA [$db createDocument]
    $xdA setName "docA"
    $xdA setContent $contentA

    wrap XmlDocument xdB [$db createDocument]
    $xdB setName "docB"
    $xdB setContent $contentB

    wrap XmlDocument xdC [$db createDocument]
    $xdC setName "docC"
    $xdC setContent $contentC

    wrap XmlDocument xdD [$db createDocument]
    $xdD setName "docD"
    $xdD setContent $contentD

    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]

        $container putDocument $txn $xdA $uc
        $container putDocument $txn $xdB $uc
        $container putDocument $txn $xdC $uc
        $container putDocument $txn $xdD $uc

        $container addIndex $txn "" "foo" "node-element-equality-string" $uc
        $container addIndex $txn "http://www.example.com/schema" "foo" "node-element-equality-string" $uc
        $container addIndex $txn "" "foo" "node-element-presence" $uc
        $container addIndex $txn "http://www.example.com/schema" "foo" "node-element-presence" $uc
        $container addIndex $txn "" "baz" "edge-element-equality-string" $uc
        $container addIndex $txn "http://www.example.com/schema" "baz" "edge-element-equality-string" $uc
        $container addIndex $txn "" "baz" "edge-element-presence" $uc
        $container addIndex $txn "http://www.example.com/schema" "baz" "edge-element-presence" $uc
        $container addIndex $txn "" "foobar" "edge-element-presence-none" $uc
        $container addIndex $txn "" "docC" "edge-element-presence-none" $uc

        $txn commit
        delete txn
    } else {

        $container putDocument $xdA $uc
        $container putDocument $xdB $uc
        $container putDocument $xdC $uc
        $container putDocument $xdD $uc

        $container addIndex "" "foo" "node-element-equality-string" $uc
        $container addIndex "http://www.example.com/schema" "foo" "node-element-equality-string" $uc
        $container addIndex "" "foo" "node-element-presence" $uc
        $container addIndex "http://www.example.com/schema" "foo" "node-element-presence" $uc
        $container addIndex "" "baz" "edge-element-equality-string" $uc
        $container addIndex "http://www.example.com/schema" "baz" "edge-element-equality-string" $uc
        $container addIndex "" "baz" "edge-element-presence" $uc
        $container addIndex "http://www.example.com/schema" "baz" "edge-element-presence" $uc
        $container addIndex "" "foobar" "edge-element-presence-none" $uc
        $container addIndex "" "docC" "edge-element-presence-none" $uc
    }
    delete xdA
    delete xdB
    delete xdC
    delete xdD

    set id 0
    set txn NULL

    #####################################################
    # node presence
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Node presence"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
    }

    # without URI
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "foo" "node-element-presence"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "foo" "node-element-presence"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "foo" "node-element-presence"]
        wrap XmlStatistics xs [$container lookupStatistics "" "foo" "node-element-presence"]
    }
    if { $index_nodes } {
        xml010_check_results $testid.a $xr 2 "docA" "foo"
        xml010_check_statistics $testid.a $xs 2 1
    } else {
        xml010_check_results $testid.a $xr 1 "docA"
        xml010_check_statistics $testid.a $xs 1 1
    }
    delete xr
    delete xs

    # with URI
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "http://www.example.com/schema" "foo" "node-element-presence"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "http://www.example.com/schema" "foo" "node-element-presence"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "http://www.example.com/schema" "foo" "node-element-presence"]
        wrap XmlStatistics xs [$container lookupStatistics "http://www.example.com/schema" "foo" "node-element-presence"]
    }

    if { $index_nodes } {
        xml010_check_results $testid.a $xr 2 "docB" "bar:foo"
        xml010_check_statistics $testid.b $xs 2 1
    } else {
        xml010_check_results $testid.a $xr 1 "docB"
        xml010_check_statistics $testid.b $xs 1 1
    }
    delete xr
    delete xs

    if { $txnenv == 1 } {
        $txn commit
        delete txn
    }

    #####################################################
    # node presence on equality index
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Node presence on equality index"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
    }

    # without URI
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "foo" "node-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "foo" "node-element-equality-string"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "foo" "node-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics "" "foo" "node-element-equality-string"]
    }
    if { $index_nodes } {
        xml010_check_results $testid.a $xr 2 "docA" "foo"
    } else {
        xml010_check_results $testid.a $xr 1 "docA"
    }
    xml010_check_statistics $testid.a $xs 2 2
    delete xr
    delete xs

    # with URI
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "http://www.example.com/schema" "foo" "node-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "http://www.example.com/schema" "foo" "node-element-equality-string"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "http://www.example.com/schema" "foo" "node-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics "http://www.example.com/schema" "foo" "node-element-equality-string"]
    }

    if { $index_nodes } {
        xml010_check_results $testid.a $xr 2 "docB" "bar:foo"
    } else {
        xml010_check_results $testid.a $xr 1 "docB"
    }
    xml010_check_statistics $testid.b $xs 2 2
    delete xr
    delete xs

    if { $txnenv == 1 } {
        $txn commit
        delete txn
    }

    #####################################################
    # edge presence
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Edge presence"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
    }

    # without URI
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "baz" "" "foobar" "edge-element-presence"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "baz" "" "foobar" "edge-element-presence"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "baz" "" "foobar" "edge-element-presence"]
        wrap XmlStatistics xs [$container lookupStatistics "" "baz" "" "foobar" "edge-element-presence"]
    }

    if { $index_nodes } {
        xml010_check_results $testid.a $xr 2 "docC" "baz"
        xml010_check_statistics $testid.a $xs 2 1
    } else {
        xml010_check_results $testid.a $xr 1 "docC"
        xml010_check_statistics $testid.a $xs 1 1
    }
    delete xr
    delete xs

    # with URI
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "http://www.example.com/schema" "baz" "http://www.example.com/schema" "foobar" "edge-element-presence"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "http://www.example.com/schema" "baz" "http://www.example.com/schema" "foobar" "edge-element-presence"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "http://www.example.com/schema" "baz" "http://www.example.com/schema" "foobar" "edge-element-presence"]
        wrap XmlStatistics xs [$container lookupStatistics "http://www.example.com/schema" "baz" "http://www.example.com/schema" "foobar" "edge-element-presence"]
    }

    if { $index_nodes } {
        xml010_check_results $testid.a $xr 2 "docD" "bar:baz"
        xml010_check_statistics $testid.b $xs 2 1
    } else {
        xml010_check_results $testid.a $xr 1 "docD"
        xml010_check_statistics $testid.b $xs 1 1
    }
    delete xr
    delete xs

    if { $txnenv == 1 } {
        $txn commit
        delete txn
    }

    #####################################################
    # edge presence on equality index
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Edge presence on equality index"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
    }

    # without URI
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "baz" "" "foobar" "edge-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "baz" "" "foobar" "edge-element-equality-string"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "baz" "" "foobar" "edge-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics "" "baz" "" "foobar" "edge-element-equality-string"]
    }
    if { $index_nodes } {
        xml010_check_results $testid.a $xr 2 "docC" "baz"
    } else {
        xml010_check_results $testid.a $xr 1 "docC"
    }
    xml010_check_statistics $testid.a $xs 2 2
    delete xr
    delete xs

    # with URI
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "http://www.example.com/schema" "baz" "http://www.example.com/schema" "foobar" "edge-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "http://www.example.com/schema" "baz" "http://www.example.com/schema" "foobar" "edge-element-equality-string"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "http://www.example.com/schema" "baz" "http://www.example.com/schema" "foobar" "edge-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics "http://www.example.com/schema" "baz" "http://www.example.com/schema" "foobar" "edge-element-equality-string"]
    }

    if { $index_nodes } {
        xml010_check_results $testid.a $xr 2 "docD" "bar:baz"
    } else {
        xml010_check_results $testid.a $xr 1 "docD"
    }
    xml010_check_statistics $testid.b $xs 2 2
    delete xr
    delete xs

    if { $txnenv == 1 } {
        $txn commit
        delete txn
    }

    #####################################################
    # node equality
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Node equality"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
    }

    # hit the target
    new XmlValue xv "hello"
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "foo" "node-element-equality-string" $xv]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "foo" "node-element-equality-string" $xv]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "foo" "node-element-equality-string" $xv]
        wrap XmlStatistics xs [$container lookupStatistics "" "foo" "node-element-equality-string" $xv]
    }
    delete xv
    xml010_check_results $testid.a $xr 1 "docA"
    xml010_check_statistics $testid.a $xs 1 1
    delete xr
    delete xs

    # miss the target
    #  - index will have 0 results, but stats estimates are unchanged
    new XmlValue xv "ooops"
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "foo" "node-element-equality-string" $xv]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "foo" "node-element-equality-string" $xv]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "foo" "node-element-equality-string" $xv]
        wrap XmlStatistics xs [$container lookupStatistics "" "foo" "node-element-equality-string" $xv]
    }
    delete xv

    xml010_check_results $testid.b $xr 0
    xml010_check_statistics $testid.b $xs 1 1
    delete xr
    delete xs

    if { $txnenv == 1 } {
        $txn commit
        delete txn
    }

    #####################################################
    # edge equality
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Edge equality"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
    }

    # hit the target
    new XmlValue xv "goodbye"
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "baz" "" "foobar" "edge-element-equality-string" $xv]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "baz" "" "foobar" "edge-element-equality-string" $xv]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "baz" "" "foobar" "edge-element-equality-string" $xv]
        wrap XmlStatistics xs [$container lookupStatistics "" "baz" "" "foobar" "edge-element-equality-string" $xv]
    }
    delete xv
    xml010_check_results $testid.a $xr 1
    xml010_check_statistics $testid.a $xs 1 1
    delete xr
    delete xs

    # miss the target
    #  - index will have 0 results, but stats estimates are unchanged
    new XmlValue xv "foo"
    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "baz" "" "foobar" "edge-element-equality-string" $xv]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "baz" "" "foobar" "edge-element-equality-string" $xv]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "baz" "" "foobar" "edge-element-equality-string" $xv]
        wrap XmlStatistics xs [$container lookupStatistics "" "baz" "" "foobar" "edge-element-equality-string" $xv]
    }
    delete xv

    xml010_check_results $testid.b $xr 0
    xml010_check_statistics $testid.b $xs 1 1
    delete xr
    delete xs

    if { $txnenv == 1 } {
        $txn commit
        delete txn
    }

    #####################################################
    # dbxml:root
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: dbxml:root"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
    }

    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "" "docC" "http://www.sleepycat.com/2002/dbxml" "root" "edge-element-presence-none"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "" "docC" "http://www.sleepycat.com/2002/dbxml" "root" "edge-element-presence-none"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "" "docC" "http://www.sleepycat.com/2002/dbxml" "root" "edge-element-presence-none"]
        wrap XmlStatistics xs [$container lookupStatistics "" "docC" "http://www.sleepycat.com/2002/dbxml" "root" "edge-element-presence-none"]
    }
    xml010_check_results $testid $xr 1 "docC"
    xml010_check_statistics $testid $xs 1 1
    delete xr
    delete xs

    if { $txnenv == 1 } {
        $txn commit
        delete txn
    }

    # clean up
    delete container
    delete uc
    delete qc
    delete db
}

# lookupIndex - failure cases
proc xml010_1_6_2 { env txnenv basename oargs et } {
    source ./include.tcl

    # explain what is happening
    set msg "\t\t\t$basename: Index failure cases / "
    if { $et == $XmlQueryContext_Eager } {
        append msg "eager"
    } elseif { $et == $XmlQueryContext_Lazy } {
        append msg "lazy"
    } else {
        dbxml_error_check_good "$basename - unexpected eval type $et" 1 0
        return
    }
    if { [expr $oargs & $DBXML_INDEX_NODES] } {
        append msg " / node indexes"
    } else {
        append msg " / document indexes"
    }
    puts $msg

    # prepare - database, contexts, container
    xml_cleanup $testdir $env
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]
    wrap XmlQueryContext qc [$db createQueryContext]
    $qc setEvaluationType $et

    set txn 0
    if {$txnenv == 1} {
        wrap XmlTransaction txn [$db createTransaction]
        wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
        $txn commit
        delete txn
    } else {
        wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }
    set id 0

    # start with tests on an unindexed container

    #####################################################
    # look up an index that does not exist (should not complain)
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Unindexed container"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]
    }

    if { $txnenv == 1 } {
        wrap XmlResults xr [$container lookupIndex $txn $qc "url" "node" "node-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics $txn "url" "node" "node-element-equality-string"]
    } else {
        wrap XmlResults xr [$container lookupIndex $qc "url" "node" "node-element-equality-string"]
        wrap XmlStatistics xs [$container lookupStatistics "url" "node" "node-element-equality-string"]
    }

    # should return zero hits
    xml010_check_results $testid $xr 0
    xml010_check_statistics $testid $xs 0 0
    delete xr
    delete xs

    if { $txnenv == 1 } {
        $txn commit
        delete txn
    }

    #####################################################
    # look up an index and statistics with an illegal index specfication
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Illegal index specification"
    catch {
        if { $txnenv == 1 } {
            wrap XmlTransaction txn2 [$db createTransaction]
            wrap XmlResults xr [$container lookupIndex $txn2 $qc "url" "node" "node-element-substring-none"]
            $txn2 commit
            delete txn2
        } else {
            wrap XmlResults xr [$container lookupIndex $qc "url" "node" "node-element-substring-none"]
        }
    } ret
    # should have raised an exception
    dbxml_error_check_good $basename [string match "Error*index*" $ret] 1

    # now add documents and index the container
    set contentA " \
<docA> \
  <foo>hello</foo> \
</docA> \
"

    set contentB " \
<docB> \
  <foo> \
    <bar>goodbye</bar> \
  </foo> \
</docB> \
"

    set contentC " \
<docC> \
  <foobar> \
    <baz>hmmm</baz> \
  </foobar> \
</docC> \
"

    wrap XmlDocument xdA [$db createDocument]
    $xdA setName "docA"
    $xdA setContent $contentA

    wrap XmlDocument xdB [$db createDocument]
    $xdB setName "docB"
    $xdB setContent $contentB

    wrap XmlDocument xdC [$db createDocument]
    $xdC setName "docC"
    $xdC setContent $contentC

    if { $txnenv == 1 } {
        wrap XmlTransaction txn [$db createTransaction]

        $container putDocument $txn $xdA $uc
        $container putDocument $txn $xdB $uc
        $container putDocument $txn $xdC $uc

        $container addIndex $txn "" "foo" "node-element-equality-string" $uc
        $container addIndex $txn "" "foo" "edge-element-equality-string" $uc
        $container addIndex $txn "" "bar" "node-element-presence-none" $uc
        $container addIndex $txn "" "bar" "edge-element-equality-string" $uc
        $container addIndex $txn "" "baz" "node-element-equality-string" $uc

        $txn commit
        delete txn
    } else {
        $container putDocument $xdA $uc
        $container putDocument $xdB $uc
        $container putDocument $xdC $uc

        $container addIndex "" "foo" "node-element-equality-string" $uc
        $container addIndex "" "foo" "edge-element-equality-string" $uc
        $container addIndex "" "bar" "node-element-presence-none" $uc
        $container addIndex "" "bar" "edge-element-equality-string" $uc
        $container addIndex "" "baz" "node-element-equality-string" $uc
    }
    delete xdA
    delete xdB
    delete xdC

    #####################################################
    # equality look up with the wrong type of value
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Value of wrong type"
    catch {
        new XmlValue xv $XmlValue_BOOLEAN "true"
        if { $txnenv == 1 } {
            wrap XmlTransaction txn4 [$db createTransaction]
            wrap XmlResults xr [$container lookupIndex $txn4 $qc "" "foo" "node-element-equality-string" $xv]
            $txn4 commit
            delete txn4
        } else {
            wrap XmlResults xr [$container lookupIndex $qc "" "foo" "node-element-equality-string" $xv]
        }
    } ret
    # should have raised an exception
    dbxml_error_check_good $basename [string match "Error*type*" $ret] 1

    #####################################################
    # equality look-up on a presence
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Equality lookup on presence"
    catch {
        new XmlValue xv2 "goodbye"
        if { $txnenv == 1 } {
            wrap XmlTransaction txn [$db createTransaction]
            wrap XmlResults xr [$container lookupIndex $txn $qc "" "bar" "node-element-presence-none" $xv2]
            $txn commit
            delete txn
        } else {
            wrap XmlResults xr [$container lookupIndex $qc "" "bar" "node-element-presence-none" $xv2]
        }
    } ret
    # should have raised an exception
    dbxml_error_check_good $basename [string match "Error*type*" $ret] 1

    #####################################################
    # edge lookup on a node index
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Edge lookup on a node index"

    if { $txnenv == 1 } {
        wrap XmlTransaction txn3 [$db createTransaction]
        wrap XmlResults xr3 [$container lookupIndex $txn3 $qc "" "baz" "" "foobar" "node-element-equality-string"]
        xml010_check_results $testid $xr3 0
        delete xr3
        $txn3 commit
        delete txn3
    } else {
        wrap XmlResults xr3 [$container lookupIndex $qc "" "baz" "" "foobar" "node-element-equality-string"]
        xml010_check_results $testid $xr3 0
        delete xr3
    }

    #####################################################
    # node equality on edge equality
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Node equality on edge equality"

    new XmlValue xv4 "hello"
    if { $txnenv == 1 } {
        wrap XmlResults xr4 [$container lookupIndex $txn $qc "" "foo" "edge-element-equality-string" $xv4]
    } else {
        wrap XmlResults xr4 [$container lookupIndex $qc "" "foo" "edge-element-equality-string" $xv4]
    }
    delete xv4

    # should have returned zero results
    xml010_check_results $testid $xr4 0
    delete xr4

    # clean up
    delete container
    delete uc
    delete qc
    delete db
}

# lookupStatistics - failure cases
#  2004-09-10 arw these tests are not grouped with lookup index failures
#  since I cannot figure out how Tcl should cope with memory etc. after
#  an exception is thrown
proc xml010_1_6_3 { env txnenv basename oargs et } {
    source ./include.tcl

    # explain what is happening
    set msg "\t\t\t$basename: Stats failure cases / "
    if { $et == $XmlQueryContext_Eager } {
        append msg "eager"
    } elseif { $et == $XmlQueryContext_Lazy } {
        append msg "lazy"
    } else {
        dbxml_error_check_good "$basename - unexpected eval type $et" 1 0
        return
    }
    if { [expr $oargs & $DBXML_INDEX_NODES] } {
        append msg " / node indexes"
    } else {
        append msg " / document indexes"
    }
    puts $msg

    # prepare - database, contexts, container
    xml_cleanup $testdir $env
    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]
    wrap XmlQueryContext qc [$db createQueryContext]
    $qc setEvaluationType $et

    if {$txnenv == 1} {
        wrap XmlTransaction txn [$db createTransaction]
        wrap XmlContainer container [$db createContainer $txn $basename.dbxml $oargs $global_container_type]
        $txn commit
        delete txn
    } else {
        wrap XmlContainer container [$db createContainer $basename.dbxml $oargs $global_container_type]
    }
    set id 0
    set txn 0

    # start with tests on an unindexed container

    #####################################################
    # look up statistics with an illegal index specfication
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Illegal index specification"
    catch {
        if { $txnenv == 1 } {
            wrap XmlTransaction txn [$db createTransaction]
            wrap XmlStatistics xs [$container lookupStatistics $txn "url" "node" "node-element-substring-none"]
            $txn commit
            delete txn
        } else {
            wrap XmlStatistics xs [$container lookupStatistics "url" "node" "node-element-substring-none"]
        }
    } ret

    # should have raised an exception
    dbxml_error_check_good $basename [string match "Error*index*" $ret] 1

    # now add a document and index the container
    wrap XmlDocument xd [$db createDocument]
    set content " \
<root> \
  <foo>hello</foo> \
</root> \
"
    $xd setContent $content
    $xd setName "foo"
    if { $txnenv == 1 } {
        wrap XmlTransaction txn2 [$db createTransaction]

        $container putDocument $txn2 $xd $uc
        $container addIndex $txn2 "" "foo" "node-element-presence-none" $uc

        $txn2 commit
        delete txn2
    } else {
        $container putDocument $xd $uc
        $container addIndex "" "foo" "node-element-presence-none" $uc
    }

    #####################################################
    # equality look up with the wrong type of value
    set testid $basename.[incr id]
    puts "\t\t\t\t$testid: Value of wrong type"

    catch {
        new XmlValue xv $XmlValue_BOOLEAN "true"
        if { $txnenv == 1 } {
            wrap XmlTransaction xtxn [$db createTransaction]
            wrap XmlStatistics xs [$container lookupStatistics $xtxn "" "foo" "node-element-presence-none" $xv]
            $xtxn commit
            delete xtxn
        } else {
            wrap XmlStatistics xs [$container lookupStatistics "" "foo" "node-element-presence-none" $xv]
        }
    } ret
    # should have raised an exception
    dbxml_error_check_good $basename [string match "Error*type*" $ret] 1

    # clean up
    delete container
    delete uc
    delete qc
    delete db
}

# check results by verifying count, a document name
proc xml010_check_results { testid results count { doc_name "" } { node_name "" } } {
    set c 0
    set err 0
    set found_named_doc 0
    while { ! $err } {
        new XmlValue xv
        $results next $xv
	if { ! [$xv getType] } {
            set err 1
        } else {
            incr c
            wrap XmlDocument xd [$xv asDocument]
            if { ! [string compare [$xd getName] $doc_name] } {
                set found_named_doc 1
            }
            if {[string compare $node_name ""]} {
                dbxml_error_check_good "$testid - node name" [$xv getNodeName] $node_name
            }
            #puts ">>>>>>>>>\n[$xv asString]\n<<<<<<<<\n"
            delete xd
        }
        delete xv
    }
    dbxml_error_check_good "$testid - result count" $c $count
    if { $count && [string compare $doc_name ""] } {
        dbxml_error_check_good "$testid - found document" $found_named_doc 1
    }
}

# check statistics by verifying count. ...
proc xml010_check_statistics { testid statistics indexed unique } {
    # allow for decimals...
    dbxml_error_check_good "$testid - indexed keys" [$statistics getNumberOfIndexedKeys] $indexed.0
    dbxml_error_check_good "$testid - unique keys" [$statistics getNumberOfUniqueKeys] $unique.0

    #puts "Check stats for $testid:\nIndexed keys = [$statistics getNumberOfIndexedKeys], unique keys = [$statistics getNumberOfUniqueKeys], key size = [$statistics getSumKeyValueSize]"
}

# Container Indexing (Restricted)
#
# The tests are now restricted to comparing the results for an indexed and
# unindexed container
#
# If the "regress" flag is set then an external program is used to dump
# the query plans to disk.
#
# The new container indexing tests are implemented in group 10.4

proc xml010_2_and_3 { {env "NULL"} {txnenv 0} {basename $tnum} oargs {regress 0} } {
    source ./include.tcl

    # documents apply to the entire test suite
    set documents {
	document_set_10_2/document_10_2_match.xml
	document_set_10_2/document_10_2_not_match_1.xml
	document_set_10_2/document_10_2_not_match_2.xml
    }

    # basename ignored
    set dir "$test_path/data_set_10"
    set exclude 0
    container_indexing_tests "10_2" "10.2" $dir $documents 0 $exclude $env $txnenv $oargs $regress
    container_indexing_tests "10_3" "10.3" $dir $documents 0 $exclude $env $txnenv $oargs $regress
}

# Drives one or more test suites based on a glob of files in a directory
#
# All files of the form "data_set_<file_match_chars>*.txt" in <dir> are processed.
#
# Also called by 12.3 (same tests, but with the "match" document deleted)
#
proc container_indexing_tests {file_match_chars basename dir documents delete exclude env txnenv oargs regress} {
    set ret [catch { glob $dir/data_set_$file_match_chars*.txt } result]
    if { $ret == 0 } {
	foreach file [lsort -dictionary $result] {
	    xml010_run $file $file_match_chars $basename $documents $delete $exclude $env $txnenv $oargs $regress
	}
    }
}

# Executes a group of tests
#
# Tests take the form of a container query using result documents. The
# container contains a number of documents, of which one will provide a match.
# Each test specifies an index and a query. The index is added to the container
# and the query is executed. The query plan, execution order and results are
# verified. The index is then removed from the container.
#
# The query is repeated on an unindexed container and the results are verified
# to be the same as those from the first query.
#
# The input data is read from a text file and written to the global array
# 'test_data'. See the "getIndexingTestData" procedure for a description of the file
# format and the 'test_data' array.
#
# 2004/07/05 ARW Tests restricted to comparing the results for indexed and
#                unindexed containers. This is because the query plans can
#                no longer be verifed using the mechanisms deployed in this
#                test suite.
#
#                If this method is called with the "regress" flag then
#                an external program is used to dump query plan information
#                to disk (for manual inspection).

proc xml010_run {file file_match_chars basename documents delete exclude {env "NULL"} {txnenv 0} oargs regress} {
    source ./include.tcl

    # refresh test environment
    xml_cleanup $testdir $env

    xml_database db $testdir $env
    wrap XmlUpdateContext uc [$db createUpdateContext]

    # get data for this group of tests (written to global array 'test_data')
    global test_data
    array set test_data {}

    set id {}
    set title {}
    set test_count [getIndexingTestData $file $basename $exclude id title]

    # test header

    # 2004/06/25 ARW restrict tests to comparing results for [un]indexed
    #  containers (query plans verified in once-only regression method)
    if { ! $regress} {
	puts "\n\t$id: $title ($test_count tests)"
	puts "\tNOTE: tests restricted to comparing \[un\]indexed results"
	puts "\t(output supressed)"
    }

    # create two containers - one of which will be unindexed
    regsub -all "\[ \(\)\]" $id _ name

    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlContainer container [$db createContainer $xtxn "$name.dbxml" $oargs]
	wrap XmlContainer container_unindexed [$db createContainer $xtxn "$name.unindexed.dbxml" $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$db createContainer "$name.dbxml" $oargs]
	wrap XmlContainer container_unindexed [$db createContainer "$name.unindexed.dbxml" $oargs]
    }

    # add the test documents to each of the containers
    set txn "NULL"
    if {$txnenv == 1} {
	wrap XmlTransaction txn [$db createTransaction]
    }
    foreach file $documents {
	set ok [putFileInContainer $db $container "$test_path/$file" $txn 1]

	if { $ok != 0 } {
	    set ok [putFileInContainer $db $container_unindexed "$test_path/$file" $txn 1]
	}

	dbxml_error_check_bad "$basename: Error putting $file in container" $ok 0
    }
    if {$txnenv == 1} {
	$txn commit
	delete txn
    }

    set match_name document_${file_match_chars}_match.xml

    # create query context
    wrap XmlQueryContext context [$db createQueryContext $XmlQueryContext_LiveValues $XmlQueryContext_Eager]
    $context setNamespace "xs" "http://www.w3.org/2001/XMLSchema"

    # set the test information for the regression option (name of, and args for,
    # the external program)
    set regress_args {}
    set regress_program {}
    if { $regress} {
	# parse the index file to get program name, additional args (ignore test)
	set f [file join $test_path document_set_regression indexer index.xml]
	set dummy {}
	set program {}
	set args {}
	set ret [parse_index_file $f dummy program args]
	dbxml_error_check_good "Could not execute indexer regression tests - \"$f\" was not recognized as a valid test driver" $ret 1

	set regress_program $program

	# basic arguments - test materials are in the same directory as the index
	set d [file dirname $f]
	set p [file normalize $d]
	set args [concat $args "--datadir $p"]

	# basic arguments - DB XML environment
	set args [concat $args "--env $testdir"]

	# arguments - switch on transactioned environment
	if {$txnenv == 1} {
	    set args [concat $args " --transacted"]
	}

	# arguments - switch on NLS
	# arguments - switch on secure mode?

	# additional arguments - outputs directory
	set d [file join regression_results indexer]
	set d [file normalize $d]
	set args [concat $args "--outdir $d"]
	if { ! [file exists $d] } {
	    file mkdir $d
	}

	set regress_args $args
    }

    # iterate over the individual test specifications
    foreach key [lsort -dictionary [array names test_data -regexp "^$key_groupid" ] ] {
	# value is a list: id, operation, (index spec), query, oqp, eqp, stats
	set testid     [lindex $test_data($key) 0]
	set operation  [lindex $test_data($key) 1]
	set uri        [lindex $test_data($key) 2]
	set node       [lindex $test_data($key) 3]
	set index      [lindex $test_data($key) 4]
	set query      [lindex $test_data($key) 5]
	set oqp        [lindex $test_data($key) 6]
	set eqp        [lindex $test_data($key) 7]
	set keys       [lindex $test_data($key) 8]
	set indexcount [lindex $test_data($key) 9]

	# add index
	if { [string compare $node ""] == 0 } {
	    if { $txnenv == 1 } {
		wrap XmlTransaction txn [$db createTransaction]
		$container addDefaultIndex $txn $index $uc
		$txn commit
		delete txn
	    } else {
		$container addDefaultIndex $index $uc
	    }
	} else {
	    if { $txnenv == 1 } {
		wrap XmlTransaction txn [$db createTransaction]
		$container addIndex $txn $uri $node $index $uc
		$txn commit
		delete txn
	    } else {
		$container addIndex $uri $node $index $uc
	    }
	}

	# if requested, delete "match" document from each container (i.e. the one
	# document that would have been found)
	if { $delete } {
	    if { $txnenv == 1 } {
		wrap XmlTransaction txn [$db createTransaction]
		$container deleteDocument $txn $match_name $uc
		$container_unindexed deleteDocument $txn $match_name $uc
		$txn commit
		delete txn
	    } else {
		$container deleteDocument $match_name $uc
		$container_unindexed deleteDocument $match_name $uc
	    }
	}

	# substitute '$test_path' string if it appears in the query
	set pattern {\$test_path}
	if { [regexp $pattern $query] == 1 } {
	    regsub -all $pattern $query $test_path query
	    regsub -all $pattern $oqp $test_path oqp
	    regsub -all $pattern $eqp $test_path eqp
	}

	# diagnostics
	puts "\t\t$testid: $operation / '$index'"

	#################################################################
	if { $regress} {

	    # append test id, container, query flags, old result to the arguments
	    set args [concat $regress_args "--container [$container getName]"]
	    set args [concat $args "--query \"$query\""]
	    set args [concat $args "--oldresult \"$oqp\""]

	    # this will dump the query plans to disk
	    run_external_test_program $txnenv $testid $regress_program $args

	} else {

	    # compare the query results for indexed and unindexed containers
	    set results_indexed {}
	    queryContainer $db $container $context $query $env $txnenv results_indexed
	    set results_unindexed {}
	    queryContainer $db $container_unindexed $context $query $env $txnenv results_unindexed
	    #puts "\t\t\t\t($results_indexed, $results_unindexed)"

	    dbxml_error_check_good $testid $results_indexed $results_unindexed
	}

	#################################################################

	# put back first document if it was deleted
	if { $delete } {
	    set file [lindex $documents 0]

	    set txn "NULL"
	    if {$txnenv == 1} {
		wrap XmlTransaction txn [$db createTransaction]
	    }

	    set ok [putFileInContainer $db $container "$test_path/$file" $txn]

	    dbxml_error_check_bad "Couldn't reload $file into container" $ok 0

	    set ok [putFileInContainer $db $container_unindexed "$test_path/$file" $txn]

	    dbxml_error_check_bad "Couldn't reload $file into container" $ok 0

	    if {$txnenv == 1} {
		$txn commit
		delete txn
	    }

	}

	# delete index from the container
	if { [string compare $node ""] == 0 } {
	    if { $txnenv == 1 } {
		wrap XmlTransaction txn [$db createTransaction]
		$container deleteDefaultIndex $txn $index $uc
		$txn commit
		delete txn
	    } else {
		$container deleteDefaultIndex $index $uc
	    }
	} else {
	    if { $txnenv == 1 } {
		wrap XmlTransaction txn [$db createTransaction]
		$container deleteIndex $txn $uri $node $index $uc
		$txn commit
		delete txn
	    } else {
		$container deleteIndex $uri $node $index $uc
	    }
	}

    }

    # housekeeping
    catch {unset test_data}

    delete context
    delete container
    delete container_unindexed
    delete uc
    delete db
}

# Executes a container query using a return type of result documents
#
# The 'results' parameter is overwritten by the number of hits
proc queryContainer { db container context query {env "NULL"} {txnenv 0} results} {
    source ./include.tcl
    set txn NULL

    upvar $results hits

    # TODO allow for deletion

    # execute query
    set fullQuery "collection('[$container getName]')$query"
    if { $txnenv == 1 } {
	wrap XmlTransaction xtxn [$db createTransaction]
	wrap XmlResults xr [$db query $xtxn $fullQuery $context 0]
	set hits [$xr size]
	delete xr
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlResults xr [$db query $fullQuery $context 0]
	set hits [$xr size]
	delete xr
    }
}

# Reads the input data for a group of tests from a text file
#
# The file is formatted using line-separated records. Data that applies to
# all tests is specified first. Individual test specifications then follow.
#
# Data that applies to all tests in the file is prefixed by a keyword:
#  - id of test group [mandatory]
#  - title of test group [mandatory]
#
# Individual test specifications do not have keywords. The data is expected
# in the following order:
#  - test id
#  - operation (test type)
#  - index specifcation of the form "uri "node" "index"
#  - query
#  - expected optimized query plan
#  - expected execution order
#  - statistics keys (space delimited, use "" if none)
#  - statistics entry count
#
# Blank lines and lines beginning with '#' are ignored.
# A comment block is enclosed by lines starting with "SKIP BEGIN" and
# "SKIP END" (nesting does not work).
# An isolated SKIP ends reading of the file.
#
# The data is written to the global array 'test_data'. The keys have a prefix
# appropriate to the type of data. The test specification keys are ordered
# to allow the correct execution order.
#
# The parameters are overwritten as follows:
#  'id'    -> test group id
#  'title' -> test group title
#
# The 'basename' corresponds to the prefix of each of the test ids. If two
# arguments are present then the first is the prefix read from the data source
# and the second is the prefix of the actual test id.
#
# Tests with ids that include any items from 'exclude' are skipped.
#
# Returns the number of tests.

proc getIndexingTestData {file basename exclude id title} {
    source ./include.tcl

    upvar $id test_group_id
    upvar $title test_group_title

    # reset the outputs
    global test_data
    catch { unset test_data }
    set test_group_id {}
    set test_group_title {}

    set state STATE_NONE
    set skip 0

    # tests have a prefix that normally that of the group name
    set data_prefix $basename
    set test_prefix $basename

    # sometimes a separate test group is using this data source
    if { [regexp " " $basename] == 1} {
	set data_prefix [lindex [split $basename] 0]
	set test_prefix [lindex [split $basename] 1]
    }

    # open file and read sequentially
    dbxml_error_check_good "Cannot open $file" [file exists $file] 1
    set fh [open $file r]

    # file has strings in utf-8 encoding
    fconfigure $fh -encoding utf-8
    set lineno 0
    set testcount 0
    set errmsg {}
    while { [gets $fh line] >= 0 } {
	set lineno [expr $lineno+1]

	# skip blank lines and comments
	if { $line == "" || [regexp "^#" $line] == 1 } { continue }

	# manage the skip mode
	if { [regexp "^$key_skip" $line] == 1 } {
	    if { [regexp BEGIN $line] == 1 } {
		set skip 1
	    } elseif { [regexp END $line] == 1 } {
		set skip 0
	    }

	    if [ string equal [string trim $line] $key_skip ] { break }
	}

	if { $skip } { continue }

	# group data - id of test group
	if { [regexp "^$key_groupid" $line] == 1 } {
	    regsub $key_groupid $line {} test_group_id
	    set test_group_id [string trim $test_group_id]
	    continue
	}

	# group data - title
	if { [regexp "^$key_grouptitle" $line] == 1 } {
	    regsub $key_grouptitle $line {} test_group_title
	    set test_group_title [string trim $test_group_title]
	    continue
	}

	# test specification - a test id marks the start
	if { [regexp "^$data_prefix\." $line] == 1 } {

	    if { $state != "STATE_NONE" } {
		set errmsg "Previous test specification incomplete?"
		break
	    }

	    # check if this test should be excluded
	    if { $exclude != 0 } {
		set exclude_test 0
		set start [string length "$data_prefix."]
		foreach item [split $exclude] {
		    if { [string first $item $line $start] != -1 } {
			set exclude_test 1
			continue
		    }
		}
		if { $exclude_test} { continue }
	    }

	    # substitute in the test prefix if this differs from the data source
	    if {$data_prefix != $test_prefix} {
		regsub $data_prefix $line $test_prefix line
	    }
	    set testid $line
	    set state STATE_TESTID
	}

	# test specification - expect consecutive lines in the correct order
	switch $state {
	    STATE_TESTID {
		set state STATE_OPERATION
	    }
	    STATE_OPERATION {
		set operation $line
		set state STATE_INDEX
	    }
	    STATE_INDEX {
		# "uri" "node" "index"
		set idx_spec [split $line]
		set uri [string trim [lindex $idx_spec 0] \"]
		set node [string trim [lindex $idx_spec 1] \"]
		set index [string trim [lindex $idx_spec 2] \"]

		set state STATE_QUERY
	    }
	    STATE_QUERY {
		set query $line
		set state STATE_OQP
	    }
	    STATE_OQP {
		set oqp $line
		set state STATE_EQP
	    }
	    STATE_EQP {
		set eqp $line
		set state STATE_STATISTICS
	    }
	    STATE_STATISTICS {
		# key1 [key2] [keyN] indexcount
		set statistics [split $line]

		if {[string first \"\" $line] == 0} {
		    # special case of empty keys - pair of quotes ""
		    set keys {}
		} else {
		    # space delimited list of words
		    set key_count [expr [llength $statistics] - 1]
		    if { [expr $key_count < 1] } {
			set errmsg "No statistics keys found"
			break
		    }
		    set keys [lrange $statistics 0 [expr $key_count - 1]]
		}

		set indexcount [lindex $statistics end]
		if { ! [string is integer $indexcount] } {
		    set errmsg "No statistics index count found"
		    break
		}

		unset statistics
		set state STATE_COMPLETE
	    }
	}

	# cache details of a complete specification
	if { [string equal $state STATE_COMPLETE] } {

	    # the key prefix is not actually needed since only test specifications
	    # are stored in the hash. However, this is consistent with the QPG tests.
	    set test_data($key_groupid:$testid) \
		[list $testid $operation $uri $node $index $query $oqp $eqp $keys $indexcount]

	    set testcount [expr $testcount+1]
	    unset keys
	    set state STATE_NONE
	}
    }
    close $fh

    if { [string length $errmsg] } {
	append msg "$file (line $lineno): " $errmsg
	dbxml_error_check_good $msg BAD GOOD
    }

    # sanity checks
    dbxml_error_check_bad "$file: Cannot find group id" $test_group_id ""
    dbxml_error_check_bad "$file: Cannot find group title" $test_group_title ""

    return $testcount
}

# regression tests -- bugs found and fixed
proc xml010_4 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\txml010.4: Regression testing -- bugs fixed"
    set id 0

    # 1.x regression tests
    xml010_4_1 $env $txnenv $basename.[incr id] $oargs
}

proc xml010_4_1 { {env "NULL"} {txnenv 0} basename oargs } {
    puts "\t\t\t$basename: equality lookups on substring indexes."

    source ./include.tcl
    set txn NULL
    xml_cleanup $testdir $env

    xml_database mgr $testdir $env
    wrap XmlUpdateContext uc [$mgr createUpdateContext]

    set uri ""
    set node "name"
    set node1 "aname"
    set index "node-element-substring-string"
    set index1 "node-attribute-substring-string"

    set content "<root><name>Bob</name></root>"
    set content1 "<root aname='Bob'/>"
    set content2 "<root><name>Bobo</name></root>"
    set content3 "<root aname='Bobo'/>"

    # create container
    if {$txnenv == 1} {
	wrap XmlTransaction xtxn [$mgr createTransaction]
	wrap XmlContainer container [$mgr createContainer $xtxn $basename.dbxml $oargs]
	$xtxn commit
	delete xtxn
    } else {
	wrap XmlContainer container [$mgr createContainer $basename.dbxml $oargs]
    }

    # add indexes
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$mgr createTransaction]
	$container addIndex $txn $uri $node $index $uc
	$container addIndex $txn $uri $node1 $index1 $uc
	$txn commit
	delete txn
    } else {
	$container addIndex $uri $node $index $uc
	$container addIndex $uri $node1 $index1 $uc
    }

    # add documents
    if { $txnenv == 1 } {
	wrap XmlTransaction txn [$mgr createTransaction]
	$container putDocument $txn "" $content $uc $DBXML_GEN_NAME
	$container putDocument $txn "" $content1 $uc $DBXML_GEN_NAME
	$container putDocument $txn "" $content2 $uc $DBXML_GEN_NAME
	$container putDocument $txn "" $content3 $uc $DBXML_GEN_NAME
	$txn commit
	delete txn
    } else {
	$container putDocument "" $content $uc $DBXML_GEN_NAME
	$container putDocument "" $content1 $uc $DBXML_GEN_NAME
	$container putDocument "" $content2 $uc $DBXML_GEN_NAME
	$container putDocument "" $content3 $uc $DBXML_GEN_NAME
    }

    set queries {
	"collection('10.4.1.dbxml')/root[name='Bob']"
	"collection('10.4.1.dbxml')/root[name='Bobo']"
	"collection('10.4.1.dbxml')/root[@aname='Bob']"
	"collection('10.4.1.dbxml')/root[@aname='Bobo']"
    }
    wrap XmlQueryContext context [$mgr createQueryContext]
    foreach query $queries {
	if { $txnenv == 1 } {
	    wrap XmlTransaction txn [$mgr createTransaction]
	    wrap XmlResults xr [$mgr query $txn $query $context 0]
	    set hits [$xr size]
	    delete xr
	    $txn commit
	    delete txn
	} else {
	    wrap XmlResults xr [$mgr query $query $context 0]
	    set hits [$xr size]
	    delete xr
	}
        dbxml_error_check_good "$query hit" $hits 1
    }

    delete context
    delete container
    delete uc
    delete mgr
}
